package ru.yandex.dispatcher.common.connection;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.EventType;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;

import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.dispatcher.common.BooleanCallback;
import ru.yandex.dispatcher.common.ByteArrayCallback;
import ru.yandex.dispatcher.common.VoidCallback;
import ru.yandex.dispatcher.common.StatCallback;
import ru.yandex.dispatcher.common.ZooCallback;
import ru.yandex.dispatcher.common.ZooDataCallback;
import ru.yandex.dispatcher.common.ZooDisconnectedException;
import ru.yandex.dispatcher.common.ZooException;
import ru.yandex.dispatcher.consumer.ZooHost;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;

public abstract class ZooConnection {
    protected static final int RECONNECT_INTERVAL = 3000;
    protected static final int RECONNECT_THREADS = reconnectThreads();
    private static final ScheduledExecutorService delayer =
        new ScheduledThreadPoolExecutor(
            RECONNECT_THREADS,
            new NamedThreadFactory("ZooReconnector"));

    private
        final
            ConcurrentHashMap<ZooConnectionCallback,ZooConnectionCallback>
                clients =
        new ConcurrentHashMap<ZooConnectionCallback,ZooConnectionCallback>();
    private final ConcurrentHashMap<ZooCallback,ZooCallback> requestsInFly =
        new ConcurrentHashMap<ZooCallback,ZooCallback>();

    private static int reconnectThreads() {
        String value = System.getProperty(
            "ru.yandex.dispatcher-zk.reconnect-threads",
            "32");
        return Integer.parseInt(value);
    }

    protected final AsyncClient httpClient;
    protected final boolean autoReconnect;
    protected boolean preferMaster = false;
    protected PrefixedLogger logger;

    protected ZooConnection(
        final AsyncClient httpClient,
        final boolean autoReconnect,
        final PrefixedLogger logger)
    {
        this.httpClient = httpClient;
        this.autoReconnect = autoReconnect;
        this.logger = logger;
    }

    public static ZooConnection createConnection(
        final AsyncClient httpClient,
        final ZooConnectionDSN dsn,
        final int timeout,
        final boolean autoReconnect,
        final PrefixedLogger logger)
    {
        if (dsn.size() > 1) {
            return new ZooPingedMultiServerConnection(
                httpClient,
                dsn.hosts(),
                timeout,
                autoReconnect, logger);
        } else {
            return new ZooSingleServerConnection(
                httpClient,
                dsn.hosts().get(0),
                timeout,
                autoReconnect,
                logger);
        }
    }

    public void log(final Level level, final String message) {
        if (logger != null) {
            logger.log(level, message);
        }
    }

    public void log(
        final Level level,
        final String message,
        final Throwable t)
    {
        if (logger != null) {
            logger.log(level, message, t);
        }
    }

    public boolean preferMaster() {
        return preferMaster;
    }

    public void preferMaster(final boolean preferMaster) {
        this.preferMaster = preferMaster;
    }

    public void setLogger(final PrefixedLogger logger) {
        this.logger = logger;
    }

    public void connect() throws ZooException {
        if (!autoReconnect) {
            connectImpl();
        } else {
            try {
                connectImpl();
            } catch (Exception e) {
                e.printStackTrace();
                scheduleReconnect(RECONNECT_INTERVAL);
            }
        }
    }

    public void connectAsync() {
        try {
            connectImpl();
        } catch (Exception e) {
            e.printStackTrace();
            scheduleReconnect(RECONNECT_INTERVAL);
        }
    }

    public abstract boolean isConnected();

    public abstract void connectImpl() throws ZooException;

    public abstract void reconnectImpl() throws ZooException;

    public abstract ZooKeeper getZk(ZooCallback callback);

    public abstract ZooHost currentServer();

    public abstract boolean master();

    public abstract int port();

    public abstract String ZKhost();

    public void register(final ZooConnectionCallback callback) {
        clients.put(callback, callback);
    }

    public void unregister(final ZooConnectionCallback callback) {
        clients.remove(callback);
    }

    protected void scheduleReconnect(int delayMillis) {
        Runnable reconnecter = new Runnable() {
            public void run() {
                try {
                    reconnectImpl();
                } catch (Exception e) {
                    scheduleReconnect(RECONNECT_INTERVAL);
                }
            }
        };
        delayer.schedule(reconnecter, delayMillis,
            TimeUnit.MILLISECONDS);
    }

    public void disconnected() {
        for (ZooConnectionCallback c : clients.keySet()) {
            c.disconnected();
        }
        ZooException e = new ZooDisconnectedException();
        for (ZooCallback c : requestsInFly.keySet()) {
            error(c, e);
        }
        if (autoReconnect) {
            scheduleReconnect(0);
        }
    }

    public void connected() {
        for (ZooConnectionCallback c : clients.keySet()) {
            c.connected();
        }
    }

    protected void error(final ZooCallback callback, final ZooException e) {
        requestsInFly.remove(callback);
        callback.error(e);
    }

    protected void finished(final ZooCallback callback) {
        requestsInFly.remove(callback);
    }

    protected <E> void finished(final ZooDataCallback<E> callback, E data) {
        requestsInFly.remove(callback);
        callback.data(data);
    }

    public void exists(final String path,
        final BooleanCallback callback)
    {
        requestsInFly.put(callback, callback);
        existsImpl(path, callback);
    }

    public void exists(final String path,
        final StatCallback callback)
    {
        requestsInFly.put(callback, callback);
        existsImpl(path, callback);
    }

    public void data(final String path,
        final ByteArrayCallback callback, final boolean watch)
    {
        requestsInFly.put(callback, callback);
        dataImpl(path, callback, watch);
    }

    public void setData(final String path, final byte[] data,
        final VoidCallback callback)
    {
        requestsInFly.put(callback, callback);
        setDataImpl(path, data, callback);
    }

    public void createNode(final String path, final byte[] data,
        final boolean persistent,
        final BooleanCallback callback)
    {
        requestsInFly.put(callback, callback);
        createNodeImpl(path, data, persistent, callback);
    }

    public void deleteNode(final String path, final VoidCallback callback) {
        requestsInFly.put(callback, callback);
        deleteNodeImpl(path, callback);
    }

    public void existsImpl(final String path,
        final BooleanCallback callback)
    {
        ZooKeeper zk = getZk(callback);
        if (zk == null) {
            return;
        }
        AsyncCallback.StatCallback statCallback =
            new AsyncCallback.StatCallback() {
                @Override
                public void processResult(final int rc, final String path,
                    final Object ctx, final Stat stat)
                {
                    KeeperException.Code code = KeeperException.Code.get(rc);
                    switch (code) {
                        case OK:
                            finished(callback, true);
                            break;
                        case NONODE:
                            finished(callback, false);
                            break;
                        default:
                            error(callback, ZooException.fromCode(code, path));
                    }
                }
            };
        zk.exists(path, false, statCallback, null);
    }

    public void existsImpl(final String path,
        final StatCallback callback)
    {
        ZooKeeper zk = getZk(callback);
        if (zk == null) {
            return;
        }
        AsyncCallback.StatCallback statCallback =
            new AsyncCallback.StatCallback() {
                @Override
                public void processResult(final int rc, final String path,
                    final Object ctx, final Stat stat)
                {
                    KeeperException.Code code = KeeperException.Code.get(rc);
                    switch (code) {
                        case OK:
                            finished(callback, stat);
                            break;
                        case NONODE:
                            finished(callback, null);
                            break;
                        default:
                            error(callback, ZooException.fromCode(code, path));
                    }
                }
            };
        zk.exists(path, false, statCallback, null);
    }

    public void dataImpl(final String path,
        final ByteArrayCallback callback, final boolean watch)
    {
        ZooKeeper zk = getZk(callback);
        if (zk == null) {
            return;
        }
        AsyncCallback.DataCallback dataCallback =
            new AsyncCallback.DataCallback() {
                @Override
                public void processResult(final int rc, final String path,
                    final Object ctx, final byte[] data, final Stat stat)
                {
                    KeeperException.Code code = KeeperException.Code.get(rc);
                    switch (code) {
                        case OK:
                            finished(callback, data);
                            break;
                        default:
                            error(callback, ZooException.fromCode(code, path));
                    }
                }
            };
        Watcher dataWatcher = null;
        if (watch) {
            dataWatcher = new Watcher() {
                @Override
                public void process(org.apache.zookeeper.WatchedEvent event) {
                    if (logger != null && logger.isLoggable(Level.FINE)) {
                        logger.fine("Watcher: " + event);
                    }
                    if (event.getState() == KeeperState.SyncConnected)
//                        && event.getType() == EventType.NodeDataChanged)
//TODO: Fix zoolooser: sometime it sends watcher event NodeDeleted instead of NodeDataChanged
                    {
                        callback.dataChanged();
                    }
                }
            };
        }
        zk.getData(path, dataWatcher, dataCallback, null);
    }

    public void setDataImpl(final String path, final byte[] data,
        final VoidCallback callback)
    {
        ZooKeeper zk = getZk(callback);
        if (zk == null) {
            return;
        }
        AsyncCallback.StatCallback statCallback =
            new AsyncCallback.StatCallback() {
                @Override
                public void processResult(final int rc, final String path,
                    final Object ctx, final Stat stat)
                {
                    KeeperException.Code code = KeeperException.Code.get(rc);
                    switch (code) {
                        case OK:
                            finished(callback, null);
                            break;
                        default:
                            error(callback, ZooException.fromCode(code, path));
                    }
                }
            };
        zk.setData(path, data, -1, statCallback, null);
    }

    public void createNodeImpl(final String path, final byte[] data,
        final boolean persistent, final BooleanCallback callback)
    {
        ZooKeeper zk = getZk(callback);
        if (zk == null) {
            return;
        }
        AsyncCallback.StringCallback stringCallback =
            new AsyncCallback.StringCallback() {
                @Override
                public void processResult(final int rc, final String path,
                    final Object ctx, final String name)
                {
                    KeeperException.Code code = KeeperException.Code.get(rc);
                    switch (code) {
                        case OK:
                            finished(callback, true);
                            break;
                        case NODEEXISTS:
                            finished(callback, false);
                            break;
                        default:
                            error(callback, ZooException.fromCode(code, path));
                    }
                }
            };
        CreateMode mode =
            persistent ? CreateMode.PERSISTENT : CreateMode.EPHEMERAL;
        zk.create(path, data, null, mode, stringCallback, null);
    }

    public void deleteNodeImpl(
        final String path,
        final VoidCallback callback)
    {
        ZooKeeper zk = getZk(callback);
        if (zk == null) {
            return;
        }
        AsyncCallback.VoidCallback voidCallback =
            new AsyncCallback.VoidCallback() {
                @Override
                public void processResult(final int rc, final String path,
                    final Object ctx)
                {
                    KeeperException.Code code = KeeperException.Code.get(rc);
                    switch (code) {
                        case OK:
                            finished(callback, null);
                            break;
                        default:
                            error(callback, ZooException.fromCode(code, path));
                    }
                }
            };
        zk.delete(path, -1, voidCallback, null);
    }

}
