package ru.yandex.dispatcher.common.connection;

import java.io.IOException;
import java.util.List;
import java.util.logging.Level;

import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;

import ru.yandex.dispatcher.common.ZooCallback;
import ru.yandex.dispatcher.common.ZooDisconnectedException;
import ru.yandex.dispatcher.common.ZooException;
import ru.yandex.dispatcher.consumer.ZooHost;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.util.timesource.TimeSource;

public abstract class ZooMultiServerConnection extends ZooConnection
    implements Watcher
{
    private static final long MASTER_CHECK_INTERVAL = 30000;

    protected final int timeout;
    protected volatile ZooKeeper zk;
    protected volatile boolean connected;
    protected ZooSingleServerConnection currentConnection = null;
    protected long lastMasterCheck = 0;

    protected ZooMultiServerConnection(
        final AsyncClient httpClient,
        final int timeout,
        final boolean autoReconnect,
        final PrefixedLogger logger)
    {
        super(httpClient, autoReconnect, logger);
        this.timeout = timeout;
        zk = null;
        connected = false;
    }

    public abstract ZooSingleServerConnection selectServer(
        final boolean preferMaster);

    @Override
    public ZooHost currentServer() {
        ZooSingleServerConnection conn = currentConnection;
        if (conn != null) {
            return conn.currentServer();
        } else {
            return null;
        }
    }

    @Override
    public boolean master() {
        ZooSingleServerConnection conn = currentConnection;
        if (conn != null) {
            return conn.master();
        } else {
            return false;
        }
    }

    public boolean isConnected() {
        return connected;
    }

    public synchronized void connectImpl() throws ZooException {
        if (zk != null) {
            return;
        }
        currentConnection = selectServer(preferMaster());
        if (currentConnection == null) {
            throw new ZooException("No zookeeper servers are reachable");
        }
        final String server = currentConnection.currentServer().zkAddress();
        try {
            zk = new ZooKeeper(
                server,
                timeout,
                this,
                logger);
            lastMasterCheck = TimeSource.INSTANCE.currentTimeMillis();
        } catch (IOException e) {
            throw new ZooException("Failed to connect to <" + server + ">", e);
        }
    }

    public synchronized void reconnectImpl() throws ZooException {
        connectImpl();
    }

    public synchronized void process(final WatchedEvent event) {
        switch(event.getState()) {
            case SyncConnected:
                connected = true;
                connected();
                break;
            case Disconnected:
            case Expired:
                disconnect();
                break;
            default:
                break;
        }
    }

    private synchronized void disconnect() {
        connected = false;
        if (zk != null) {
            zk.close();
            zk = null;
            currentConnection = null;
        }
        disconnected();
    }

    public ZooKeeper getZk(final ZooCallback callback) {
        if (!connected) {
            callback.error(new ZooDisconnectedException());
            return null;
        }
        if (currentConnection != null && !currentConnection.master()
            && preferMaster())
        {
            long masterCheckInteval =
                TimeSource.INSTANCE.currentTimeMillis() - lastMasterCheck;
            if (masterCheckInteval > MASTER_CHECK_INTERVAL) {
                final ZooSingleServerConnection conn = selectServer(true);
                if (!conn.master()) {
                    lastMasterCheck = TimeSource.INSTANCE.currentTimeMillis();
                    log(Level.INFO, "No master available, staying on slave");
                } else {
                    log(Level.INFO, "Trying to reconnect to master");
                    disconnect();
                }
            }
        }
        return zk;
    }

}
