package ru.yandex.dispatcher.common.connection;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;

import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.dispatcher.common.ZooCallback;
import ru.yandex.dispatcher.common.ZooDisconnectedException;
import ru.yandex.dispatcher.common.ZooException;
import ru.yandex.dispatcher.consumer.ZooHost;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;

public class ZooPingedMultiServerConnection
    extends ZooMultiServerConnection
{
    private static final int PING_INTERVAL = 500;
    private static final int RECONNECT_INTERVAL = 3000;
    private static final ScheduledExecutorService delayer =
        new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("ZooPinger"));
    private static final ConcurrentHashMap<
        ZooHost,
        ZooSingleServerConnection> pingers = new ConcurrentHashMap<>();

    private final String shortDesc;

    private final List<ZooHost> servers;
    private boolean pingersStarted = false;

    protected ZooPingedMultiServerConnection(
        final AsyncClient httpClient,
        final List<ZooHost> servers,
        final int timeout,
        final boolean autoReconnect,
        final PrefixedLogger logger)
    {
        super(httpClient, timeout, autoReconnect, logger);
        this.servers = servers;
        zk = null;
        connected = false;
        final StringBuilder sb = new StringBuilder();
        sb.append("ZPMSC<");
        String prevPorts = null;
        String sep = "";
        for (ZooHost zh : servers) {
            String server = zh.zkAddress();
            sb.append(sep);
            sep = ",";
            String ports = "";
            int colon =  server.lastIndexOf(':');
            if (colon != -1) {
                ports = server.substring(colon + 1);
            }
            if (prevPorts == null || !prevPorts.equals(ports)) {
                sb.append(ports);
                sb.append(':');
                prevPorts = ports;
            }
            int dot = server.indexOf('.');
            if (dot != -1) {
                server = server.substring(0, dot);
            }
            sb.append(server);
        }
        sb.append('>');
        this.shortDesc = new String(sb);
    }

    @Override
    public synchronized void connectImpl() throws ZooException {
        runPingers();
        super.connectImpl();
    }

    @Override
    public String toString() {
        return shortDesc;
    }

    @Override
    public int port() {
        return servers.get(0).port();
    }

    @Override
    public String ZKhost() {
        ArrayList<String> sortedZk = new ArrayList<String>();
        for (final ZooHost server : servers) {
            sortedZk.add(server.httpHost().getHostName());
        }
        sortedZk.sort(Comparator.naturalOrder());
        return sortedZk.get(0);
    }

    private void runPingers() {
        if (pingersStarted) {
            return;
        }
        for (final ZooHost server : servers) {
            ZooSingleServerConnection pinger = pingers.get(server);
            if (pinger != null) {
                continue;
            }
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("Running pinger for: " + server);
            }
            ZooSingleServerConnection newPinger =
                new ZooSingleServerConnection(
                    httpClient,
                    server,
                    timeout,
                    true,
                    logger)
                {
                    @Override
                    public void connected() {
                        if (logger.isLoggable(Level.FINE)) {
                            logger.fine("Pinger: server=" + server + ", connected");
                        }
                        schedulePing(this);
                    }
                };
            pinger = pingers.putIfAbsent(server, newPinger);
            if (pinger == null) {
                pinger = newPinger;
                try {
                    pinger.connect();
                } catch (ZooException ign) {
                }
            }
        }
        pingersStarted = true;
    }

    private void schedulePing(final ZooSingleServerConnection conn) {
        Runnable pinger = new Runnable() {
            @Override
            public void run() {
                conn.ping();
                schedulePing(conn);
            }
        };
        delayer.schedule(pinger, PING_INTERVAL,
            TimeUnit.MILLISECONDS);
    }

    private void scheduleReconnect(final ZooSingleServerConnection conn) {
        Runnable reconnecter = new Runnable() {
            @Override
            public void run() {
                try {
                    conn.reconnectImpl();
                } catch (ZooException e) {
                    scheduleReconnect(conn);
                } catch (Exception e) {
                    e.printStackTrace();
                    scheduleReconnect(conn);
                }
            }
        };
        delayer.schedule(reconnecter, RECONNECT_INTERVAL,
            TimeUnit.MILLISECONDS);
    }

    public ZooSingleServerConnection selectServer(final boolean preferMaster) {
        ZooSingleServerConnection server = null;
        if (preferMaster) {
            ZooSingleServerConnection master = selectMasterServer();
            if (master != null) {
                server = master;
            } else {
                server = selectFastestServer();
            }
        } else {
            server = selectFastestServer();
        }
        return server;
    }

    private ZooSingleServerConnection selectMasterServer() {
        for (ZooHost s : servers) {
            ZooSingleServerConnection conn = pingers.get(s);
            if (conn.isConnected() && conn.master()) {
                return conn;
            }
        }
        return null;
    }

    private ZooSingleServerConnection selectFastestServer() {
        long minPing = Long.MAX_VALUE;
        ZooSingleServerConnection server = null;
        for (ZooHost s : servers) {
            ZooSingleServerConnection conn = pingers.get(s);
            long connTime = conn.pingTime();
            if (conn.isConnected() &&  connTime != -1
                && minPing > connTime)
            {
                minPing = connTime;
                server = conn;
            }
        }
        return server;
    }

}
