package ru.yandex.dispatcher.common.connection;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;

import ru.yandex.dispatcher.common.BooleanCallback;
import ru.yandex.dispatcher.common.ZooCallback;
import ru.yandex.dispatcher.common.ZooDisconnectedException;
import ru.yandex.dispatcher.common.ZooException;
import ru.yandex.dispatcher.consumer.ZooHost;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;

public class ZooSingleServerConnection extends ZooConnection
    implements Watcher, FutureCallback<String>
{
    protected static final int PING_AVG_DEPTH = 30;
    private final ZooHost server;
    private final int timeout;
    private volatile ZooKeeper zk;
    private volatile boolean connected;
    private final AtomicLong pingStartTime = new AtomicLong(-1);
    private volatile long pingTime = -1;
    private long avgPingSum;
    private final long[] avgBuffer = new long[PING_AVG_DEPTH];
    private volatile long avgPingCount = 0;
    private boolean master = false;

    public ZooSingleServerConnection(
        final AsyncClient httpClient,
        final ZooHost server,
        final int timeout,
        final boolean autoReconnect,
        final PrefixedLogger logger)
    {
        super(httpClient, autoReconnect, logger);
        this.server = server;
        this.timeout = timeout;
        zk = null;
        connected = false;
    }

    @Override
    public String toString() {
        return this.getClass().getName() + ": " + server;
    }

    @Override
    public int port() {
        return server.port();
    }
    @Override
    public String ZKhost() {
        return server.httpHost().getHostName();
    }

    public boolean isConnected() {
        return connected;
    }

    public ZooHost currentServer() {
        return server;
    }

    public void connectImpl() throws ZooException {
        if (zk != null) {
            return;
        }
        try {
            pingStartTime.set(System.nanoTime());
            zk = new ZooKeeper(server.zkAddress(), timeout, this, logger);
        } catch (IOException e) {
            throw new ZooException("Failed to connect to <" + server + ">", e);
        }
    }

    public synchronized void reconnectImpl() throws ZooException {
        if (zk != null) {
            return;
        }
        try {
            pingStartTime.set(System.nanoTime());
            zk = new ZooKeeper(server.zkAddress(), timeout, this, logger);
        } catch (IOException e) {
            throw new ZooException("Failed to connect to <" + server + ">", e);
        }
    }

    public synchronized void process(final WatchedEvent event) {
        if (logger != null) {
            logger.fine("Server=" + server + ", event=" + event);
        }
        switch(event.getState()) {
            case SyncConnected:
                pingTime =
                    System.nanoTime() - pingStartTime.getAndSet(-1);
                httpPing();
                break;
            case Disconnected:
            case Expired:
                connected = false;
                try {
                    zk.close();
                } finally {
                    zk = null;
                }
                master = false;
                disconnected();
                break;
            default:
                break;
        }
    }

    private void httpPong(final boolean master) {
        this.master = master;
        connected = true;
        connected();
    }

    public ZooKeeper getZk(final ZooCallback callback) {
        if (!connected) {
            callback.error(new ZooDisconnectedException());
            return null;
        }
        return zk;
    }

    public void ping() {
        if (pingStartTime.compareAndSet(-1, System.nanoTime())) {
            BooleanCallback bcb = new BooleanCallback() {
                @Override
                public void dataImpl(Boolean b) {
                    long time =
                        System.nanoTime() - pingStartTime.getAndSet(-1);
                    pushAvgTime(time);
                    if (avgPingCount % PING_AVG_DEPTH == 0) {
                        log(Level.INFO, "ZooPing: server=" + server
                            + ", time=" + pingTime + ", data=" + b
                            + ", master=" + master);
                    }
                }

                @Override
                public void errorImpl(ZooException e) {
                    pingStartTime.set(-1);
                    pingTime = -1;
                    log(Level.INFO, "ZooPing: server=" + server
                            + ", error=" + e.getMessage());
                }

                @Override
                public void dataChangedImpl() {
                }
            };
            exists("/", bcb);
        } //else there is a ping already in fly
    }

    private void httpPing() {
        if (httpClient != null) {
            httpClient.execute(
                server.httpHost(),
                new BasicAsyncRequestProducerGenerator("/stat"),
                AsyncStringConsumerFactory.OK,
                this);
        } else {
            httpPong(false);
        }
    }

    public synchronized void pushAvgTime(final long time) {
        avgPingCount++;
        final int bufferPos = (int)(avgPingCount % PING_AVG_DEPTH);
        avgBuffer[bufferPos] = time;
        avgPingSum += time;
        if (avgPingCount > PING_AVG_DEPTH) {
            final int prevPos = (bufferPos + 1) % PING_AVG_DEPTH;
            avgPingSum -= avgBuffer[prevPos];
            pingTime = avgPingSum / PING_AVG_DEPTH;
        } else {
            pingTime = avgPingSum / avgPingCount;
        }
    }

    public long pingCount() {
        return avgPingCount;
    }

    @Override
    public boolean master() {
        return master;
    }

    public long pingTime() {
        //get micros
        return pingTime / 1000;
    }

    @Override
    public void cancelled() {
        log(Level.INFO, "ZooPing: server=" + server
            + ": httpPing is cancelled");
        httpPong(false);
    }

    @Override
    public void failed(final Exception e) {
        e.printStackTrace();
        log(Level.INFO, "ZooPing: server=" + server
            + ": httpPing is failed", e);
        httpPong(false);
        //retry
        httpPing();
    }

    @Override
    public void completed(final String body) {
        final boolean master = !body.equals("Not a leader");
        log(Level.INFO, "ZooPing: server=" + server
            + ": httpPong.master: " + master);
        httpPong(master);
    }
}
