package ru.yandex.dispatcher.common.mappedvars;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

import ru.yandex.dispatcher.common.BooleanCallback;
import ru.yandex.dispatcher.common.ByteArrayCallback;
import ru.yandex.dispatcher.common.VoidCallback;
import ru.yandex.dispatcher.common.ZooException;
import ru.yandex.dispatcher.common.ZooNodeExistsException;
import ru.yandex.dispatcher.common.connection.ZooConnection;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.util.timesource.TimeSource;

public class ZooLock implements GenericAutoCloseable<ZooException>, Runnable {
    private static final String hostname = obtainHostName();
    private static final ScheduledExecutorService expirerScheduler =
        new ScheduledThreadPoolExecutor(1,
            new NamedThreadFactory("LockExpirer"));
    private static final ThreadPoolExecutor unparkExecutor =
        new ThreadPoolExecutor(1, 1, 1, TimeUnit.DAYS,
            new LinkedBlockingQueue<Runnable>(),
            new NamedThreadFactory("Unparker"));

    private final String path;
    private final HashSet<Runnable> expireCallbacks = new HashSet<>();
    private int lockTimeout;
    private final ZooNodeMapping node;
    private String id;
    private final ConcurrentLinkedQueue<UnparkCallback> parked =
        new ConcurrentLinkedQueue<UnparkCallback>();
    private final AtomicBoolean autoLock = new AtomicBoolean(false);
    private Logger logger = null;
    private ScheduledFuture expirer = null;
    private volatile long lastCheckTime = 0;
    private boolean autoUpdateOnSync = false;

    private static String obtainHostName() {
        try {
            return InetAddress.getLocalHost().getHostName();
        } catch (Exception e) {
            try {
                return exec("hostname -f");
            } catch (Exception ee) {
                return "localhost.localdomain";
            }
        }
    }

    private static String exec(String cmd) throws IOException {
        Process p = Runtime.getRuntime().exec(cmd);
        BufferedReader in = new BufferedReader(
            new InputStreamReader(
                p.getInputStream()));
        StringBuilder sb = new StringBuilder();
        String line = null;
        String sep = "";
        while ((line = in.readLine()) != null) {
            sb.append(sep);
            sb.append("line");
            sep = " ";
        }
        return new String(sb);
    }

    public ZooLock(final ZooConnection connection, final String path,
        final int connTimeout)
    {
        this.path = path;
        this.lockTimeout = connTimeout;
        this.id = generateId();
        node = new ZooNodeMapping(connection, path);
        node.setPersistent(false);
        node.setTimeout(connTimeout);
        node.registerSyncCallback(
            new SyncCallback() {
                @Override
                public void synced() {
                    if (autoLock.get()) {
                        lockAsync();
                    }
                    if (autoUpdateOnSync) {
                        try {
                            checkId(true, true);
                        } catch (ZooException e) {
                            if (logger != null) {
                                logger.log(
                                    Level.SEVERE,
                                    "auto update on sync failed",
                                    e);
                            }
                        }
                    }
                }
                @Override
                public void error(ZooException e) {
                    if (autoLock.get()) {
                        lockAsync();
                    }
                    if (autoUpdateOnSync) {
                        try {
                            checkId(true, true);
                        } catch (ZooException ee) {
                            if (logger != null) {
                                logger.log(
                                    Level.SEVERE,
                                    "auto update on sync failed",
                                    ee);
                            }
                        }
                    }
                }
            });
        node.map();
    }

    public String id() {
        return this.id;
    }

    public void id(final String id) {
        this.id = id;
    }

    public String lockData() throws ZooException {
        final byte[] data = node.data(true);
        if (data == null) {
            return "null";
        }
        return new String(data);
    }

    public void setTimeout(final int timeout) {
        this.lockTimeout = timeout;
    }

    public void setAutoExpire(final boolean expire) {
        if (expire && expirer == null) {
            expirer = expirerScheduler.scheduleAtFixedRate(
                this,
                1000,
                1000,
                TimeUnit.MILLISECONDS);
        }
        if (!expire && expirer != null) {
            expirer.cancel(true);
            expirer = null;
        }
    }

    public void setAutoUpdateOnSync(final boolean update) {
        this.autoUpdateOnSync = update;
    }

    public void addExpireCallback(final Runnable callback) {
        expireCallbacks.add(callback);
    }

    @Override
    public void run() {
        long lastCheck = lastCheckTime;
        try {
            boolean runCallbacks = false;
            if (isLocked(false)) {
                if (logger != null) logger.info("Expire check");
                if (TimeSource.INSTANCE.currentTimeMillis() - lastCheck > lockTimeout) {
                    if (logger != null) logger.info("Lock expired. Unlocking.");
                    runCallbacks = true;
                    unlock();
                }
            } else {
                runCallbacks = true;
            }
            if (runCallbacks) {
                for (final Runnable cb : expireCallbacks) {
                    try {
                        cb.run();
                    } catch (Throwable t) {
                        if (logger != null) {
                            logger.log(
                                Level.SEVERE,
                                "Expire callback execution failed",
                                t);
                        }
                    }
                }
            }
        } catch (ZooException e) {
            e.printStackTrace();
        }
    }

    public void setLogger(final Logger logger) {
        this.logger = logger;
        node.setLogger(logger);
    }

    private final String generateId() {
        return hostname + '_' + TimeSource.INSTANCE.currentTimeMillis() + '_'
            + UUID.randomUUID();
    }

    public boolean lock(final boolean auto) throws ZooException {
        lastCheckTime = TimeSource.INSTANCE.currentTimeMillis();
        autoLock.set(auto);
        if (!node.exists()) {
            try {
                node.create(id.getBytes());
                return checkId(true, false);
            } catch (ZooNodeExistsException e) {
                return checkId(true, false);
            }
        } else {
            return checkId(true, false);
        }
    }

    public void refresh() throws ZooException {
        if (isLocked(true)) {
            node.setData(id.getBytes());
        }
    }

    public boolean lockAndRefresh(final boolean auto) throws ZooException {
        boolean retval = lock(auto);
        if (retval) {
            refresh();
        }
        return retval;
    }

    public void forceLock(final boolean auto) throws ZooException {
        while (!lock(auto)) {
            node.delete();
        }
    }

    public void forceUnlock() throws ZooException {
        node.delete();
    }

    public void lockAsync() {
        autoLock.set(true);
        if (logger != null) logger.fine("lockAsync");
        final VoidCallback createCallback = 
            new VoidCallback() {
                @Override
                public void dataImpl(Void v) {
                    if (logger != null) logger.fine("lockAsync.createcb.dataImpl");
                    checkIdSilent();
                }
                @Override
                public void errorImpl(ZooException e) {
                    if (logger != null) logger.fine("lockAsync.createcb.errorImpl");
                    if (e instanceof ZooNodeExistsException) {
                        checkIdSilent();
                    } else {
                        if (logger != null) {
                            logger.log(Level.SEVERE, "lockAsync error", e);
                        }
//                        lockAsync();
                    }
                }
                @Override
                public void dataChangedImpl() {
                    if (logger != null) logger.fine("lockAsync.createcb.dataChangedImpl");
                    lockAsync();
                }
            };

        node.exists(
            new BooleanCallback() {
                @Override
                public void dataImpl(Boolean exists) {
                    if (logger != null) logger.fine("lockAsync.existscb.dataImpl: " + exists);
                    if (exists) {
                        checkIdSilent();
                    } else {
                        node.create(id.getBytes(), createCallback);
                    }
                }
                @Override
                public void errorImpl(ZooException e) {
                    if (logger != null) logger.fine("lockAsync.existscb.errorImpl");
                    if (e instanceof ZooNodeExistsException) {
                        checkIdSilent();
                    } else {
                        if (logger != null) {
                            logger.log(Level.SEVERE, "lockAsync error", e);
                        }
//                        lockAsync();
                    }
                }
                @Override
                public void dataChangedImpl() {
                    if (logger != null) logger.fine("lockAsync.existscb.dataChangedImpl");
                    lockAsync();
                }
            });
    }

    public boolean isLocked() throws ZooException {
        if (!node.exists(false)) {
            return false;
        }
        return checkId(true, false);
    }

    public boolean isLockedNonBlock() throws ZooException {
        if (!node.exists(true)) {
            return false;
        }
        return checkId(true, true);
    }

    public boolean isLocked(final boolean updateTime) throws ZooException {
        if (!node.exists()) {
            return false;
        }
        return checkId(updateTime, false);
    }

    private boolean checkId(final boolean updateTime, final boolean nonBlock)
        throws ZooException
    {
        byte[] data = node.data(nonBlock);
        if (data == null) {
            if (logger != null) logger.fine("checkId: data=null");
            return false;
        }
        String dataString = new String(data);
        if (logger != null) logger.fine("checkId: data=" + dataString + ", id=" + id);
        if (dataString.equals(id)) {
            unparkAll();
            if (updateTime) {
                lastCheckTime = TimeSource.INSTANCE.currentTimeMillis();
            }
            return true;
        }
        return false;
    }

    private void checkIdAsync() {
        final ByteArrayCallback dataCb =
            new ByteArrayCallback() {
                @Override
                public void dataImpl(final byte[] data) {
                    if (data == null) {
                        if (logger != null) logger.fine("checkIdAsync: data=null");
                        lockAsync();
                        return;
                    }
                    String dataString = new String(data);
                    if (logger != null) logger.fine("checkIdAsync: data=" + dataString + ", id=" + id);
                    if (dataString.equals(id)) {
                        unparkAll();
                        return;
                    }
                }
                @Override
                public void errorImpl(ZooException e) {
                    if (logger != null) logger.fine("checkIdAsync: error=" + e);
                    lockAsync();
                }
                @Override
                public void dataChangedImpl() {
                    if (logger != null) logger.fine("checkIdAsync: dataChanged=");
                    lockAsync();
                }
            };
        node.data(dataCb);
    }

    private void checkIdSilent() {
        if (logger != null) logger.fine("checkIdSilent");
        checkIdAsync();
    }

    private void unparkAll() {
        if (logger != null) logger.fine("unparkAll");
        UnparkCallback ucb;
        while ((ucb = parked.poll()) != null) {
            final UnparkCallback cb = ucb;
            unparkExecutor.execute(
                new Runnable() {
                    @Override
                    public void run() {
                        cb.unpark();
                    }
                });
        }
    }

    public void unlock() throws ZooException {
        autoLock.set(false);
        if (!lock(false)) {
            return;
        }
        node.delete();
    }

    public void park(final UnparkCallback unparkCb) {
        parked.add(unparkCb);
        lockAsync();
    }

    @Override
    public void close() throws ZooException {
        unlock();
    }

    public void unmap() {
        node.unmap();
    }
}

