package ru.yandex.dispatcher.common.mappedvars;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Logger;

import ru.yandex.dispatcher.common.BooleanCallback;
import ru.yandex.dispatcher.common.ByteArrayCallback;
import ru.yandex.dispatcher.common.VoidCallback;
import ru.yandex.dispatcher.common.ZooCallback;
import ru.yandex.dispatcher.common.ZooException;
import ru.yandex.dispatcher.common.ZooNoNodeException;
import ru.yandex.dispatcher.common.ZooNodeExistsException;
import ru.yandex.dispatcher.common.connection.ZooConnection;
import ru.yandex.dispatcher.common.connection.ZooConnectionCallback;
import ru.yandex.util.timesource.TimeSource;

public class ZooNodeMapping implements ZooConnectionCallback {
    private static final int DEFAULT_UPDATE_TIMEOUT = 1000;

    private final ZooConnection conn;
    private final String path;
    private final AtomicBoolean watchInProgress =
        new AtomicBoolean(false);
    private final AtomicReference<SyncCallbackAdapter> syncInProgress =
        new AtomicReference<>(null);
    private final ConcurrentHashMap<SyncCallback,Object> syncCallbacks =
        new ConcurrentHashMap<SyncCallback,Object>();

    private int timeout;
    private volatile byte[] data;
    private volatile long lastSync;
    private volatile boolean exists;
    private boolean persistent;
    private boolean autoCreate;
    private Logger logger = null;

    public ZooNodeMapping(final ZooConnection conn, final String path) {
        this.conn = conn;
        this.path = path.intern();
        this.lastSync = 0;
        this.exists = false;
        this.persistent = false;
        this.autoCreate = false;
        this.timeout = DEFAULT_UPDATE_TIMEOUT;
    }

    public void map() {
        conn.register(this);
        if (conn.isConnected()) {
            sync();
        }
    }

    public void unmap() {
        conn.unregister(this);
    }

    public String path() {
        return this.path;
    }

    @Override
    public void connected() {
        sync();
    }

    @Override
    public void disconnected() {
        lastSync = 0;
        watchInProgress.set(false);
    }

    public void setLogger(final Logger logger) {
        this.logger = logger;
    }

    public void setTimeout(final int timeout) {
        this.timeout = timeout;
    }

    public void setPersistent(final boolean persistent) {
        this.persistent = persistent;
    }

    public void setAutoCreate(final boolean autoCreate) {
        this.autoCreate = autoCreate;
    }

    public void registerSyncCallback(final SyncCallback cb) {
        syncCallbacks.put(cb, cb);
    }

    public boolean exists() throws ZooException {
        return exists(false);
    }

    public boolean exists(final boolean nonBlock) throws ZooException {
        trySync(nonBlock);
        return exists;
    }

    public void exists(final BooleanCallback callback) {
        trySync(
            new SyncCallback() {
                @Override
                public void synced() {
                    callback.data(exists);
                }

                @Override
                public void error(ZooException e) {
                    callback.error(e);
                }
            });
    }

    public byte[] data() throws ZooException {
        return data(false);
    }

    public byte[] data(final boolean nonBlock) throws ZooException {
        trySync(nonBlock);
        return data;
    }

    public void data(final ByteArrayCallback callback) {
        trySync(
            new SyncCallback() {
                @Override
                public void synced() {
                    callback.data(data);
                }

                @Override
                public void error(ZooException e) {
                    callback.error(e);
                }
            });
    }

    public void setData(final byte[] data) throws ZooException {
        final AtomicReference<ZooException> error =
            new AtomicReference<ZooException>(null);
        final AtomicBoolean finished = new AtomicBoolean(false);
        VoidCallback cb = new VoidCallback() {
            @Override
            public void dataImpl(Void v) {
                finished.set(true);
                synchronized(this) {
                    this.notify();
                }
            }

            @Override
            public void errorImpl(ZooException e) {
                finished.set(true);
                synchronized(this) {
                    error.set(e);
                    this.notifyAll();
                }
            }

            @Override
            public void dataChangedImpl() {
            }
        };
        synchronized(cb) {
            setData(data, cb);
            if (error.get() != null) {
                throw error.get();
            }
            if (finished.get()) {
                return;
            }
            try {
                cb.wait();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        if (error.get() != null) {
            throw error.get();
        }
    }

    public void setData(final byte[] data, final VoidCallback callback) {
        trySync(
            new SyncCallback() {
                @Override
                public void synced() {
                    if (exists) {
                        setNodeData(data, callback);
                    } else {
                        if (autoCreate) {
                            createNode(data, callback, true);
                        } else {
                            callback.error(new ZooNoNodeException(path));
                        }
                    }
                }

                @Override
                public void error(ZooException e) {
                    callback.error(e);
                }
            });
    }

    public void create(final byte[] data, final VoidCallback callback) {
        trySync(
            new SyncCallback() {
                @Override
                public void synced() {
                    if (exists) {
                        callback.error(new ZooNodeExistsException("path"));
                    } else {
                        createNode(data, callback, false);
                    }
                }

                @Override
                public void error(ZooException e) {
                    callback.error(e);
                }
            });
    }

    public void create(final byte[] data) throws ZooException {
        final AtomicReference<ZooException> error =
            new AtomicReference<ZooException>(null);
        final AtomicBoolean finished = new AtomicBoolean(false);
        VoidCallback cb = new VoidCallback() {
            @Override
            public void dataImpl(Void v) {
                finished.set(true);
                synchronized(this) {
                    this.notify();
                }
            }

            @Override
            public void errorImpl(ZooException e) {
                finished.set(true);
                synchronized(this) {
                    error.set(e);
                    this.notifyAll();
                }
            }

            @Override
            public void dataChangedImpl() {
            }
        };
        synchronized(cb) {
            create(data, cb);
            if (error.get() != null) {
                throw error.get();
            }
            if (finished.get()) {
                return;
            }
            try {
                cb.wait();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        if (error.get() != null) {
            throw error.get();
        }
    }

    public void delete(final VoidCallback callback) {
        trySync(
            new SyncCallback() {
                @Override
                public void synced() {
                    if (exists) {
                        deleteNode(callback);
                    } else {
                        callback.data(null);
                    }
                }

                @Override
                public void error(ZooException e) {
                    callback.error(e);
                }
            });
    }

    public void delete() throws ZooException {
        final AtomicReference<ZooException> error =
            new AtomicReference<ZooException>(null);
        final AtomicBoolean finished = new AtomicBoolean(false);
        VoidCallback cb = new VoidCallback() {
            @Override
            public void dataImpl(Void v) {
                finished.set(true);
                synchronized(this) {
                    this.notify();
                }
            }

            @Override
            public void errorImpl(ZooException e) {
                finished.set(true);
                synchronized(this) {
                    error.set(e);
                    this.notifyAll();
                }
            }

            @Override
            public void dataChangedImpl() {
            }
        };
        synchronized(cb) {
            delete(cb);
            if (error.get() != null) {
                throw error.get();
            }
            if (finished.get()) {
                return;
            }
            try {
                cb.wait();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        if (error.get() != null) {
            throw error.get();
        }
    }

    private void deleteNode(final VoidCallback callback) {
        conn.deleteNode(path, callback);
    }

    private void setNodeData(final byte[] data, final VoidCallback callback) {
        conn.setData(path, data, callback);
    }

    private void createNode(final byte[] data, final VoidCallback callback,
        final boolean setDataIfExists) {
        BooleanCallback createCallback = new BooleanCallback() {
            @Override
            public void dataImpl(Boolean b) {
                lastSync = 0;
                if (b) {
                    callback.data(null);
                } else {
                    //already exists
                    if (setDataIfExists) {
                        setNodeData(data, callback);
                    } else {
                        callback.error(
                            new ZooNodeExistsException("path: " + path));
                    }
                }
            }
            @Override
            public void dataChangedImpl() {
                lastSync = 0;
            }
            @Override
            public void errorImpl(ZooException e) {
                lastSync = 0;
                if (e instanceof ZooNoNodeException) {
                    int slash = path.lastIndexOf('/');
                    if (slash == -1 || slash == 0) {
                        callback.error(e);
                        return;
                    } 
                    String parent = path.substring(0, slash);
                    final BooleanCallback createParentCallback =
                        new BooleanCallback() {
                            @Override
                            public void dataImpl(Boolean b) {
                                createNode(data, callback, setDataIfExists);
                            }
                            @Override
                            public void dataChangedImpl() {
                            }
                            @Override
                            public void errorImpl(ZooException e) {
                                callback.error(e);
                            }
                        };
                    createParents(parent, createParentCallback);
                } else {
                    callback.error(e);
                }
            }
        };
        conn.createNode(path, data, persistent, createCallback);
    }

    private void createParents(final String path,
        final BooleanCallback callback) {
        final BooleanCallback createParentCallback = new BooleanCallback() {
            @Override
            public void dataImpl(Boolean b) {
                callback.data(b);
            }
            @Override
            public void dataChangedImpl() {
            }
            @Override
            public void errorImpl(ZooException e) {
                if (e instanceof ZooNoNodeException) {
                    int slash = path.lastIndexOf('/');
                    if (slash == -1 || slash == 0) {
                        callback.error(e);
                        return;
                    } else {
                        String parent = path.substring(0, slash);
                        createParents(parent, this);
                    }
                } else {
                    callback.error(e);
                }
            }
        };
        //parents are only valid for persistent nodes
        conn.createNode(path, null, true, createParentCallback);
    }

    private void waitCallback(
        final SyncCallbackAdapter cb,
        final boolean nonBlock)
        throws ZooException
    {
        synchronized(cb) {
            if (cb.error() != null) {
                throw cb.error();
            }
            if (cb.finished()) {
                return;
            }
            if (!nonBlock) {
                try {
                    cb.wait();
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }
        if (cb.error() != null) {
            throw cb.error();
        }
    }

    protected void trySync() throws ZooException {
        trySync(false);
    }

    protected void trySync(final boolean nonBlock) throws ZooException {
        if (lastSync + timeout < TimeSource.INSTANCE.currentTimeMillis()
            && syncInProgress.get() == null)
        {
            SyncCallbackAdapter cb = new SyncCallbackAdapter(syncInProgress);
            while (lastSync + timeout < TimeSource.INSTANCE.currentTimeMillis()) {
                if (syncInProgress.compareAndSet(null, cb)) {
                    trySync(cb);
                    waitCallback(cb, nonBlock);
                    break;
                } else {
                    SyncCallbackAdapter runningCb = syncInProgress.get();
                    if (runningCb != null) {
                        waitCallback(runningCb, nonBlock);
                        break;
                    }
                }
            }
        }
    }

    protected void updateData(final byte[] data) {
        this.data = data;
    }

    private void trySync(final SyncCallback callback) {
        if (lastSync + timeout < TimeSource.INSTANCE.currentTimeMillis()) {
            boolean watch = watchInProgress.compareAndSet(false, true);
            conn.data(
                path,
                new ByteArrayCallback() {
//                    private final long ver =
//                        changeNotifyVersion.incrementAndGet();
                    @Override
                    public void dataImpl(byte[] data) {
                        exists = true;
                        updateData(data);
                        lastSync = TimeSource.INSTANCE.currentTimeMillis();
                        callback.synced();
                    }
                    @Override
                    public void errorImpl(ZooException e) {
                        watchInProgress.set(false);
                        if (e instanceof ZooNoNodeException) {
                            exists = false;
                            updateData(null);
                            lastSync = TimeSource.INSTANCE.currentTimeMillis();
                            callback.synced();
                        } else {
                            callback.error(e);
                        }
                    }
                    @Override
                    public void dataChangedImpl() {
                        ZooNodeMapping.this.dataChanged();
                    }
                },
                watch);
        } else {
            callback.synced();
        }
    }

    private void sync() {
        boolean watch = watchInProgress.compareAndSet(false, true);
        conn.data(
            path,
            new ByteArrayCallback() {
//                private final long ver = changeNotifyVersion.incrementAndGet();
                @Override
                public void dataImpl(byte[] data) {
                    exists = true;
                    updateData(data);
                    lastSync = TimeSource.INSTANCE.currentTimeMillis();
                    notifySync();
                }
                @Override
                public void errorImpl(ZooException e) {
                    if (logger != null) logger.fine("ZooNodeMapping.sync.errorImpl: " + e);
                    watchInProgress.set(false);
                    if (e instanceof ZooNoNodeException) {
                        exists = false;
                        updateData(null);
                        lastSync = TimeSource.INSTANCE.currentTimeMillis();
                        notifySync();
                    }
                }
                @Override
                public void dataChangedImpl() {
                    ZooNodeMapping.this.dataChanged();
                }
            },
            watch);
    }

    private void dataChanged() {
        lastSync = 0;
        watchInProgress.set(false);
        sync();
    }

    private void notifySync() {
        for (SyncCallback scb : syncCallbacks.keySet()) {
            scb.synced();
        }
    }

    private static class SyncCallbackAdapter implements SyncCallback {
        private final AtomicReference<ZooException> error =
            new AtomicReference<ZooException>(null);
        private final AtomicBoolean finished = new AtomicBoolean(false);
        private final AtomicReference<SyncCallbackAdapter> syncInProgress;

        public SyncCallbackAdapter(
            final AtomicReference<SyncCallbackAdapter> syncInProgress)
        {
            this.syncInProgress = syncInProgress;
        }

        @Override
        public void synced() {
            finished.set(true);
            synchronized(this) {
                this.notifyAll();
            }
            syncInProgress.compareAndSet(this, null);
        }

        @Override
        public void error(ZooException e) {
            finished.set(true);
            synchronized(this) {
                error.set(e);
                this.notifyAll();
            }
            syncInProgress.compareAndSet(this, null);
        }

        public boolean finished() {
            return finished.get();
        }

        public ZooException error() {
            return error.get();
        }
    }

}
