package ru.yandex.dispatcher.consumer;

import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.IntPair;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.dispatcher.common.HttpMessage;
import ru.yandex.dispatcher.consumer.shard.Shard;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.proxy.MultiClientTvm2TicketSupplier;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.stater.ImmutableStatersConfig;
import ru.yandex.util.timesource.TimeSource;

public abstract class AbstractHttpBackendConsumer implements BackendConsumer {
    private final HttpHost queueIdHost;
    private final String service;
    private final int workers;
    protected final Logger logger;
    protected final MultiClientTvm2TicketSupplier tvmRenewalTask;
    protected final PrefixedLogger outlog;
    protected final HttpHost host;
    protected final AsyncClient httpClient;
    private final long timeout;
    private final long watchdogDelay;
    protected final boolean zeroTolerance;
    protected ExecutorService workExecutor;

    public AbstractHttpBackendConsumer(
        final HttpHost host,
        final HttpHost queueIdHost,
        final String service,
        final int workers,
        final ConsumerServer server,
        final ImmutableHttpTargetConfig targetConfig,
        final Logger logger,
        final long watchdogDelay,
        final boolean zeroTolerance)
    {
        this.host = host;
        this.queueIdHost = queueIdHost;
        this.service = service;
        this.workers = workers;
        this.logger = logger;
        tvmRenewalTask = server.tvm2TicketSupplier();
        if (server.outlog() != null) {
            outlog = server.outlog().addPrefix(service);
        } else {
            outlog = null;
        }
        AsyncClient httpClient =
            new AsyncClient(server.reactor(service, null), targetConfig);
        ImmutableStatersConfig statersConfig = targetConfig.statersConfig();
        if (statersConfig == null) {
            this.httpClient = httpClient;
        } else {
            this.httpClient = httpClient.adjustAsteriskStater();
            server.registerStaters(statersConfig);
        }

        timeout = targetConfig.timeout();
        this.watchdogDelay = watchdogDelay;
        this.zeroTolerance = zeroTolerance;
    }

    @Override
    public long maxNodeTimeout() {
        return watchdogDelay;
    }

    @Override
    public ShardsPositions getPositions(final String service) {
        return null;
    }

    @Override
    public HttpHost host() {
        return host;
    }

    public String service() {
        return service;
    }

    public int workers() {
        return workers;
    }

//    @Override
    public HttpHost queueIdHost() {
        return queueIdHost;
    }

    public AsyncClient httpClient() {
        return httpClient;
    }

    public long timeout() {
        return timeout;
    }

    @Override
    public void start() {
        if (logger.isLoggable(Level.INFO)) {
            logger.info(
                "BackendConsumer.start executor: threads=" + workers
                + ", targetHost=" + host + ", service=" + service
                + ", watchdogDelay=" + maxNodeTimeout());
        }
        httpClient.start();
        workExecutor = new ThreadPoolExecutor(
        	workers,
    	        workers,
        	1,
        	TimeUnit.DAYS,
        	new LinkedBlockingQueue<Runnable>(),
        	new NamedThreadFactory(
                    getClass().getSimpleName() + '-' + service + '-',
                    true));
    }

    @Override
    public void updatePosition(final Shard shard, final long pos) {
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        return httpClient.status(verbose);
    }

    protected String deadline(final String cgiParams) {
        if (timeout == 0L) {
            return cgiParams;
        } else {
            StringBuilder sb = new StringBuilder(cgiParams);
            if (sb.length() != 0) {
                sb.append('&');
            }
            sb.append("deadline=");
            sb.append(TimeSource.INSTANCE.currentTimeMillis() + timeout);
            return new String(sb);
        }
    }

    protected Future<IntPair<Void>> runMessage(
        final HttpMessage msg,
        final long queueId,
        final int shard,
        final String service,
        final String cgiParams)
    {
        if (tvmRenewalTask != null) {
            msg.deleteHeader(YandexHeaders.X_YA_SERVICE_TICKET);
            msg.setHeader(YandexHeaders.X_YA_SERVICE_TICKET, tvmRenewalTask.ticket());
        }
        return msg.run(
            host,
            httpClient,
            queueId,
            shard,
            service,
            deadline(cgiParams));
    }

    protected void runMessage(
        final HttpMessage msg,
        final long queueId,
        final int shard,
        final String service,
        final String cgiParams,
        final FutureCallback<IntPair<Void>> callback)
    {
        if (tvmRenewalTask != null) {
            msg.deleteHeader(YandexHeaders.X_YA_SERVICE_TICKET);
            msg.setHeader(YandexHeaders.X_YA_SERVICE_TICKET, tvmRenewalTask.ticket());
        }
        msg.run(
            host,
            httpClient,
            queueId,
            shard,
            service,
            deadline(cgiParams),
            callback);
    }
}

