package ru.yandex.dispatcher.consumer;

import java.io.IOException;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TimerTask;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.collection.IntPair;
import ru.yandex.dispatcher.common.DelayShardMessage;
import ru.yandex.dispatcher.common.HangShardMessage;
import ru.yandex.dispatcher.common.HttpMessage;
import ru.yandex.dispatcher.common.SerializeUtils;
import ru.yandex.dispatcher.consumer.shard.Shard;
import ru.yandex.function.BasicConsumer;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServerErrorStatusPredicate;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryParameter;
import ru.yandex.parser.uri.UriParser;
import ru.yandex.util.timesource.TimeSource;

public class AsyncConsumer extends AbstractHttpBackendConsumer {
    private static final long RETRY_DELAY = 1000L;
    private static final long RATE_LIMIT_DELAY = 10000L;
    private static final long WRITE_ERROR_TIME_THRESHOLD = 5 * 60 * 1000;

    protected final boolean queueIdHeader;
    private final Function<String, Object> primaryKeyFactory;
    private final boolean groupNullKeys;
    private final int maxRetryCount;
    private final Function<String, Object> deduplicationKeyFactory;
    private final boolean ignoreBackendPosition;
    private final BackendCommitPolicy commitPolicy;
    private final long rateLimitDelay;

    public AsyncConsumer(
        final HttpHost host,
        final HttpHost queueIdHost,
        final String service,
        final int workers,
        final ConsumerServer server,
        final ImmutableHttpTargetConfig targetConfig,
        final List<String> primaryKeyFields,
        final boolean groupNullKeys,
        final boolean queueIdHeader,
        final int maxRetryCount,
        final List<String> deduplicationKeyFields,
        final boolean ignoreBackendPosition,
        final BackendCommitPolicy commitPolicy,
        final long watchdogDelay,
        final boolean zeroTolerance,
        final long rateLimitDelay)
    {
        super(
            host,
            queueIdHost,
            service,
            workers,
            server,
            targetConfig,
            server.logger().replacePrefix("AsyncConsumer"),
            watchdogDelay,
            zeroTolerance);
        this.rateLimitDelay = rateLimitDelay;
        this.queueIdHeader = queueIdHeader;
        this.maxRetryCount = maxRetryCount;
        Supplier<?> groupDefaultSupplier;
        if (groupNullKeys) {
            groupDefaultSupplier = Object::new;
        } else {
            groupDefaultSupplier = new BasicConsumer<>();
        }
        primaryKeyFactory =
            createKeyFactory(primaryKeyFields, groupDefaultSupplier);
        this.groupNullKeys = groupNullKeys;
        deduplicationKeyFactory =
            createKeyFactory(deduplicationKeyFields, Object::new);
        this.ignoreBackendPosition = ignoreBackendPosition;
        this.commitPolicy = commitPolicy;
    }

    public BackendCommitPolicy commitPolicy() {
        return commitPolicy;
    }

    public boolean commitPositionAllowed() {
        return commitPolicy.ordinal() < BackendCommitPolicy.NEVER.ordinal();
    }

    private static Function<String, Object> createKeyFactory(
        final List<String> fields,
        final Supplier<?> defaultValueSupplier)
    {
        Function<String, Object> keyFactory;
        switch (fields.size()) {
            case 0:
                keyFactory = x -> defaultValueSupplier.get();
                break;
            case 1:
                keyFactory =
                    new BasicKeyFactory(fields.get(0), defaultValueSupplier);
                break;
            case 2:
                keyFactory = new DoubleKeyFactory(
                    fields.get(0),
                    fields.get(1),
                    defaultValueSupplier);
                break;
            default:
                keyFactory = new ListKeyFactory(fields, defaultValueSupplier);
                break;
        }
        return keyFactory;
    }

    @Override
    public void dispatchNodes(final List<Node> nodes, final Shard shard) {
        try {
            List<List<Node>> groups = groups(nodes, shard);
            if (groups == null) {
                return;
            }
            ProcessContext ctx = new ProcessContext(nodes, shard, this);
            int count = groups.size();
            if (logger.isLoggable(Level.FINE)) {
                shard.logger().fine(
                    nodes.size() + " nodes splitted in " + count + " groups");
            }
            GroupProcessor gp =
                new GroupProcessor(
                    ctx,
                    groups,
                    0,
                    new NodeFinishedCallback(nodes, shard));
            gp.process();
        } catch (Throwable t) {
            shard.logger().log(Level.SEVERE, "Processing failed", t);
        }
    }

    protected List<List<Node>> groups(final List<Node> nodes, final Shard shard)
        throws IOException
    {
        final int size = nodes.size();
        final Logger logger = shard.logger();
        Map<Object, Node> deduplicated =
            new LinkedHashMap<>(size << 1, 0.5f, true);
        for (Node node: nodes) {
            if (node.msg == null) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("Loading node " + node.path);
                }
                node.msg =
                    SerializeUtils.deserializeHttpMessage(node.data);
            }
            String uri = node.msg.uri();
            Object deduplicationKey;
            if (uri == null) {
                deduplicationKey = new Object();
            } else {
                deduplicationKey = deduplicationKeyFactory.apply(uri);
            }
            deduplicated.put(deduplicationKey, node);
        }
        Node lastNode = null;
        Iterator<Node> iter = deduplicated.values().iterator();
        while (true) {
            if (iter.hasNext()) {
                lastNode = iter.next();
            } else {
                iter.remove();
                break;
            }
        }
        Set<Object> primaryKeys = new HashSet<>(size << 1);
        List<List<Node>> groups = new ArrayList<>(size);
        List<Node> currentGroup = new ArrayList<>(1);
        for (Node node: deduplicated.values()) {
            String uri = node.msg.uri();
            if (uri == null) {
                if (node.msg instanceof HangShardMessage) {
                    logger.info("Hang requested");
                    // fail fast
                    shard.hang(node);
                    return null;
                }
                if (node.msg instanceof DelayShardMessage) {
                    if (!currentGroup.isEmpty()) {
                        groups.add(currentGroup);
                        currentGroup = new ArrayList<>(1);
                        primaryKeys.clear();
                    }
                    groups.add(Collections.singletonList(node));
                }
            } else {
                Object primaryKey = primaryKeyFactory.apply(uri);
                if (primaryKeys.add(primaryKey)) {
                    currentGroup.add(node);
                } else {
                    groups.add(currentGroup);
                    currentGroup = new ArrayList<>(1);
                    currentGroup.add(node);
                    primaryKeys.clear();
                    primaryKeys.add(primaryKey);
                }
            }
        }
        if (!currentGroup.isEmpty()) {
            groups.add(currentGroup);
        }
        if (lastNode != null) {
            groups.add(Collections.singletonList(lastNode));
        }
        return groups;
    }

    public static void scheduleRetry(
        final AsyncClient client,
        final NodeProcessor callback,
        final long delay)
    {
        scheduleTask(client, new RetryTask(callback), delay);
    }

    public static void scheduleDelay(
        final AsyncClient client,
        final GroupProcessor callback,
        final long delay)
    {
        scheduleTask(client, new DelayTask(callback), delay);
    }

    public static void scheduleTask(
        final AsyncClient client,
        final TimerTask task,
        final long delay)
    {
        client.scheduleRetry(task, delay);
    }

    public ShardsPositions getPositions(final String service) {
        if (ignoreBackendPosition
            && commitPolicy != BackendCommitPolicy.ALWAYS_FORCE)
        {
            return null;
        } else {
            return new LuceneShardsPosition(service, logger);
        }
    }

    public boolean queueIdHeader() {
        return queueIdHeader;
    }

    @Override
    public void start() {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine(
                "AsyncBackendConsumer.start httpClient: threads=" + workers()
                    + ", targetHost=" + host() + ", service=" + service()
                    + ", watchdogDelay=" + maxNodeTimeout());
        }
        httpClient().start();
    }

    protected class ProcessContext {
        private final List<Node> nodes;
        private final Shard shard;
        private final AsyncConsumer consumer;
        private final AsyncClient httpClient;
        protected final Logger logger;

        public ProcessContext(
            final List<Node> nodes,
            final Shard shard,
            final AsyncConsumer consumer)
        {
            this.nodes = nodes;
            this.shard = shard;
            this.consumer = consumer;
            this.httpClient = consumer.httpClient();
            logger = shard.logger();
        }

        public boolean queueIdHeader() {
            return consumer.queueIdHeader();
        }

        public AsyncClient httpClient() {
            return httpClient;
        }

        private boolean shouldStop() {
            final boolean result;
            if (!shard.checkLock()) {
                logger.fine(
                    "Processing node stopped: lock consumer lock expired");
                result = true;
            } else if (shard.shouldReset()) {
                logger.fine(
                    "Processing node stopped: shard was resetted");
                shard.finishReset();
                result = true;
            } else if (shard.consumerExpired(nodes)) {
                result = true;
            } else {
                result = false;
            }
            return result;
        }

        public Shard shard() {
            return shard;
        }

        public Logger logger() {
            return logger;
        }

        public List<Node> nodes() {
            return nodes;
        }

        public PrefixedLogger outlog() {
            return consumer.outlog;
        }

        public void runMessage(
            final HttpMessage msg,
            final long queueId,
            final int shard,
            final String service,
            final String cgiParams,
            final FutureCallback<IntPair<Void>> callback)
        {
            consumer.runMessage(
                msg,
                queueId,
                shard,
                service,
                cgiParams,
                callback);
        }

        public int maxRetryCount() {
            return consumer.maxRetryCount;
        }

        public void updatePosition(final long position) {
            consumer.updatePosition(shard, position);
        }
    }

    private static class GroupProcessor implements FutureCallback<List> {
        private final ProcessContext ctx;
        private final List<List<Node>> groups;
        private final int currentGroup;
        private final NodeFinishedCallback finalCallback;

        GroupProcessor(
            final ProcessContext ctx,
            final List<List<Node>> groups,
            final int currentGroup,
            final NodeFinishedCallback finalCallback)
        {
            this.ctx = ctx;
            this.groups = groups;
            this.currentGroup = currentGroup;
            this.finalCallback = finalCallback;
        }

        @Override
        public void completed(final List result) {
            if (currentGroup >= groups.size()) {
                finalCallback.completed(null);
            } else {
                process();
            }
        }

        @Override
        public void failed(final Exception e) {
            if (ctx.logger().isLoggable(Level.SEVERE)) {
                ctx.logger().log(Level.SEVERE, "Unhandled exception", e);
            }
        }

        @Override
        public void cancelled() {
        }

        public void process() {
            if (ctx.logger().isLoggable(Level.FINE)) {
                ctx.logger().fine("Processing group #" + currentGroup);
            }
            if (ctx.shouldStop()) {
                finalCallback.cancelled();
                return;
            }
            List<Node> nodes = groups.get(currentGroup);
            if (nodes.size() == 1) {
                HttpMessage msg = nodes.get(0).msg;
                if (msg instanceof DelayShardMessage) {
                    scheduleDelay(
                        ctx.httpClient(),
                        new GroupProcessor(
                            ctx,
                            groups,
                            currentGroup + 1,
                            finalCallback),
                        ((DelayShardMessage) msg).delay());
                    return;
                }
            }
            final BitSet finishedBits = new BitSet(nodes.size());
            final NodeProcessor[] processors = new NodeProcessor[nodes.size()];
            final MultiFutureCallback multiCallback =
                new MultiFutureCallback(
                    new GroupProcessor(
                        ctx,
                        groups,
                        currentGroup + 1,
                        finalCallback));
            for (int i = 0; i < nodes.size(); i++) {
                NodeProcessor np =
                    new NodeProcessor(
                        i,
                        nodes.get(i),
                        finishedBits,
                        multiCallback.newCallback(),
                        processors,
                        ctx);
                processors[i] = np;
            }
            multiCallback.done();
            for (NodeProcessor np: processors) {
                np.process();
            }
        }
    }

    private static class NodeProcessor
        implements FutureCallback<IntPair<Void>>
    {
        private final Node node;
        private final int nodeNumber;
        private final BitSet finishedBits;
        private final FutureCallback<IntPair<Void>> callback;
        private final NodeProcessor[] processors;
        private final ProcessContext ctx;
        private Exception error = null;
        private int errorStatus = 0;
        private boolean postProcessed = false;
        private String processingStatus;
        private final long queueId;

        NodeProcessor(
            final int nodeNumber,
            final Node node,
            final BitSet finishedBits,
            final FutureCallback<IntPair<Void>> callback,
            final NodeProcessor[] processors,
            final ProcessContext ctx)
        {
            this.nodeNumber = nodeNumber;
            this.node = node;
            this.finishedBits = finishedBits;
            this.callback = callback;
            this.processors = processors;
            this.ctx = ctx;
            if (nodeNumber == 0 && ctx.queueIdHeader()) {
                queueId = node.seq;
            } else {
                queueId = Integer.MIN_VALUE;
            }
        }

        public void process() {
            if (ctx.logger().isLoggable(Level.FINE)) {
                ctx.logger().fine(
                    "Processing node: " + node.path
                        + ' ' + '(' + (nodeNumber + 1) + " of " + processors.length
                        + "), type: " + node.msg
                        + ", path: " + node.path
                        + ", delayStatus: " + !node.notifyStatus
                        + ", queueTime="
                        + (node.msg.readTime() - node.msg.createTime()));
            }
            ctx.runMessage(
                node.msg,
                queueId,
                ctx.shard().shardNo,
                ctx.shard().service(),
                "zoo-queue-id=" + node.seq,
                this);
        }

        @Override
        public void completed(final IntPair<Void> code) {
            if (ctx.logger().isLoggable(Level.FINE)) {
                ctx.logger().fine(
                    "Node finished: " + node.path);
            }
            processingStatus = Long.toString(code.first());

            boolean lastCompleted;
            synchronized(finishedBits) {
                finishedBits.set(nodeNumber);
                lastCompleted = checkPostProcess();
            }

            if (ctx.consumer.commitPositionAllowed()) {
                if (lastCompleted) {
                    ShardsPositions positions =
                        ctx.consumer.getPositions(node.shard.service());
                    if (positions != null
                        && ctx.consumer.commitPolicy.ordinal()
                        <= BackendCommitPolicy.ALWAYS.ordinal())
                    {
                        // ok looks like only one shard possible
                        new UpdateLucenePositionCallback(
                            this,
                            positions,
                            processors[processors.length -1].node.seq,
                            null,
                            callback).retry();
                        return;
                    }
                }
            }

            callback.completed(code);
        }

        private void outlog() {
            final PrefixedLogger outlog = ctx.outlog();
            if (outlog != null) {
                outlog.info(
                    ctx.shard().shardNo + outlog.separator()
                    + node.seq + outlog.separator()
                    + (node.msg.readTime() - node.msg.createTime())
                    + outlog.separator()
                    + (TimeSource.INSTANCE.currentTimeMillis() - node.msg.readTime())
                    + outlog.separator() + node.msg.uri()
                    + outlog.separator() + processingStatus);
            }
        }

        private boolean checkPostProcess() {
            //check if previous nodes were finished
            int prevClear = finishedBits.previousClearBit(nodeNumber);
            if (prevClear != -1) {
                if (ctx.logger().isLoggable(Level.FINE)) {
                    ctx.logger().fine("PostProcess is postponed, prev node has "
                        + "not been finished yet: " + (prevClear + 1)
                        + ": " + processors[prevClear].node.seq);
                }
                return false;
            }
            //all previous are finished, check forward
            int nextClear = finishedBits.nextClearBit(nodeNumber + 1);
            if (ctx.logger().isLoggable(Level.FINE)) {
                ctx.logger().fine("PostProcess.nextClear: " + nextClear);
            }
            NodeProcessor nextReady = processors[nextClear - 1];
            nextReady.postProcess();
            return nextClear == processors.length;
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(final Exception e) {
            if (ctx.logger().isLoggable(Level.WARNING)) {
                StringBuilder sb = new StringBuilder("Processing node ");
                sb.append(node.path);
                sb.append(" failed");
                if (e instanceof BadResponseException) {
                    sb.append(':');
                    sb.append(' ');
                    ((BadResponseException) e).getShortMessage(sb);
                    ctx.logger().warning(new String(sb));
                } else {
                    ctx.logger().log(Level.WARNING, new String(sb), e);
                }
            }

            int status = 0;
            if (e instanceof BadResponseException) {
                status = ((BadResponseException) e).statusCode();
                processingStatus = Integer.toString(status);
                if (ServerErrorStatusPredicate.INSTANCE.test(status)
                    || status == YandexHttpStatus.SC_UNAUTHORIZED
                    || status == YandexHttpStatus.SC_FORBIDDEN)
                {
                    status = 0;
                } else if (status == YandexHttpStatus.SC_TOO_MANY_REQUESTS
                    || status == YandexHttpStatus.SC_BUSY)
                {
                    status = 0;
                    ctx.logger().info("Hit rate limit, sleeping for " + ctx.consumer.rateLimitDelay);
                    scheduleRetry(ctx.httpClient, this, ctx.consumer.rateLimitDelay);
                    return;
                }
            } else {
                processingStatus = e.getClass().getName();
            }
            if (ctx.consumer.zeroTolerance
                || (status == 0
                    && (ctx.maxRetryCount() < 0
                        || node.retryCount++ < ctx.maxRetryCount()
                        || e instanceof TimeoutException)))
            {
                if (ctx.shouldStop()) {
                    if (ctx.logger().isLoggable(Level.INFO)) {
                        ctx.logger().info(
                            "Aborting processing at pos " + nodeNumber);
                    }
                    return;
                }
                scheduleRetry(ctx.httpClient(), this, RETRY_DELAY);
            } else {
                if (status == 0) {
                    if (ctx.logger().isLoggable(Level.INFO)) {
                        ctx.logger().severe(
                            "node max retry count exhaused: "
                                + "maxRetryCount=" + ctx.maxRetryCount()
                                + ", node.retryCount=" + node.retryCount);
                    }
                }
                long msgQueueTime = TimeSource.INSTANCE.currentTimeMillis()
                    - node.msg.createTime();
                if (ctx.logger().isLoggable(Level.INFO)) {
                    ctx.logger().info(
                        "msgTime=" + msgQueueTime
                            + ", threshold=" + WRITE_ERROR_TIME_THRESHOLD);
                }
                if (msgQueueTime < WRITE_ERROR_TIME_THRESHOLD) {
                    // do not write msg error if this is an old
                    // message
                    error = e;
                    errorStatus = status;
                }

                boolean allowedUpdatePosition =
                    ctx.consumer.commitPolicy.ordinal()
                        <= BackendCommitPolicy.FAILED.ordinal();

                ShardsPositions positions = null;

                synchronized(finishedBits) {
                    finishedBits.set(nodeNumber);
                    allowedUpdatePosition &= checkPostProcess();
                }

                outlog();

                if (allowedUpdatePosition) {
                    positions = ctx.consumer.getPositions(node.shard.service());
                    allowedUpdatePosition = positions != null;
                }

                if (allowedUpdatePosition) {
                    long seq = processors[processors.length - 1].node.seq;
                    String logStr = "OnFailure Position updated " + seq
                        + ' ' + node.path
                        + ' ' + node.shard;
                    new UpdateLucenePositionCallback(
                        this,
                        positions,
                        seq,
                        logStr,
                        callback).retry();
                } else {
                    callback.completed(null);
                }
            }
        }

        public void retry() {
            if (ctx.logger().isLoggable(Level.INFO)) {
                ctx.logger().info("Retrying " + node.path
                    + " try#: " + node.retryCount);
            }
            node.msg.deleteHeader(YandexHeaders.RETRY_COUNT);
            node.msg.setHeader(
                YandexHeaders.RETRY_COUNT,
                Integer.toString(node.retryCount));
            ctx.runMessage(
                node.msg,
                queueId,
                ctx.shard().shardNo,
                ctx.shard().service(),
                "zoo-queue-id=" + node.seq,
                this);
        }

        public void postProcess() {
            if (postProcessed) {
                return;
            }
            postProcessed = true;
            if (error != null) {
                ctx.shard().writeNodeError(
                    node,
                    error.getMessage(),
                    errorStatus);
            }
            ctx.shard().writeNodeStatus(node);
            if (ctx.logger().isLoggable(Level.INFO)) {
                ctx.logger().info("UpdatePosition: " + node.seq);
            }
            if (nodeNumber > 0) {
                ctx.updatePosition(node.seq);
            }
        }
    }

    private static final class UpdateLucenePositionCallback
        implements FutureCallback<Object>
    {
        private final NodeProcessor processor;
        private final ShardsPositions positions;
        private final long seq;
        private final String logString;
        private final FutureCallback<?> callback;

        public UpdateLucenePositionCallback(
            final NodeProcessor processor,
            final ShardsPositions positions,
            final long seq,
            final String logString,
            final FutureCallback<?> callback)
        {

            this.positions = positions;
            this.processor = processor;
            this.seq = seq;
            this.logString = logString;
            this.callback = callback;
        }

        public void retry() {
            positions.updatePosition(processor.ctx.shard().shardNo(), seq, this);
        }

        @Override
        public void failed(final Exception e) {
            if (processor.ctx.logger().isLoggable(Level.WARNING)) {
                processor.ctx.logger().log(
                    Level.WARNING,
                    "Failed to update backend position "
                        + seq + ' '
                        + processor.node.shard.shardNo
                        + ' '
                        + String.valueOf(logString));
            }
            scheduleTask(
                processor.ctx.httpClient(),
                new UpdatePositionRetryTask(this),
                RETRY_DELAY);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }

        @Override
        public void completed(final Object o) {
            if (logString != null) {
                if (processor.ctx.logger().isLoggable(Level.WARNING)) {
                    processor.ctx.logger().info(String.valueOf(logString));
                }

            }

            callback.completed(null);
        }
    }

    private static class NodeFinishedCallback implements FutureCallback {
        private final List<Node> nodes;
        private final Shard shard;

        NodeFinishedCallback(final List<Node> nodes, final Shard shard) {
            this.nodes = nodes;
            this.shard = shard;
        }

        @Override
        public void completed(final Object result) {
            try {
                shard.nodeFinished(nodes, nodes.get(nodes.size() - 1));
            } catch (Throwable t) {
                shard.logger().log(Level.SEVERE, "NodeFinish failed", t);
            }
        }

        @Override
        public void failed(final Exception e) {
        }

        @Override
        public void cancelled() {
        }
    }

    private static class BasicKeyFactory implements Function<String, Object> {
        private final String name;
        private final Supplier<?> defaultValueSupplier;

        public BasicKeyFactory(
            final String name,
            final Supplier<?> defaultValueSupplier)
        {
            this.name = name;
            this.defaultValueSupplier = defaultValueSupplier;
        }

        @Override
        public Object apply(final String request) {
            for (QueryParameter param
                    : new UriParser(request).queryParser(null))
            {
                if (name.equals(param.name())) {
                    return param.value().toString();
                }
            }
            return defaultValueSupplier.get();
        }
    }

    private static class DoubleKeyFactory implements Function<String, Object> {
        private final String first;
        private final String second;
        private final Supplier<?> defaultValueSupplier;

        public DoubleKeyFactory(
            final String first,
            final String second,
            final Supplier<?> defaultValueSupplier)
        {
            this.first = first;
            this.second = second;
            this.defaultValueSupplier = defaultValueSupplier;
        }

        @Override
        public Object apply(final String request) {
            Object first = null;
            Object second = null;
            for (QueryParameter param
                    : new UriParser(request).queryParser(null))
            {
                String name = param.name();
                if (first == null && this.first.equals(name)) {
                    first = param.value().toString();
                    if (second != null) {
                        break;
                    }
                } else if (second == null && this.second.equals(name)) {
                    second = param.value().toString();
                    if (first != null) {
                        break;
                    }
                }
            }
            if (first == null) {
                first = defaultValueSupplier.get();
            }
            if (second == null) {
                second = defaultValueSupplier.get();
            }
            return new DoubleKey(first, second);
        }
    }

    private static class DoubleKey {
        private final Object first;
        private final Object second;

        public DoubleKey(final Object first, final Object second) {
            this.first = first;
            this.second = second;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append('[');
            sb.append(first);
            sb.append(',');
            sb.append(second);
            sb.append(']');
            return new String(sb);
        }

        @Override
        public int hashCode() {
            return Objects.hashCode(first) ^ Objects.hashCode(second);
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof DoubleKey) {
                DoubleKey other = (DoubleKey) o;
                return Objects.equals(first, other.first)
                    && Objects.equals(second, other.second);
            }
            return false;
        }
    }

    private static class ListKeyFactory implements Function<String, Object> {
        private final List<String> keyFields;
        private final Supplier<?> defaultValueSupplier;

        public ListKeyFactory(
            final List<String> keyFields,
            final Supplier<?> defaultValueSupplier)
        {
            this.keyFields = keyFields;
            this.defaultValueSupplier = defaultValueSupplier;
        }

        @Override
        public Object apply(final String request) {
            List<Object> key = new ArrayList<>(keyFields.size());
            CgiParams params =
                new CgiParams(new UriParser(request).queryParser(null));
            for (String field: keyFields) {
                Object value = params.getString(field, null);
                if (value == null) {
                    value = defaultValueSupplier.get();
                }
                key.add(value);
            }
            return key;
        }
    }

    private class LuceneShardsPosition implements ShardsPositions {
        private final String service;
        private final String url;
        private final Logger logger;

        public LuceneShardsPosition(
            final String service,
            final Logger logger)
        {
            this.service = service;
            this.logger = logger;
            this.url = new String(
                "/getQueueId?service=" + service + "&shard=").intern();
        }

        @Override
        public long getPosition(int shard) throws IOException {
            final String request = url + Integer.toString(shard);
            final BasicAsyncRequestProducerGenerator get =
                new BasicAsyncRequestProducerGenerator(request);
            try {
                final String response =
                    httpClient().execute(
                        queueIdHost(),
                        get,
                        AsyncStringConsumerFactory.OK,
                        EmptyFutureCallback.INSTANCE)
                        .get(timeout() << 1, TimeUnit.MILLISECONDS);
                return Long.parseLong(response.trim());
            } catch (ExecutionException ee) {
                Throwable e = ee.getCause();
                if (e != null) {
                    if (e instanceof BadResponseException
                        && ((BadResponseException) e).statusCode() == 503)
                    {
                        throw new IOException("Can't retreive Queue position "
                            + "from <" + queueIdHost() + "> for shard: " + shard
                            + " shard not copied");
                    }
                } else {
                    e = ee;
                }

                throw new IOException("Can't retreive Queue position "
                    + "from <" + queueIdHost() + "> for shard: " + shard, e);
            } catch (Exception e) {
                throw new IOException("Can't retreive Queue position "
                    + "from <" + queueIdHost() + "> for shard: " + shard, e);
            }
        }

        @Override
        public void updatePosition(
            final int shard,
            final long position,
            final FutureCallback<Object> callback)
        {
            final String uri = "/delete?updatePosition&prefix="
                + Integer.toString(shard)
                + "&shard=" + Integer.toString(shard)
                + "&service=" + service
                + "&position=" + position;
            final String content = "{\"prefix\":" + shard
                + ",\"docs\":[]}";
            if (logger.isLoggable(Level.INFO)) {
                logger.info("LuceneConsumer: updatePosition for Shard<" + shard
                    + ">: " + position
                    + ": uri: " + uri + ", post: " + content);
            }
            final BasicAsyncRequestProducerGenerator post =
                new BasicAsyncRequestProducerGenerator(uri, content);
            post.addHeader(
                YandexHeaders.ZOO_QUEUE_ID,
                Long.toString(position));
            post.addHeader(
                YandexHeaders.ZOO_SHARD_ID,
                Integer.toString(shard));
            post.addHeader(YandexHeaders.ZOO_QUEUE, service);
            httpClient().execute(
                queueIdHost(),
                post,
                AsyncStringConsumerFactory.ANY_GOOD,
                callback);
        }
    }

    private static class RetryTask extends TimerTask {
        private final NodeProcessor callback;

        RetryTask(final NodeProcessor callback) {
            this.callback = callback;
        }

        @Override
        public void run() {
            callback.retry();
        }
    }

    private static class UpdatePositionRetryTask extends TimerTask {
        private final UpdateLucenePositionCallback callback;

        UpdatePositionRetryTask(final UpdateLucenePositionCallback callback) {
            this.callback = callback;
        }

        @Override
        public void run() {
            callback.retry();
        }
    }

    private static class DelayTask extends TimerTask {
        private final GroupProcessor callback;

        DelayTask(final GroupProcessor callback) {
            this.callback = callback;
        }

        @Override
        public void run() {
            callback.process();
        }
    }
}
