package ru.yandex.dispatcher.consumer;

import java.io.File;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpHost;

import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.stater.StatersConfig;
import ru.yandex.stater.StatersConfigBuilder;

public class BackendConsumerFactory {
    private final HttpHost host;
    private final HttpHost queueIdHost;
    private final String service;
    private final ConsumerServer server;

    public BackendConsumerFactory(
        final HttpHost host,
        final HttpHost queueIdHost,
        final String service,
        final ConsumerServer server)
    {
        this.host = host;
        this.queueIdHost = queueIdHost;
        this.service = service;
        this.server = server;
    }

    public BackendConsumer createConsumer(
        final IniConfig config,
        final int timeoutOverride)
        throws ConfigException
    {
        String consumerType = config.getString("consumer_type", "lucene");
        IniConfig consumerConfig = config.section(consumerType + "-consumer");
        int workers = consumerConfig.getInt("workers", 20);
        int prefetch = config.getInt("zoolooser.prefetch-count", 10);
        HttpTargetConfigBuilder targetConfigBuilder =
            new HttpTargetConfigBuilder(
                consumerConfig,
                new HttpTargetConfigBuilder()
                    .connections(workers * prefetch)
                    .timeout(30000)
                    .connectTimeout(3000)
                    .redirects(true));
        if (timeoutOverride != -1) {
            targetConfigBuilder.timeout(timeoutOverride);
            if (server.logger().isLoggable(Level.FINE)) {
                server.logger().fine("Overriding target timeout from "
                    + "searchmap: " + timeoutOverride);
            }
        }
        StatersConfig statersConfig = targetConfigBuilder.statersConfig();
        if (statersConfig != null) {
            StatersConfigBuilder statersConfigBuilder =
                new StatersConfigBuilder(statersConfig);
            statersConfigBuilder.staters().traverse(
                (pattern, cfg) -> {
                    if (!cfg.prefix().startsWith("ignore")) {
                        cfg.prefix(cfg.prefix() + '-' + service);
                    }
                });
            targetConfigBuilder.statersConfig(statersConfigBuilder);
        }
        long watchdogDelay = consumerConfig.getLongDuration(
            "watchdog-delay",
            (long) targetConfigBuilder.timeout());
        boolean zeroTolerance =
            consumerConfig.getBoolean("zero-tolerance", false);
        ImmutableHttpTargetConfig targetConfig =
            targetConfigBuilder.build();
        List<String> primaryKeyFields = consumerConfig.get(
            "primary-key-fields",
            Collections.emptyList(),
            new CollectionParser<>(
                NonEmptyValidator.TRIMMED,
                ArrayList::new));
        boolean groupNullKeys = consumerConfig.getBoolean(
            "group-null-keys",
            false);
        boolean queueIdHeader =
            consumerConfig.getBoolean("queue-id-header", true);
        int maxRetryCount =
            consumerConfig.getInt("max-retry-count", -1);
        int rateLimitDelay =
            consumerConfig.getIntegerDuration("rate-limit-delay", 20000);
        List<String> deduplicationKeyFields = consumerConfig.get(
            "deduplication-key-fields",
            Collections.emptyList(),
            new CollectionParser<>(
                NonEmptyValidator.TRIMMED,
                ArrayList::new));
        Set<String> ignoreBackendPositionSet =
            consumerConfig.get(
                "ignore-backend-position",
                Collections.emptySet(),
                new CollectionParser<>(
                    NonEmptyValidator.TRIMMED,
                    HashSet::new));
        BackendCommitPolicy commitPolicy =
            consumerConfig.get(
                "backend-commit-policy",
                BackendCommitPolicy.FAILED,
                new EnumParser<>(BackendCommitPolicy.class));
        boolean ignoreBackendPosition;
        if (Collections.singleton("true")
            .equals(ignoreBackendPositionSet))
        {
            ignoreBackendPosition = true;
        } else if (Collections.singleton("false")
            .equals(ignoreBackendPositionSet))
        {
            ignoreBackendPosition = false;
        } else {
            ignoreBackendPosition =
                ignoreBackendPositionSet.contains(service);
        }
        if (consumerType.equalsIgnoreCase("async")) {
            return new AsyncConsumer(
                host,
                queueIdHost,
                service,
                workers,
                server,
                targetConfig,
                primaryKeyFields,
                groupNullKeys,
                queueIdHeader,
                maxRetryCount,
                deduplicationKeyFields,
                ignoreBackendPosition,
                commitPolicy,
                watchdogDelay,
                zeroTolerance,
                rateLimitDelay);
        } else if (consumerType.equalsIgnoreCase("kamaji-async")) {
            return new KamajiAsyncConsumer(
                host,
                queueIdHost,
                service,
                workers,
                server,
                targetConfig,
                primaryKeyFields,
                groupNullKeys,
                queueIdHeader,
                maxRetryCount,
                deduplicationKeyFields,
                ignoreBackendPosition,
                commitPolicy,
                watchdogDelay,
                zeroTolerance,
                rateLimitDelay);
        } else if (consumerType.equalsIgnoreCase("tvm2-host")) {
            return new Tvm2HttpConsumer(
                host,
                queueIdHost,
                service,
                workers,
                server,
                targetConfig,
                primaryKeyFields,
                groupNullKeys,
                queueIdHeader,
                maxRetryCount,
                deduplicationKeyFields,
                ignoreBackendPosition,
                watchdogDelay,
                zeroTolerance,
                Tvm2HttpConsumer.parseHostClientIdMap(consumerConfig, targetConfig));
        } else {
            return new LuceneConsumer(
                host,
                queueIdHost,
                service,
                workers,
                server,
                targetConfig,
                primaryKeyFields,
                groupNullKeys,
                queueIdHeader,
                maxRetryCount,
                deduplicationKeyFields,
                ignoreBackendPosition,
                watchdogDelay,
                zeroTolerance);
        }
    }
}

