package ru.yandex.dispatcher.consumer;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.entity.StringEntity;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.zookeeper.KeeperException;

import ru.yandex.client.tvm2.ImmutableTvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;
import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.Pattern;
import ru.yandex.dispatcher.consumer.config.ConsumerConfigBuilder;
import ru.yandex.dispatcher.consumer.config.ImmutableConsumerConfig;
import ru.yandex.dispatcher.consumer.shard.Shard;
import ru.yandex.dispatcher.producer.SearchMap;
import ru.yandex.http.config.DnsConfigBuilder;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.proxy.HttpProxyTvm2HeaderSupplier;
import ru.yandex.http.proxy.MultiClientTvm2TicketSupplier;
import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.server.BaseServerConfigBuilder;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.logger.LoggerConfigBuilder;
import ru.yandex.logger.LoggerFileConfigBuilder;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.stater.AlertThresholds;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.ImmutableGolovanAlertsConfig;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatersRegistrar;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.string.StringUtils;

public class ConsumerServer
    extends BaseHttpServer<ImmutableBaseServerConfig>
    implements StatersRegistrar
{
    private final ImmutableDnsConfig dnsConfig;

    private final PrefixedLogger outlog;
    private final SearchMap searchMap;
    private final Map<String, ZooKeeperConsumer> zooKeeperConsumers =
        new HashMap<>();
    private final Map<String, BackendConsumer> backendConsumers;
    private final MultiClientTvm2TicketSupplier tvm2ClientTicketGenerator;
    private final Map<String, Map<String, AsyncClientsCell>> asyncClients;

    private static class BootstrapThread extends Thread {
        private final Map<String,ZooKeeperConsumer> consumers;
        private final Logger logger;

        public BootstrapThread(final Map<String,ZooKeeperConsumer> consumers,
            final Logger logger)
        {
            super("BootstrapThread");
            setDaemon(true);
            this.consumers = consumers;
            this.logger = logger;
        }

        public void run() {
            try {
                for(ZooKeeperConsumer zc : consumers.values()) {
                    zc.start();
                }
            } catch (Exception e) {
                logger.log(Level.SEVERE, "Something weird happened in bootstrapper", e);
            }
        }
    }

    public ConsumerServer(final IniConfig config) throws Exception {
        super(new BaseServerConfigBuilder(config, "http_server").build());
        dnsConfig = new DnsConfigBuilder(config.section("dns")).build();

        IniConfig outlogConfig = config.sectionOrNull("outlog");
        if (outlogConfig == null) {
            outlog = null;
        } else {
            outlog =
                new LoggerConfigBuilder(
                    outlogConfig,
                    new LoggerConfigBuilder().add(
                        new LoggerFileConfigBuilder()
                            .logFormat(
                                "%{date}%{separator}%{thread}"
                                    + "%{separator}%{message}")))
                    .build()
                    .buildPrefixed(super.config.loggers().handlersManager());
            registerLoggerForLogrotate(outlog);
        }

        IniConfig searchMapConfig = config.section("searchmap");

        searchMap = new SearchMap(
            searchMapConfig.getInputFile("file",
                new File("/var/cache/yamail/searchmap.txt")).getPath(),
            logger().replacePrefix("SearchMap"));

        HostnameResolver hr = HostnameResolverFactory.createResolver(config);
        String consumerHostname = hr.resolve();

        if (logger().isLoggable(Level.INFO)) {
            logger().info("HostNameResolver.hostname: hostname="
                + consumerHostname);
        }

        if (serviceContextRenewalTask == null) {
            tvm2ClientTicketGenerator = null;
        } else {
            tvm2ClientTicketGenerator = new MultiClientTvm2TicketSupplier(
                logger.addPrefix("tvm2"),
                serviceContextRenewalTask);
        }

        ImmutableConsumerConfig consumerConfig =
            new ConsumerConfigBuilder(config.section("zoolooser")).build();
        Set<String> consumerTags = consumerConfig.consumerTags();

        AsyncClientGenerator asyncClientGenerator =
            new AsyncClientGenerator(
                tvm2ClientTicketGenerator,
                config,
                consumerConfig.workers());

        List<SearchMap.Interval> intervals =
            searchMap.getBackendIntervals(consumerHostname);
        if (intervals == null) {
            throw new ConfigException(
                "No searchmap records found for this backend");
        }

        backendConsumers = new HashMap<>();

        // Code below is about fixing issue, when all threads are blocked
        // in case of one queue unavailability, because of sync code in
        // Shard.getZk()
        // ok now we trying to dance along the fire, trying to reduce damage
        // in perfect reality we need all async and so on, but consumer
        // works per service, so we building map service -> zks
        // and creating reactor per zks
        Map<String, Map<String, AsyncClientsCell>> asyncClients
            = new LinkedHashMap<>();
        for (SearchMap.Interval interval : intervals) {
            if (!consumerTags.isEmpty()
                && !consumerTags.contains(interval.tag()))
            {
                continue;
            }
            String zk = interval.zk();
            String service = interval.service();
            Map<String, AsyncClientsCell> cells =
                asyncClients.computeIfAbsent(
                    service,
                    (s) -> new LinkedHashMap<>());
            if (!cells.containsKey(zk)) {
                AsyncClientsCell cell = asyncClientGenerator.create();
                cell.start();
                cells.put(zk, cell);
            }
        }

        this.asyncClients = Collections.unmodifiableMap(asyncClients);

        for (SearchMap.Interval interval : intervals) {
            if (!consumerTags.isEmpty()
                && !consumerTags.contains(interval.tag()))
            {
                if (logger().isLoggable(Level.INFO)) {
                    logger().info("Skipping searchmap record: " + interval
                        + ": tag mismatch");
                }
                continue;
            }
            final String targetHostname;
            if (interval.targetHost() == null) {
                targetHostname = consumerHostname;
            } else {
                targetHostname = interval.targetHost();
            }
            String backendAddress =
                interval.service() + '/' + targetHostname
                + ':' + interval.indexPort();
            BackendConsumer bc = backendConsumers.get(backendAddress);
            if (bc == null) {
                bc = new BackendConsumerFactory(
                    new HttpHost(targetHostname, interval.indexPort()),
                    new HttpHost(targetHostname, interval.queueIdPort()),
                    interval.service(),
                    this)
                    .createConsumer(config, interval.targetTimeout());
                bc.start();
                backendConsumers.put(backendAddress, bc);
                if (logger().isLoggable(Level.INFO)) {
                    logger().info(
                        "Created BackendConsumer for service: "
                            + backendAddress);
                }
            }
        }

        for (SearchMap.Interval interval : intervals) {
            if (!consumerTags.isEmpty()
                && !consumerTags.contains(interval.tag()))
            {
                continue;
            }
            String zk = interval.zk();

            ZooKeeperConsumer zc = zooKeeperConsumers.get(zk);
            if (zc == null) {
                AsyncClientsCell clients =
                    asyncClients.get(interval.service()).get(zk);
                zc = new ZooKeeperConsumer(
                        zk,
                        consumerHostname,
                        searchMap,
                        clients.regularClient,
                        clients.nokeepAliveClient,
                        logger(),
                        consumerConfig);
                zc.init(backendConsumers, consumerTags);
                zooKeeperConsumers.put(zk, zc);
            }
        }
        config.checkUnusedKeys();

        register(
            new Pattern<>("/reset_shard", false),
            new ResetShardHandler(),
            RequestHandlerMapper.GET);

        register(
                new Pattern<>("/reset_lucene_shard", false),
                new ResetLuceneShardHandler(),
                RequestHandlerMapper.GET);
        registerStater(new HaltedShardsStater());
    }

    public SharedConnectingIOReactor reactor(
        final String service,
        final String zk)
    {
        if (zk == null) {
            return asyncClients.get(service).values().iterator().next().reactor;
        }

        return asyncClients.get(service).get(zk).reactor;
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        final List<String> haltedShards = haltedShards();
        status.put("has-halted-shards", haltedShards.size() > 0);
        status.put("halted-shards-count", haltedShards.size());
        status.put("halted-shards", haltedShards);

        for (Map.Entry<String, Map<String, AsyncClientsCell>> zkClientMap
            : asyncClients.entrySet())
        {
            for (Map.Entry<String, AsyncClientsCell> zkClient
                : zkClientMap.getValue().entrySet())
            {
                Map<String, Object> map = new LinkedHashMap<>();
                map.put(
                    "keep-alive-client",
                    zkClient.getValue().regularClient.status(verbose));
                map.put(
                    "no-keep-alive-client",
                    zkClient.getValue().nokeepAliveClient.status(verbose));
                status.put(StringUtils.concat(
                    zkClient.getKey(),
                    '_',
                    zkClient.getKey()), map);
            }
        }

        Map<String, Object> consumers = new HashMap<>();
        for (Map.Entry<String, BackendConsumer> entry
            : backendConsumers.entrySet())
        {
            consumers.put(entry.getKey(), entry.getValue().status(verbose));
        }
        status.put("consumers", consumers);
        return status;
    }

    private List<String> haltedShards() {
        List<String> haltedShards = new ArrayList<>();
        for (ZooKeeperConsumer zc: zooKeeperConsumers.values()) {
            zc.fillHaltedShards(haltedShards);
        }
        return haltedShards;
    }

    @Override
    public void start() throws IOException {
        super.start();
        BootstrapThread bt = new BootstrapThread(zooKeeperConsumers, logger());
        bt.start();

        if (tvm2ClientTicketGenerator != null) {
            tvm2ClientTicketGenerator.start();
        }
        System.err.println("ConsumerServer.Initialized");
    }

    @Override
    public void close() throws IOException {
        super.close();
    }

    public PrefixedLogger outlog() {
        return outlog;
    }

    public MultiClientTvm2TicketSupplier tvm2TicketSupplier() {
        return tvm2ClientTicketGenerator;
    }

    private class ResetShardHandler implements HttpRequestHandler {
        private StringEntity describeException(Exception e)
            throws UnsupportedEncodingException
        {
            StringBuilderWriter sbw = new StringBuilderWriter();
            e.printStackTrace(sbw);
            return new StringEntity(sbw.toString());
        }

        @Override
        public void handle(HttpRequest request, HttpResponse response,
            HttpContext context) throws HttpException, IOException
        {
            Logger logger = (Logger)context.getAttribute( BaseHttpServer.LOGGER );
            try {
                handle(
                    new CgiParams(request),
                    response, logger);
            } catch (InterruptedException e) {
                response.setStatusCode(HttpStatus.SC_GATEWAY_TIMEOUT);
                response.setEntity(describeException(e));
            } catch (KeeperException e) {
                response.setStatusCode(HttpStatus.SC_SERVICE_UNAVAILABLE);
                response.setEntity(describeException(e));
            } catch (ParseException e) {
                response.setStatusCode(HttpStatus.SC_BAD_REQUEST);
                response.setEntity(describeException(e));
            }
        }

        private void handle(CgiParams cgiParams, HttpResponse response, Logger logger)
            throws HttpException, IOException, InterruptedException,
                KeeperException, ParseException
        {
            boolean allowCached = cgiParams.getBoolean("allow_cached", false);
            String service = cgiParams.getString("service");
            int shardNumber = cgiParams.getInt("shard");
            String backend = cgiParams.getString("backend", null);
            boolean found = false;
            for (ZooKeeperConsumer zkc : zooKeeperConsumers.values()) {
                if (zkc.resetShard(service, shardNumber, backend)) {
                    found = true;
                }
            }
            if (!found) {
                response.setStatusCode(HttpStatus.SC_NOT_FOUND);
                response.setEntity(new StringEntity("This consumer has no " +
                    "shard " + shardNumber + " for service " + service));
            }
//	    response.setEntity( new StringEntity(sb.toString()) );
        }

    }

    private class ResetLuceneShardHandler implements HttpRequestHandler {
        private StringEntity describeException(Exception e)
                throws UnsupportedEncodingException
        {
            StringBuilderWriter sbw = new StringBuilderWriter();
            e.printStackTrace(sbw);
            return new StringEntity(sbw.toString());
        }

        @Override
        public void handle(HttpRequest request, HttpResponse response,
                           HttpContext context) throws HttpException, IOException
        {
            Logger logger = (Logger)context.getAttribute( BaseHttpServer.LOGGER );
            try {
                handle(
                        new CgiParams(request),
                        response, logger);
            } catch (InterruptedException e) {
                response.setStatusCode(HttpStatus.SC_GATEWAY_TIMEOUT);
                response.setEntity(describeException(e));
            } catch (KeeperException e) {
                response.setStatusCode(HttpStatus.SC_SERVICE_UNAVAILABLE);
                response.setEntity(describeException(e));
            } catch (ParseException e) {
                response.setStatusCode(HttpStatus.SC_BAD_REQUEST);
                response.setEntity(describeException(e));
            }
        }

        private void handle(CgiParams cgiParams, HttpResponse response, Logger logger)
                throws HttpException, IOException, InterruptedException,
                KeeperException, ParseException
        {
            int luceneShardsCount = cgiParams.getInt("lucene_shards_count");
            int shardNumber = cgiParams.getInt("shard");
            boolean found = false;
            if ( luceneShardsCount > 0 ) {
                for (ZooKeeperConsumer zkc : zooKeeperConsumers.values()) {
                    if (zkc.resetShardBatch(shardNumber, luceneShardsCount)) {
                        found = true;
                    }
                }
            }

            if (!found) {
                response.setStatusCode(HttpStatus.SC_NOT_FOUND);
                response.setEntity(new StringEntity("This consumer has no " +
                        "lucene shard " + shardNumber));
            }
        }
    }

    private class HaltedShardsStater implements Stater {
        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            List<String> haltedShards = haltedShards();
            statsConsumer.stat(
                "has-halted-shards_ammx",
                Math.min(1, haltedShards.size()));
            statsConsumer.stat(
                "halted-shards-count_ammx",
                haltedShards.size());
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);

            GolovanChart haltedShards = new GolovanChart(
                "halted-shards",
                " halted shards",
                false,
                false,
                0d);
            haltedShards.addSplitSignal(
                config,
                statsPrefix + "halted-shards-count_ammx",
                null,
                false,
                false);
            group.addChart(haltedShards);
            panel.addCharts("halted-shards", null, group);
        }

        @Override
        public void addToAlertsConfig(
            final IniConfig alertsConfig,
            final ImmutableGolovanPanelConfig panelConfig,
            final String statsPrefix)
            throws BadRequestException
        {
            ImmutableGolovanAlertsConfig alerts = panelConfig.alerts();
            alerts.createAlert(
                alertsConfig,
                alerts.module() + "-halted-shards",
                statsPrefix + "halted-shards-count_ammx",
                new AlertThresholds(0.5d, null, null));
        }
    }

    public static void main(String[] args)
        throws ConfigException, IOException
    {
        if (args.length < 2) {
            System.out.println("###failed_to_start###");
            System.err.println("Usage: "
                + "ru.yandex.dispatcher.consumer.ConsumerServer "
                + "-config <file>");
            return;
        }

        main(new ConsumerServerFactory(), args[1]);
    }

    private class AsyncClientGenerator {
        private final MultiClientTvm2TicketSupplier tvm2ClientTicketGenerator;
        private final ImmutableHttpTargetConfig clientConfig;
        private final ImmutableHttpTargetConfig noKeepAliveConfig;
        private final ImmutableBaseServerConfig serverConfig;

        public AsyncClientGenerator(
            final MultiClientTvm2TicketSupplier tvm2ClientTicketGenerator,
            final IniConfig config,
            final int zooWorkers)
            throws ConfigException
        {
            this.tvm2ClientTicketGenerator = tvm2ClientTicketGenerator;
            final IniConfig httpSection = config.section("zoolooser-http");
            final int asyncClientMaxPerRoute =
                config.getInt("max_per_route", zooWorkers);
            BaseServerConfigBuilder threadsConfig =
                new BaseServerConfigBuilder(ConsumerServer.this.config);

            final int reactorThreads =
                config.getInt(
                    "reactor_workers",
                    threadsConfig.workers());
            final int reactorThreadsPercent =
                config.get(
                    "reactor_workers_percent",
                    100,
                    PositiveIntegerValidator.INSTANCE);

            int workers =
                Math.max(1, (reactorThreads * reactorThreadsPercent) / 100);
            logger().info("SharedReactor workers: " + workers);

            threadsConfig.workers(workers);
            serverConfig =
                new ImmutableBaseServerConfig(
                    threadsConfig,
                    ConsumerServer.this.config.dnsConfig(),
                    ConsumerServer.this.config.tvm2ServiceConfig(),
                    ConsumerServer.this.config.loggers(),
                    ConsumerServer.this.config.staters(),
                    ConsumerServer.this.config.auths(),
                    ConsumerServer.this.config.limiters(),
                    ConsumerServer.this.config.golovanPanel(),
                    ConsumerServer.this.config.autoRegisterRequestStater());

            final int connectTimeout = config.getInt("connect_timeout", 3000);
            final int socketTimeout = config.getInt("so_timeout", 30000);
            final int soLinger = config.getInt("so_linger", 3);


            ImmutableHttpTargetConfig zooHttpTargetDefaults =
                new HttpTargetConfigBuilder()
                    .connections(asyncClientMaxPerRoute)
                    .connectTimeout(connectTimeout)
                    .timeout(socketTimeout)
                    .poolTimeout(connectTimeout)
                    .build();

            clientConfig =
                new HttpTargetConfigBuilder(httpSection, zooHttpTargetDefaults)
                    .briefHeaders(true)
                    .build();
            noKeepAliveConfig =
                new HttpTargetConfigBuilder(httpSection, zooHttpTargetDefaults)
                    .briefHeaders(true)
                    .keepAlive(false)
                    .build();
        }

        public AsyncClientsCell create() throws IOException {
            return new AsyncClientsCell(
                clientConfig,
                noKeepAliveConfig,
                serverConfig,
                tvm2ClientTicketGenerator);
        }
    }

    private final class AsyncClientsCell {
        private final SharedConnectingIOReactor reactor;
        private final AsyncClient regularClient;
        private final AsyncClient nokeepAliveClient;

        public AsyncClientsCell(
            final ImmutableHttpTargetConfig clientConfig,
            final ImmutableHttpTargetConfig noKeepAliveConfig,
            final ImmutableBaseServerConfig serverConfig,
            final MultiClientTvm2TicketSupplier tvm2ClientTicketGenerator)
            throws IOException
        {
            this.reactor =
                new SharedConnectingIOReactor(serverConfig, dnsConfig);
            this.regularClient =
                new AsyncClient(reactor, clientConfig)
                    .addHeader(
                        new HttpProxyTvm2HeaderSupplier(
                            tvm2ClientTicketGenerator,
                            clientConfig.tvm2Headers()));
            this.nokeepAliveClient =
                new AsyncClient(reactor, noKeepAliveConfig)
                    .addHeader(
                        new HttpProxyTvm2HeaderSupplier(
                            tvm2ClientTicketGenerator,
                            noKeepAliveConfig.tvm2Headers()));
        }

        public void start() {
            this.reactor.start();
            this.regularClient.start();
            this.nokeepAliveClient.start();
        }
    }
}
