package ru.yandex.dispatcher.consumer;

import java.io.IOException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpHost;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.collection.IntPair;
import ru.yandex.dispatcher.common.DelayShardMessage;
import ru.yandex.dispatcher.common.HangShardMessage;
import ru.yandex.dispatcher.common.HttpMessage;
import ru.yandex.dispatcher.common.SerializeUtils;
import ru.yandex.dispatcher.consumer.shard.Shard;
import ru.yandex.function.BasicConsumer;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.ServerErrorStatusPredicate;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryParameter;
import ru.yandex.parser.uri.UriParser;
import ru.yandex.util.timesource.TimeSource;

public class LuceneConsumer extends AbstractHttpBackendConsumer {
    private static final long WRITE_ERROR_TIME_THRESHOLD = 5 * 60 * 1000;

    private final boolean queueIdHeader;
    private final Function<String, Object> primaryKeyFactory;
    private final boolean groupNullKeys;
    private final int maxRetryCount;
    private final Function<String, Object> deduplicationKeyFactory;
    private final boolean ignoreBackendPosition;

    public LuceneConsumer(
        final HttpHost host,
        final HttpHost queueIdHost,
        final String service,
        final int workers,
        final ConsumerServer server,
        final ImmutableHttpTargetConfig targetConfig,
        final List<String> primaryKeyFields,
        final boolean groupNullKeys,
        final boolean queueIdHeader,
        final int maxRetryCount,
        final List<String> deduplicationKeyFields,
        final boolean ignoreBackendPosition,
        final long watchdogDelay,
        final boolean zeroTolerance)
    {
        super(
            host,
            queueIdHost,
            service,
            workers,
            server,
            targetConfig,
            server.logger().replacePrefix("LuceneConsumer"),
            watchdogDelay,
            zeroTolerance);
        this.queueIdHeader = queueIdHeader;
        this.maxRetryCount = maxRetryCount;
        Supplier<?> groupDefaultSupplier;
        if (groupNullKeys) {
            groupDefaultSupplier = Object::new;
        } else {
            groupDefaultSupplier = new BasicConsumer<>();
        }
        primaryKeyFactory =
            createKeyFactory(primaryKeyFields, groupDefaultSupplier);
        this.groupNullKeys = groupNullKeys;
        deduplicationKeyFactory =
            createKeyFactory(deduplicationKeyFields, Object::new);
        this.ignoreBackendPosition = ignoreBackendPosition;
    }

    private static Function<String, Object> createKeyFactory(
        final List<String> fields,
        final Supplier<?> defaultValueSupplier)
    {
        Function<String, Object> keyFactory;
        switch (fields.size()) {
            case 0:
                keyFactory = x -> defaultValueSupplier.get();
                break;
            case 1:
                keyFactory =
                    new BasicKeyFactory(fields.get(0), defaultValueSupplier);
                break;
            case 2:
                keyFactory = new DoubleKeyFactory(
                    fields.get(0),
                    fields.get(1),
                    defaultValueSupplier);
                break;
            default:
                keyFactory = new ListKeyFactory(fields, defaultValueSupplier);
                break;
        }
        return keyFactory;
    }

    protected Runnable createNodesTask(
        final List<Node> nodes,
        final Shard shard)
    {
        return new ProcessNodesTask(nodes, shard);
    }

    @Override
    public void dispatchNodes(final List<Node> nodes, final Shard shard) {
        workExecutor.execute(createNodesTask(nodes, shard));
    }

    public ShardsPositions getPositions(final String service) {
        if (ignoreBackendPosition) {
            return null;
        } else {
            return new LuceneShardsPosition(
                service,
                queueIdHost(),
                httpClient(),
                logger);
        }
    }

    protected class ProcessNodesTask implements Runnable {
        private final List<Node> nodes;
        private final Shard shard;
        private final int size;
        protected final Logger logger;

        public ProcessNodesTask(final List<Node> nodes, final Shard shard) {
            this.nodes = nodes;
            this.shard = shard;
            size = nodes.size();
            logger = shard.logger();
        }

        protected List<List<Node>> groups() throws IOException {
            Map<Object, Node> deduplicated =
                new LinkedHashMap<>(size << 1, 0.5f, true);
            for (Node node: nodes) {
                if (node.msg == null) {
                    if (logger.isLoggable(Level.FINE)) {
                        logger.fine("Loading node " + node.path);
                    }
                    node.msg =
                        SerializeUtils.deserializeHttpMessage(node.data);
                }
                String uri = node.msg.uri();
                Object deduplicationKey;
                if (uri == null) {
                    deduplicationKey = new Object();
                } else {
                    deduplicationKey = deduplicationKeyFactory.apply(uri);
                }
                deduplicated.put(deduplicationKey, node);
            }
            Node lastNode = null;
            Iterator<Node> iter = deduplicated.values().iterator();
            while (true) {
                if (iter.hasNext()) {
                    lastNode = iter.next();
                } else {
                    iter.remove();
                    break;
                }
            }
            Set<Object> primaryKeys = new HashSet<>(size << 1);
            List<List<Node>> groups = new ArrayList<>(size);
            List<Node> currentGroup = new ArrayList<>(1);
            for (Node node: deduplicated.values()) {
                String uri = node.msg.uri();
                if (uri == null) {
                    if (node.msg instanceof HangShardMessage) {
                        if (logger.isLoggable(Level.INFO)) {
                            logger.info("Hang requested");
                        }
                        // fail fast
                        shard.hang(node);
                        return null;
                    }
                    if (node.msg instanceof DelayShardMessage) {
                        if (!currentGroup.isEmpty()) {
                            groups.add(currentGroup);
                            currentGroup = new ArrayList<>(1);
                            primaryKeys.clear();
                        }
                        groups.add(Collections.singletonList(node));
                    }
                } else {
                    Object primaryKey = primaryKeyFactory.apply(uri);
                    if (primaryKeys.add(primaryKey)) {
                        currentGroup.add(node);
                    } else {
                        groups.add(currentGroup);
                        currentGroup = new ArrayList<>(1);
                        currentGroup.add(node);
                        primaryKeys.clear();
                        primaryKeys.add(primaryKey);
                    }
                }
            }
            if (!currentGroup.isEmpty()) {
                groups.add(currentGroup);
            }
            if (lastNode != null) {
                groups.add(Collections.singletonList(lastNode));
            }
            return groups;
        }

        private List<Future<IntPair<Void>>> spawnRequests(
            final List<Node> nodes)
        {
            int size = nodes.size();
            List<Future<IntPair<Void>>> requests = new ArrayList<>(size);
            for (int i = 0; i < size; ++i) {
                Node node = nodes.get(i);
                long queueId;
                if (i == 0 && queueIdHeader) {
                    queueId = node.seq;
                } else {
                    queueId = Integer.MIN_VALUE;
                }
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine(
                        "Processing node: " + node.path
                        + ' ' + '(' + (i + 1) + " of " + size
                        + "), type: " + node.msg
                        + ", path: " + node.path
                        + ", delayStatus: " + !node.notifyStatus
                        + ", queueTime="
                        + (node.msg.readTime() - node.msg.createTime()));
                }
                requests.add(
                    runMessage(
                        node.msg,
                        queueId,
                        shard.shardNo,
                        shard.service(),
                        "zoo-queue-id=" + node.seq));
            }
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("All nodes group requests spawned");
            }
            return requests;
        }

        private boolean process(final List<Node> nodes)
            throws InterruptedException
        {
            if (shouldStop()) {
                return false;
            }
            if (nodes.size() == 1) {
                HttpMessage msg = nodes.get(0).msg;
                if (msg instanceof DelayShardMessage) {
                    Thread.sleep(((DelayShardMessage) msg).delay());
                    return true;
                }
            }
            List<Future<IntPair<Void>>> requests = spawnRequests(nodes);
            boolean commitLucenePosition = nodes.size() > 1;

            for (int i = 0; i < nodes.size(); ++i) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("Checking status for node #" + (i + 1));
                }
                Node node = nodes.get(i);
                String processingStatus = "200";
                Future<IntPair<Void>> request = requests.get(i);
                long retryDelay = 1;
                while (request != null) {
                    try {
                        processingStatus = "200";
                        request.get(
                            maxNodeTimeout() << 1,
                            TimeUnit.MILLISECONDS);
                        shard.writeNodeStatus(node);
                        if (commitLucenePosition) {
                            updatePosition(shard, node.seq);
                        }

                        request = null;
                    } catch (ExecutionException|TimeoutException e) {
                        Throwable cause = e.getCause();
                        if (cause == null) {
                            cause = e;
                        }
                        if (logger.isLoggable(Level.WARNING)) {
                            logger.log(
                                Level.WARNING,
                                "Processing node " + node.path + " failed",
                                cause);
                        }
                        if (cause instanceof InterruptedException) {
                            throw (InterruptedException) cause;
                        }
                        int status = 0;
                        if (cause instanceof BadResponseException) {
                            status = ((BadResponseException) cause).statusCode();
                            processingStatus = Integer.toString(status);
                            if (ServerErrorStatusPredicate.INSTANCE.test(status)
                                || status == YandexHttpStatus.SC_UNAUTHORIZED
                                || status == YandexHttpStatus.SC_FORBIDDEN
                                || status == YandexHttpStatus.SC_BUSY)
                            {
                                status = 0;
                            } else if (status == YandexHttpStatus.SC_TOO_MANY_REQUESTS) {
                                status = 0;
                                if (logger.isLoggable(Level.INFO)) {
                                    logger.info("Hit rate limit, sleeping");
                                }
                                Thread.sleep(10000L);
                            }
                        } else {
                            processingStatus = cause.getClass().getName();
                        }
                        if (zeroTolerance
                            || (status == 0
                                && (maxRetryCount < 0
                                    || node.retryCount++ < maxRetryCount
                                    || cause instanceof TimeoutException)))
                        {
                            if (shouldStop()) {
                                if (logger.isLoggable(Level.INFO)) {
                                    logger.info("Aborting processing at pos "
                                        + i);
                                }
                                return false;
                            }
                            Thread.sleep(retryDelay);
                            retryDelay *= 2;
                            if (retryDelay > 1000) {
                                retryDelay = 1000;
                            }
                            if (logger.isLoggable(Level.INFO)) {
                                logger.info("Retrying " + node.path
                                    + " try#: " + node.retryCount);
                            }
                            node.msg.deleteHeader(YandexHeaders.RETRY_COUNT);
                            node.msg.setHeader(
                                YandexHeaders.RETRY_COUNT,
                                Integer.toString(node.retryCount));
                            request = runMessage(
                                node.msg,
                                node.seq,
                                shard.shardNo,
                                shard.service(),
                                "zoo-queue-id=" + node.seq);
                        } else {
                            if (status == 0) {
                                if (logger.isLoggable(Level.SEVERE)) {
                                    logger.severe(
                                        "node max retry count exhaused: "
                                        + "maxRetryCount=" + maxRetryCount
                                        + ", node.retryCount="
                                        + node.retryCount);
                                }
                            }
                            request = null;
                            long msgQueueTime = TimeSource.INSTANCE.currentTimeMillis()
                                - node.msg.createTime();
                            if (logger.isLoggable(Level.SEVERE)) {
                                logger.severe(
                                    "msgTime=" + msgQueueTime
                                    + ", threshold="
                                    + WRITE_ERROR_TIME_THRESHOLD);
                            }
                            if (msgQueueTime < WRITE_ERROR_TIME_THRESHOLD) {
                                // do not write msg error if this is an old
                                // message
                                shard.writeNodeError(
                                    node,
                                    cause.getMessage(),
                                    status);
                                shard.writeNodeStatus(node);
                            }
                        }
                    }
                }
                if (outlog != null) {
                    outlog.info(
                        shard.shardNo + outlog.separator()
                        + node.seq + outlog.separator()
                        + (node.msg.readTime() - node.msg.createTime())
                        + outlog.separator()
                        + (TimeSource.INSTANCE.currentTimeMillis() - node.msg.readTime())
                        + outlog.separator() + node.msg.uri()
                        + outlog.separator() + processingStatus);
                }
            }
            return true;
        }

        private boolean shouldStop() {
            final boolean result;
            if (!shard.checkLock()) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine(
                        "Processing node stopped: lock consumer lock expired");
                }
                result = true;
            } else if (shard.shouldReset()) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine(
                        "Processing node stopped: shard was resetted");
                }
                shard.finishReset();
                result = true;
            } else if (shard.consumerExpired(nodes)) {
                result = true;
            } else {
                result = false;
            }
            return result;
        }

        @Override
        public void run() {
            try {
                List<List<Node>> groups = groups();
                if (groups == null) {
                    return;
                }
                int count = groups.size();
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine(size + " nodes splitted in " + count
                        + " groups");
                }
                for (int i = 0; i < count; ++i) {
                    if (logger.isLoggable(Level.FINE)) {
                        logger.fine("Processing group #" + i);
                    }
                    if (!process(groups.get(i))) {
                        return;
                    }
                }
                shard.nodeFinished(nodes, nodes.get(nodes.size() - 1));
            } catch (Throwable t) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.log(Level.SEVERE, "Processing failed", t);
                }
            }
        }
    }

    private static class BasicKeyFactory implements Function<String, Object> {
        private final String name;
        private final Supplier<?> defaultValueSupplier;

        public BasicKeyFactory(
            final String name,
            final Supplier<?> defaultValueSupplier)
        {
            this.name = name;
            this.defaultValueSupplier = defaultValueSupplier;
        }

        @Override
        public Object apply(final String request) {
            for (QueryParameter param
                    : new UriParser(request).queryParser(null))
            {
                if (name.equals(param.name())) {
                    return param.value().toString();
                }
            }
            return defaultValueSupplier.get();
        }
    }

    private static class DoubleKeyFactory implements Function<String, Object> {
        private final String first;
        private final String second;
        private final Supplier<?> defaultValueSupplier;

        public DoubleKeyFactory(
            final String first,
            final String second,
            final Supplier<?> defaultValueSupplier)
        {
            this.first = first;
            this.second = second;
            this.defaultValueSupplier = defaultValueSupplier;
        }

        @Override
        public Object apply(final String request) {
            Object first = null;
            Object second = null;
            for (QueryParameter param
                    : new UriParser(request).queryParser(null))
            {
                String name = param.name();
                if (first == null && this.first.equals(name)) {
                    first = param.value().toString();
                    if (second != null) {
                        break;
                    }
                } else if (second == null && this.second.equals(name)) {
                    second = param.value().toString();
                    if (first != null) {
                        break;
                    }
                }
            }
            if (first == null) {
                first = defaultValueSupplier.get();
            }
            if (second == null) {
                second = defaultValueSupplier.get();
            }
            return new DoubleKey(first, second);
        }
    }

    private static class DoubleKey {
        private final Object first;
        private final Object second;

        public DoubleKey(final Object first, final Object second) {
            this.first = first;
            this.second = second;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append('[');
            sb.append(first);
            sb.append(',');
            sb.append(second);
            sb.append(']');
            return new String(sb);
        }

        @Override
        public int hashCode() {
            return Objects.hashCode(first) ^ Objects.hashCode(second);
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof DoubleKey) {
                DoubleKey other = (DoubleKey) o;
                return Objects.equals(first, other.first)
                    && Objects.equals(second, other.second);
            }
            return false;
        }
    }

    private static class ListKeyFactory implements Function<String, Object> {
        private final List<String> keyFields;
        private final Supplier<?> defaultValueSupplier;

        public ListKeyFactory(
            final List<String> keyFields,
            final Supplier<?> defaultValueSupplier)
        {
            this.keyFields = keyFields;
            this.defaultValueSupplier = defaultValueSupplier;
        }

        @Override
        public Object apply(final String request) {
            List<Object> key = new ArrayList<>(keyFields.size());
            CgiParams params =
                new CgiParams(new UriParser(request).queryParser(null));
            for (String field: keyFields) {
                Object value = params.getString(field, null);
                if (value == null) {
                    value = defaultValueSupplier.get();
                }
                key.add(value);
            }
            return key;
        }
    }

    private static class LuceneShardsPosition implements ShardsPositions {
        private final String service;
        private final HttpHost host;
        private final AsyncClient httpClient;
        private final String url;
        private final Logger logger;

        public LuceneShardsPosition(
            final String service,
            final HttpHost host,
            final AsyncClient httpClient,
            final Logger logger)
        {
            this.service = service;
            this.host = host;
            this.httpClient = httpClient;
            this.logger = logger;
            this.url = "/getQueueId?service=" + service + "&shard=";
        }

        @Override
        public long getPosition(int shard) throws IOException {
            final String request = url + Integer.toString(shard);
            final BasicAsyncRequestProducerGenerator get =
                new BasicAsyncRequestProducerGenerator(request);
            try {
                final String response =
                    httpClient.execute(
                        host,
                        get,
                        AsyncStringConsumerFactory.OK,
                        EmptyFutureCallback.INSTANCE).get();
                return Long.parseLong(response.trim());
            } catch (Exception e) {
                throw new IOException("Can't retreive Queue position "
                    + "from <" + host + "> for shard: " + shard, e);
            }
        }

        @Override
        public void updatePosition(
            final int shard,
            final long position,
            final FutureCallback<Object> callback)
        {
            final String uri = "/delete?updatePosition&prefix="
                + Integer.toString(shard)
                + "&shard=" + Integer.toString(shard)
                + "&service=" + service
                + "&position=" + position;
            final String content = "{\"prefix\":" + shard
                + ",\"docs\":[]}";
            if (logger.isLoggable(Level.INFO)) {
                logger.info("LuceneConsumer: updatePosition for Shard<" + shard
                    + ">: " + position
                    + ": uri: " + uri + ", post: " + content);
            }
            final BasicAsyncRequestProducerGenerator post =
                new BasicAsyncRequestProducerGenerator(uri, content);
            post.addHeader(
                YandexHeaders.ZOO_QUEUE_ID,
                Long.toString(position));
            post.addHeader(
                YandexHeaders.ZOO_SHARD_ID,
                Integer.toString(shard));
            post.addHeader(YandexHeaders.ZOO_QUEUE, service);
            httpClient.execute(
                host,
                post,
                AsyncStringConsumerFactory.OK,
                callback);
        }
    }
}
