package ru.yandex.dispatcher.consumer;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.logging.Level;

import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;
import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.OpResult;
import org.apache.zookeeper.Transaction;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;

import ru.yandex.dispatcher.common.ErrorMessage;
import ru.yandex.dispatcher.common.SerializeUtils;
import ru.yandex.dispatcher.producer.SearchMap;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.util.timesource.TimeSource;

public class StatusQueue extends Thread implements Watcher
{
private static final int SUFFLE_DELAY = 1000;
private ZooKeeperDSN dsn;
private BlockingQueue<Node> queue;
private BlockingQueue<Node> failedQueue;
private ZooKeeper zk;
private int timeout;
//private NodeChecker checker;
private String backendName;
private final String service;
private long[] statuses = new long[SearchMap.SHARDS_COUNT];
private HashMap<Integer,String> dirty = new HashMap<Integer,String>();
private LinkedHashMap<Integer,DelayedNodeStatus> delayedStatuses = new LinkedHashMap<Integer,DelayedNodeStatus>(SearchMap.SHARDS_COUNT, (float)0.75, true);
private ConcurrentHashMap<Integer, String> createNodes = new ConcurrentHashMap<Integer, String>();
private Object shardsLock = new Object();
private final Timer delayer;
private final long MAX_NOTIFY_DELAY = 50;
private final PrefixedLogger logger;
private final CreateNodeCallback createNodeCallback = new CreateNodeCallback();
private final WriteStatusesCallback writeStatusesCallback = new WriteStatusesCallback();
private final long groupingTime;
private final LinkedList<HttpHost> producersList;
private final AsyncClient httpClient;
private long lastShuffle = 0;

    private static class DelayedNodeStatus {
        private long seq;
        private long lastUpdate;
        private final String path;

        public DelayedNodeStatus(final String path, final long seq) {
            this.path = path;
            this.seq = seq;
            this.lastUpdate = TimeSource.INSTANCE.currentTimeMillis();
        }

        public void touch(final long seq) {
            lastUpdate = TimeSource.INSTANCE.currentTimeMillis();
            if (this.seq < seq) {
                this.seq = seq;
            }
        }

        public String path() {
            return path;
        }

        public long seq() {
            return seq;
        }

        public long lastUpdate() {
            return lastUpdate;
        }
    }

    private class WriteStatusesCallback
        implements org.apache.zookeeper.AsyncCallback.MultiCallback
    {
        @Override
        public void processResult(final int rc, final String path,
            final Object ctx, final List<OpResult> results)
        {
            LinkedHashMap<Integer,String> dirtyNodes =
                (LinkedHashMap<Integer,String>)ctx;

            if (results.size() != dirtyNodes.size()) {
                int resSize = results == null ? 0 : results.size();
                int dirtyNodesSize = dirtyNodes == null ? 0 : dirtyNodes.size();
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("Error in merged transaction: RC != 0: " + rc
                        + ", results.size<" + resSize + "> != dirty.size<"
                        + dirtyNodesSize +">");
                }
                synchronized (shardsLock) {
                    for (Map.Entry<Integer,String> entry :
                        dirtyNodes.entrySet())
                    {
                        int shard = entry.getKey();
                        dirty.put(shard, entry.getValue());
                    }
                }
                return;
            }

            if (rc != 0) {
                Iterator<Map.Entry<Integer,String>> dirtyIter =
                    dirtyNodes.entrySet().iterator();
                for (OpResult op : results) {
                    Map.Entry<Integer,String> dirtyEntry = dirtyIter.next();
                    if (op.getType() == ZooDefs.OpCode.error) {
                        OpResult.ErrorResult errorResult =
                            (OpResult.ErrorResult)op;
                        int error = errorResult.getErr();
                        if (error == KeeperException.Code.NONODE.intValue()) {
                            createNodes.put(dirtyEntry.getKey(),
                                dirtyEntry.getValue());
                            synchronized (shardsLock) {
                                dirty.put(dirtyEntry.getKey(),
                                    dirtyEntry.getValue());
                            }
                            if (logger.isLoggable(Level.WARNING)) {
                                if (logger.isLoggable(Level.WARNING)) {
                                    logger.warning("Node "
                                        + dirtyEntry.getValue()
                                        + " does not exists, will try to create"
                                        + " first");
                                }
                            }
                        } else if (error == KeeperException.Code.RUNTIMEINCONSISTENCY.intValue() ||
                            error == KeeperException.Code.OK.intValue())
                        {
                            if (logger.isLoggable(Level.FINE)) {
                                logger.fine("Remarking path "
                                    + dirtyEntry.getValue()
                                    + " as dirty after failed multiop");
                            }
                            synchronized (shardsLock) {
                                int shard = dirtyEntry.getKey();
                                dirty.put(shard, dirtyEntry.getValue());
                            }
                        } else {
                            if (logger.isLoggable(Level.SEVERE)) {
                                logger.severe("Unhandler error code in MultiOp "
                                + "for path " + dirtyEntry.getValue() + " : "
                                + error);
                            }
                        }
                    } else {
                        if (logger.isLoggable(Level.FINE)) {
                            logger.fine("Remarking path "
                                + dirtyEntry.getValue()
                                + " as dirty after failed multiop");
                        }
                        synchronized (shardsLock) {
                            int shard = dirtyEntry.getKey();
                            dirty.put(shard, dirtyEntry.getValue());
                        }
                    }
                }
                return;
            } else {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("StatusQueue.request success");
                }
            }
        }
    }

    private class CreateNodeCallback
        implements org.apache.zookeeper.AsyncCallback.StringCallback
    {
        @Override
        public void processResult(final int rc, final String path,
            final Object ctx, String name)
        {
            if (ctx == null) {
                return;
            }

            Map.Entry<Integer,String> entry = (Map.Entry<Integer,String>)ctx;
            Integer shard = entry.getKey();
            String nodePath = entry.getValue();

            if (rc != 0) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("Error in create node request for path "
                        + nodePath + " : RC != 0: " + rc
                        + ". Delaying node processing");
                }
                synchronized (shardsLock) {
                    DelayedNodeStatus status = delayedStatuses.get(shard);
                    if (status == null) {
                        delayedStatuses.put(shard,
                            new DelayedNodeStatus(nodePath,statuses[shard]));
                    }
                }
            } else {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("Successfully created node for path " + path);
                }
            }
        }
    }

    public StatusQueue(
        final ZooKeeperDSN dsn,
        final int timeout,
        final String backendName,
        final String service,
        final PrefixedLogger logger,
        final long groupingTimeMicros,
        final List<String> producersList,
        final AsyncClient httpClient)
    {
        super( "StatusQueue-" + backendName + '@' + service );
        this.dsn = dsn;
        this.timeout = timeout;
        this.backendName = backendName;
        this.service = service;
        this.logger = logger;
        this.groupingTime = groupingTimeMicros * 1000;
        this.httpClient = httpClient;
        queue = new ArrayBlockingQueue<Node>(10000);
        failedQueue = new LinkedBlockingQueue<Node>(10000);
        start();
//        checker = new NodeChecker();
//        checker.start();
        delayer = new Timer("DelayedStatusNotifyer");
        delayer.schedule( new DelayedStatusesTask(), 10L, 10L );
        if (logger.isLoggable(Level.INFO)) {
            logger.info("Started status queue for: " + backendName + "@"
                + service);
        }
        this.producersList = new LinkedList<>();
        for (String producer : producersList) {
            this.producersList.add(producerHost(producer));
        }
        Collections.shuffle(this.producersList);
    }

    public String consumerName() {
        return backendName;
    }

    private HttpHost producerHost(final String str) {
        final int sep = str.lastIndexOf(':');
        return new HttpHost(
            str.substring(0, sep),
            Integer.parseInt(str.substring(sep + 1)));
    }

    public void run()
    {
        long[] statusCopy = new long[SearchMap.SHARDS_COUNT];
        while (true) {
            try {
                long startTime = 0;// = System.nanoTime();
                boolean first = true;
                long nanoTime = startTime;
                int dataSize = 0;
                int grouped = 0;
                Map<Integer,String> dirtyCopy;
                while (true) {
                    synchronized (shardsLock) {
                        if (dirty.size() == 0) {
                            shardsLock.wait(1);
                        } else {
                            if (first) {
                                startTime = System.nanoTime();
                                first = false;
                                shardsLock.wait(1);
                            } else {
                                nanoTime = System.nanoTime();
                                if (nanoTime - startTime > groupingTime
                                    || nanoTime < startTime)
                                {
                                    //Linked is important here as we must
                                    //support proper iteration order in
                                    //the async callback later.
                                    dirtyCopy =
                                        new LinkedHashMap<Integer,String>(
                                            dirty);
                                    dirty.clear();
                                    for (Integer s: dirtyCopy.keySet()) {
                                        statusCopy[s] = statuses[s];
                                    }
                                    break;
                                } else {
                                    shardsLock.wait(1);
                                }
                            }
                        }
                    }
                }

                while (true) {
                    try {
                        ZooKeeper zk = getZk();

                        if (createNodes.size() > 0) {
                            Iterator<Map.Entry<Integer,String>> iter =
                                createNodes.entrySet().iterator();
                            while (iter.hasNext()) {
                                Map.Entry<Integer,String> entry = iter.next();
                                Integer shard = entry.getKey();
                                String path = entry.getValue();
                                String parent = path.substring(0,
                                    path.lastIndexOf('/'));
                                zk.create(
                                    parent,
                                    new byte[0],
                                    null,
                                    CreateMode.PERSISTENT,
                                    createNodeCallback,
                                    null);
                                zk.create(
                                    path,
                                    Long.toString(statusCopy[shard]).getBytes(),
                                    null,
                                    CreateMode.PERSISTENT,
                                    createNodeCallback,
                                    entry);
                                dirtyCopy.remove(shard);
                                iter.remove();
                            }
                        }
                        if (producersList.size() > 0) {
                            for (Map.Entry<Integer,String> entry:
                                dirtyCopy.entrySet())
                            {
                                int shard = entry.getKey();
                                updateStatus(
                                    shard,
                                    statusCopy[shard],
                                    entry.getValue());
                            }
                        } else {
                            Transaction trans = zk.transaction();
                            for (Map.Entry<Integer,String> entry:
                                dirtyCopy.entrySet())
                            {
                                int shard = entry.getKey();
                                trans.setData(
                                    entry.getValue(),
                                    Long.toString(statusCopy[shard]).getBytes(),
                                    -1);
                                dataSize += entry.getValue().length() * 2;
                            }
                            trans.commit(writeStatusesCallback, dirtyCopy);
                        }
                        grouped = dirtyCopy.size();
                        dirtyCopy = null;
                        break;
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("Grouped: " + grouped + ", dataSize: "
                        + dataSize);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private void updateStatus(
        final int shard,
        final long position,
        final String path)
        throws Exception
    {
        HttpHost producer;
        synchronized (producersList) {
            producer = producersList.getFirst();
        }
        httpClient.execute(
            producer,
            new BasicAsyncRequestProducerGenerator(
                "/_updateConsumerPosition?service="
                + service + "&shard=" + shard
                + "&position=" + position + "&backend=" + backendName),
            new HttpStatusCallback(shard, position, path));
    }

    public void writeNodeStatus(final Node node) {
        int shard = node.shard.shardNo;
        synchronized (shardsLock) {
            if (statuses[shard] <= node.seq) {
                statuses[shard] = node.seq;
                dirty.put(shard, node.shard.statusPath());
                delayedStatuses.remove(shard);
            }
        }
    }

    public void writeNodeStatus(
        final int shard,
        final long position,
        final String path)
    {
        synchronized (shardsLock) {
            if (statuses[shard] <= position) {
                statuses[shard] = position;
                dirty.put(shard, path);
                delayedStatuses.remove(shard);
            }
        }
    }

    private void shuffleProducers() {
        synchronized (producersList) {
            final long currentTime = TimeSource.INSTANCE.currentTimeMillis();
            if (currentTime - lastShuffle > SUFFLE_DELAY) {
                lastShuffle = currentTime;
                producersList.addLast(producersList.removeFirst());
            }
        }
    }

    public void delayNodeStatus(final Node node) {
        int shard = node.shard.shardNo;
        synchronized (shardsLock) {
            DelayedNodeStatus status = delayedStatuses.get(shard);
            if (status == null) {
                delayedStatuses.put(
                    shard,
                    new DelayedNodeStatus(node.shard.statusPath(), node.seq));
            } else {
                status.touch(node.seq);
            }
        }
    }

    public void writeNodeError( Node node, String errorString, int code )
    {
        if (logger.isLoggable(Level.WARNING)) {
            logger.warning("writeNodeError: " + node.path + ", error:"
                + errorString);
        }
        ErrorMessage msg = new ErrorMessage( errorString, code, TimeSource.INSTANCE.currentTimeMillis() );
        byte[] data = null;
        try
        {
            data = SerializeUtils.serializeHttpMessage( msg );
        }
        catch( IOException ign )
        {
            //should never happen
        }
        while( true )
        {
            try
            {
                ZooKeeper zk = getZk();
                zk.setData( node.path, data, 0 );
                if (logger.isLoggable(Level.WARNING)) {
                    logger.warning("writeNodeError: success");
                }
                break;
            }
            catch( KeeperException.BadVersionException e )
            {
                //Other backend has been already created error message
                break;
            }
            catch( KeeperException.NoNodeException e )
            {
                //Message was washed out from queue memory, recreating
                try
                {
                    zk.create( node.path, data, null, CreateMode.EPHEMERAL ); //creating temporary node
                }
                catch( Exception ee )
                {
                    ee.printStackTrace();
                }
            }
            catch( Exception e )
            {
                e.printStackTrace();
            }
        }
    }

/*
    public void failed( Node n )
    {
        while( true )
        {
            try
            {
                failedQueue.put( n );
                return;
            }
            catch( java.lang.InterruptedException ign )
            {
            }
        }
    }
*/
    public ZooKeeper getZk() {
        while (true) {
            try {
                if( zk != null && zk.getState().isConnected() ) return zk;
                final ZooHost zh = dsn.nextHost();
                if (!dsn.isAlive(zh, logger)) {
                    Thread.sleep(1000);
                    continue;
                }
                zk = new ZooKeeper(zh.zkAddress(), timeout, this, logger);
                for (int i = 0; i < 100; i++) {
                    if (zk.getState().isConnected()) {
                        return zk;
                    }
                    Thread.sleep( 500 );
                }
                if (zk.getState().isConnected()) {
                    return zk;
                }
                zk.close();
                zk = null;
            } catch(Exception e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public void process(WatchedEvent event)
    {
//        Logger.debug("Watched event from <"+zkAddr+">: " + event);
//    	if( event.getState() != KeeperState.SyncConnected )
//    	{
//    	    zk.close();
//    	    zk = null;
//	}
    }

    private class DelayedStatusesTask extends TimerTask {
        @Override
        public void run() {
            long time = TimeSource.INSTANCE.currentTimeMillis();
            synchronized (shardsLock) {
                Iterator<Map.Entry<Integer,DelayedNodeStatus>> iter =
                    delayedStatuses.entrySet().iterator();
                while (iter.hasNext()) {
                    Map.Entry<Integer,DelayedNodeStatus> entry = iter.next();
                    if (time - entry.getValue().lastUpdate()
                        >= MAX_NOTIFY_DELAY)
                    {
                        final int shard = entry.getKey();
                        final long seq = entry.getValue().seq();
                        final String path = entry.getValue().path();
                        if (statuses[shard] <= seq) {
                            statuses[shard] = seq;
                            dirty.put(shard, path);
                            iter.remove();
                        }
                    }
                }
            }
        }
    }

    private class HttpStatusCallback implements FutureCallback<HttpResponse> {
        private final int shard;
        private final long position;
        private final String path;

        public HttpStatusCallback(
            final int shard,
            final long position,
            final String path)
        {
            this.shard = shard;
            this.position = position;
            this.path = path;
        }

        @Override
        public void completed(final HttpResponse response) {
            int code = response.getStatusLine().getStatusCode();
            if (code == 200) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("StatusQueue<" + shard + ">: updated position: "
                    + position);
                }
            } else {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("StatusQueue<" + shard + ">: "
                        + "position update failed: httpCode=" + code);
                }
                if (code == 500) {
                    shuffleProducers();
                }
                writeNodeStatus(shard, position, path);
            }
        }

        @Override
        public void cancelled() {
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe("StatusQueue<" + shard + ">: "
                    + "position update request cancelled");
            }
            writeNodeStatus(shard, position, path);
        }

        @Override
        public void failed(Exception e) {
            if (logger.isLoggable(Level.SEVERE)) {
                logger.log(
                    Level.SEVERE,
                    "StatusQueue<" + shard
                    + ">: position update request failed",
                    e);
            }
            shuffleProducers();
            writeNodeStatus(shard, position, path);
        }
    }

}
