package ru.yandex.dispatcher.consumer;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.IntPair;
import ru.yandex.dispatcher.common.HttpMessage;
import ru.yandex.http.config.AbstractHttpTargetConfigBuilder;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.HttpTargetConfig;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.proxy.HttpProxyTvm2HeaderSupplier;
import ru.yandex.http.proxy.MultiClientTvm2TicketSupplier;
import ru.yandex.http.util.HttpHostParser;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class Tvm2HttpConsumer extends LuceneConsumer {
    private final Map<String, AsyncClient> customClients;

    public Tvm2HttpConsumer(
        final HttpHost host,
        final HttpHost queueIdHost,
        final String service,
        final int workers,
        final ConsumerServer server,
        final ImmutableHttpTargetConfig targetConfig,
        final List<String> primaryKeyFields,
        final boolean groupNullKeys,
        final boolean queueIdHeader,
        final int maxRetryCount,
        final List<String> deduplicationKeyFields,
        final boolean ignoreBackendPosition,
        final long watchdogDelay,
        final boolean zeroTolerance,
        final Map<String, ImmutableHttpHostConfig> customDests)
    {
        super(
            host,
            queueIdHost,
            service,
            workers,
            server,
            targetConfig,
            primaryKeyFields,
            groupNullKeys,
            queueIdHeader,
            maxRetryCount,
            deduplicationKeyFields,
            ignoreBackendPosition,
            watchdogDelay,
            zeroTolerance);


        Map<String, AsyncClient> customClients = new LinkedHashMap<>();
        for (Map.Entry<String, ImmutableHttpHostConfig> entry: customDests.entrySet()) {
            ImmutableHttpHostConfig destConfig = entry.getValue();
            // we could just wrap super client instead of creating new - shall we (?)
            AsyncClient httpClient =
                new AsyncClient(server.reactor(service, null), destConfig);
            if (destConfig.statersConfig() != null) {
                httpClient = httpClient.adjustAsteriskStater();
                server.registerStaters(destConfig.statersConfig());
            }

            if (!destConfig.tvm2Headers().isEmpty()) {
                logger.info(
                    "Registering tvm headers for client " + destConfig.host()
                        + ": " + destConfig.tvm2Headers());
                httpClient = httpClient.addHeader(
                    new HttpProxyTvm2HeaderSupplier(
                        server.tvm2TicketSupplier(),
                        destConfig.tvm2Headers()));
            }

            customClients.put(destConfig.host().getHostName(), httpClient);
        }

        this.customClients = Collections.unmodifiableMap(customClients);
        server.logger().info("TvmHostMap " + customDests);
    }

    @Override
    public void start() {
        super.start();

        for (AsyncClient client: this.customClients.values()) {
            client.start();
        }
    }

    @Override
    protected Future<IntPair<Void>> runMessage(
        final HttpMessage msg,
        final long queueId,
        final int shard,
        final String service,
        final String cgiParams)
    {
        AsyncClient client = adjustTvm2(msg);
        // abstract http backend consumer will rewrite ticket, so we call directly
        return msg.run(
            host,
            client,
            queueId,
            shard,
            service,
            deadline(cgiParams));
    }

    @Override
    protected void runMessage(
        final HttpMessage msg,
        final long queueId,
        final int shard,
        final String service,
        final String cgiParams,
        final FutureCallback<IntPair<Void>> callback)
    {
        AsyncClient client = adjustTvm2(msg);
        // abstract http backend consumer will rewrite ticket, so we call directly
        msg.run(
            host,
            client,
            queueId,
            shard,
            service,
            deadline(cgiParams),
            callback);
    }

    protected AsyncClient adjustTvm2(final HttpMessage message) {
        try {
            URI uri = new URI(message.uri());
            AsyncClient client = customClients.get(uri.getHost());
            if (client != null) {
                message.deleteHeader(YandexHeaders.X_YA_SERVICE_TICKET);
                return client;
            } else {
                logger.info("TvmClientId null for " + uri);
                return httpClient;
            }
            //String clientId = clientIdMap.get(uri.getHost());
//            if (clientId != null) {
//                String ticket = tvm2TicketRenewalTask.ticket(clientId);
//                message.deleteHeader(YandexHeaders.X_YA_SERVICE_TICKET);
//                message.setHeader(YandexHeaders.X_YA_SERVICE_TICKET, ticket);
//                //logger.info("TvmClientId " + clientId + " for " + uri + " ticket " + ticket);
//            } else {
//                logger.info("TvmClientId null for " + uri);
//            }
        } catch (URISyntaxException urie) {
            logger.warning("Failed to parse uri " + message.uri());
            return httpClient;
        }
    }

    private static class TargetConfigWrapper
        extends AbstractHttpTargetConfigBuilder<TargetConfigWrapper>
        implements HttpHostConfig
    {
        public TargetConfigWrapper(final HttpTargetConfig config) {
            super(config);
        }

        @Override
        public TargetConfigWrapper self() {
            return this;
        }

        @Override
        public HttpHost host() {
            return null;
        }
    }
    public static Map<String, ImmutableHttpHostConfig> parseHostClientIdMap(
        final IniConfig consumerConfig, final HttpTargetConfig config)
        throws ConfigException
    {
        IniConfig clientSection = consumerConfig.sectionOrNull("client");
        Map<String, ImmutableHttpHostConfig> result = new LinkedHashMap<>();
        for (Map.Entry<String, IniConfig> entry
            : clientSection.sections().entrySet())
        {
            System.err.println("Parsing section " + entry.getKey());
            IniConfig section = entry.getValue();
            ImmutableHttpHostConfig hostConfig =
                new HttpHostConfigBuilder(section, new TargetConfigWrapper(config)).build();
            //HttpHost host =
            //    section.get("host", null, HttpHostParser.INSTANCE);
            //String clientId = section.getString("client-id", null);

//            if (host == null || clientId == null) {
//                continue;
//            }

            result.put(hostConfig.host().getHostName(), hostConfig);
        }

        return Collections.unmodifiableMap(result);
    }

    @Override
    public Map<String, Object> status(boolean verbose) {
        Map<String, Object> intStatus = new LinkedHashMap<>();
        intStatus.put("default", super.status(verbose));
        for (Map.Entry<String, AsyncClient> entry: customClients.entrySet()) {
            intStatus.put(entry.getKey(), entry.getValue().status(verbose));
        }

        return intStatus;
    }
}
