package ru.yandex.dispatcher.consumer;

import java.io.IOException;
import java.util.logging.Level;

import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.ZooKeeper;

import ru.yandex.logger.PrefixedLogger;

public class ZooKeeperConnection implements Watcher {
    public ZooKeeper zk;
    private final String zkAddr;
    private final int timeout;
    private final PrefixedLogger logger;
    private boolean connected = false;
    private boolean closed = false;

    public ZooKeeperConnection(
        final String zkAddr,
        final int timeout,
        final PrefixedLogger logger)
    {
        this.zkAddr = zkAddr;
        this.timeout = timeout;
        this.logger = logger;
    }

    public synchronized void connect(final PrefixedLogger logger) {
        if (closed) {
            return;
        }
        try {
            if (zk == null) {
                zk = new ZooKeeper(zkAddr, timeout, this, logger);
            }
            for (int i = 0; i < 100 && !closed; i++) {
                if (zk.getState().isConnected()) {
                    connected = true;
                    return;
                }
                wait(100L);
            }
            if (!closed) {
                if (zk.getState().isConnected()) {
                    connected = true;
                } else {
                    closed = true;
                    zk.close();
                }
            }
        } catch (IOException | InterruptedException e) {
            logger.log(
                Level.SEVERE,
                "ZooKeeperConnection.connect exception",
                e);
            close();
        }
    }

    public synchronized void close() {
        if (!closed) {
            closed = true;
            if (zk == null) {
                return;
            }
            zk.close();
            if (!connected) {
                // wake up all connecting threads
                notifyAll();
            }
        }
    }

    public synchronized boolean isConnected() {
        return !closed
            && zk != null
            && zk.getState().isConnected()
            && zk.getState().isAlive();
    }

    @Override
    public synchronized void process(final WatchedEvent event) {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("Watched event from <" + zkAddr + ">: " + event);
        }
        if (event.getState() != KeeperState.SyncConnected) {
            close();
        } else if (!connected) {
            // wake up all connecting threads
            notifyAll();
        }
    }

}
