package ru.yandex.dispatcher.consumer;

import java.io.IOException;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import java.util.logging.Level;

import org.apache.zookeeper.KeeperException;

import ru.yandex.dispatcher.producer.SearchMap;
import ru.yandex.dispatcher.consumer.config.ImmutableConsumerConfig;
import ru.yandex.dispatcher.consumer.lock.FakeQueueLock;
import ru.yandex.dispatcher.consumer.lock.PerConsumerQueueLock;
import ru.yandex.dispatcher.consumer.lock.QueueLock;
import ru.yandex.dispatcher.consumer.shard.Shard;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.http.util.nio.client.AsyncClient;

public class ZooKeeperConsumer
{
    private final AsyncClient httpClient;
    private final AsyncClient asyncClient;
    private final PrefixedLogger logger;
    private final String zkAddr;
    private final SearchMap searchMap;
    private final String consumerHostname;
    private final Set<Shard> haltedShards = new HashSet<>();
    private final int workers;
    private final int timeout;
    private final long watchdogDelay;
    private final long readStatusDelay;
    private final int nextIdFinderMinHostCountPct;
    private final int advanceOnMissed;
    private final Set<String> ignorePosition;
    private final boolean responseless;
    private final long sleepInterval;
    private final int prefetchCount;
    private final long statusGroupingTime;
    private final List<String> producers;

    private final Map<ShardKey, Shard> shards;

    ZooKeeperConsumer(
        final String zkAddr,
        final String consumerHostname,
        final SearchMap searchMap,
        final AsyncClient httpClient,
        final AsyncClient asyncClient,
        final PrefixedLogger logger,
        final ImmutableConsumerConfig config)
        throws IOException, InterruptedException, KeeperException
    {
        this.zkAddr = zkAddr;
        this.consumerHostname = consumerHostname;
        this.searchMap = searchMap;
        this.httpClient = httpClient;
        this.asyncClient = asyncClient;
        this.logger = logger;

        workers = config.workers();
        timeout = config.timeout();
        watchdogDelay = config.watchdogDelay();
        readStatusDelay = config.readStatusDelay();
        nextIdFinderMinHostCountPct = config.nextIdFinderMinHostCountPct();
        advanceOnMissed = config.advanceOnMissed();
        ignorePosition = config.ignorePosition();
        responseless = config.responseless();
        sleepInterval = config.sleepInterval();
        prefetchCount = config.prefetchCount();
        statusGroupingTime = config.statusGroupingTime();
        producers = config.producers();

        shards = new HashMap<>();
    }

    public void init(
        final Map<String, BackendConsumer> backendConsumers,
        final Set<String> consumerTags)
        throws IOException
    {
        PrefixedLogger zooLogger = logger.replacePrefix("ZooLooserPool");
        ZooKeeperPool zooKeeperPool = new ZooKeeperPool(timeout, asyncClient);
        HashMap<String, StatusQueue> perServiceStatusQueue = new HashMap<>();

        for (SearchMap.Interval interval: searchMap.getZooKeeperIntervals(zkAddr)) {
            if (!consumerTags.isEmpty()
                && !consumerTags.contains(interval.tag()))
            {
                continue;
            }
            if( !interval.hostname().equals(consumerHostname) ) continue;

            final String targetHostname;
            if (interval.targetHost() != null) {
                targetHostname = interval.targetHost();
            } else {
                targetHostname = consumerHostname;
            }

            final String statusQueueKey =
                interval.consumerName() + '@' + interval.service();
            StatusQueue statusQueue = perServiceStatusQueue.get(statusQueueKey);

            if (statusQueue == null) {
                statusQueue =
                    new StatusQueue(
                        new ZooKeeperDSN(zkAddr, zooLogger, asyncClient),
                        timeout,
                        interval.consumerName(),
                        interval.service(),
                        logger,
                        statusGroupingTime,
                        producers,
                        httpClient);
                perServiceStatusQueue.put(statusQueueKey, statusQueue);
            }

            String backendAddress =
                interval.service() + '/' + targetHostname
                + ':' + interval.indexPort();
            BackendConsumer bc = backendConsumers.get(backendAddress);
            String loggerPrefix = '@' + interval.service() + '/';
            int dot = targetHostname.indexOf('.');
            if (dot != -1) {
                loggerPrefix += targetHostname.substring(0, dot);
            } else {
                loggerPrefix += targetHostname;
            }

            QueueLock queueLock = createLock(interval.lockType(), zkAddr, interval.service(), interval.lockName());
            if (logger.isLoggable(Level.INFO)) {
                logger.info("Consumer lock: " + queueLock);
            }

            int minShard = interval.min();
            int maxShard = Math.min(interval.max(), SearchMap.SHARDS_COUNT - 1);
            for (int i = minShard; i <= maxShard; i++) {
                ShardKey shardKey =
                    new ShardKey(
                        interval.service(),
                        backendAddress,
                        i);
                if (!shards.containsKey(shardKey)) {
                    boolean ignorePosition;
                    if (Collections.singleton("true")
                        .equals(this.ignorePosition))
                    {
                        ignorePosition = true;
                    } else if (Collections.singleton("false")
                        .equals(this.ignorePosition))
                    {
                        ignorePosition = false;
                    } else {
                        ignorePosition =
                            this.ignorePosition.contains(interval.service());
                    }
                    Shard shard = new Shard(
                        this,
                        logger.replacePrefix(i + loggerPrefix),
                        interval.service(),
                        i,
                        zooKeeperPool,
                        new ZooKeeperDSN(zkAddr, zooLogger, asyncClient),
                        bc,
                        httpClient,
                        statusQueue,
                        advanceOnMissed,
                        ignorePosition,
                        responseless,
                        sleepInterval,
                        queueLock,
                        prefetchCount,
                        nextIdFinderMinHostCountPct,
                        watchdogDelay,
                        readStatusDelay);
                    shards.put(shardKey, shard);
                }
            }
        }

        if (logger.isLoggable(Level.INFO)) {
            logger.info("ZooKeeperConsumer<" + zkAddr + "> Initialized");
        }
    }

    private QueueLock createLock(final String lockType, final String zk,
        final String service, final String lockName)
    {
        if (lockType == null) {
            return new FakeQueueLock();
        }
        final String type = lockType.toLowerCase();
        if (type.startsWith("perconsumer")
            || type.startsWith("perhost"))
        {
            return new PerConsumerQueueLock(
                asyncClient,
                zk,
                service,
                lockName,
                logger.replacePrefix(
                    "PerConsumerQueueLock(" + lockName + '@' + service + ')'));
        }
        return new FakeQueueLock();
    }


    public void start() {
        for(Shard shard : shards.values()) {
            shard.reset();
        }
    }

    public void shardHalted(final Shard shard) {
        synchronized (haltedShards) {
            haltedShards.add(shard);
        }
    }

    public void shardUnhalted(final Shard shard) {
        synchronized (haltedShards) {
            haltedShards.remove(shard);
        }
    }

    public void fillHaltedShards(final List<String> halted) {
        synchronized (haltedShards) {
            for (Shard shard : haltedShards) {
                halted.add(shard.toString());
            }
        }
    }

    public boolean resetShard(
        final String service,
        final int shardNumber,
        final String backend)
    {
        if (backend != null) {
            ShardKey shardKey = new ShardKey(service, backend, shardNumber);
            Shard shard = shards.get(shardKey);
            if (shard == null) {
                return false;
            }
            shard.hardReset();
            return true;
        } else {
            boolean found = false;
            for (Map.Entry<ShardKey, Shard> entry : shards.entrySet()) {
                final ShardKey shardKey = entry.getKey();
                if (shardKey.service().equals(service)
                    && shardKey.shardNum() == shardNumber)
                {
                    final Shard shard = entry.getValue();
                    shard.hardReset();
                    found = true;
                }
            }
            return found;
        }
    }

    public boolean resetShardBatch(
            final int shardNumber,
            final int shardCount)
    {
            boolean found = false;
            for (Map.Entry<ShardKey, Shard> entry : shards.entrySet()) {
                final ShardKey shardKey = entry.getKey();
                if (shardKey.shardNum() % shardCount == shardNumber)
                {
                    final Shard shard = entry.getValue();
                    shard.hardReset();
                    found = true;
                }
            }
            return found;
    }

    private final static int FAILED_RETRY_TIMEOUT = 2000;
    private static final Object watchWatcherCheck = new Object();

    private static class ShardKey {
        private final String service;
        private final String backend;
        private final int shardNum;
        public ShardKey(
            final String service,
            final String backend,
            final int shardNum)
        {
            this.service = service;
            this.backend = backend;
            this.shardNum = shardNum;
        }

        public String service() {
            return service;
        }

        public int shardNum() {
            return shardNum;
        }

        @Override
        public int hashCode() {
            return service().hashCode() ^ backend.hashCode() ^ shardNum;
        }

        @Override
        public boolean equals(Object o) {
            if (o instanceof ShardKey) {
                ShardKey other = (ShardKey) o;
                return service.equals(other.service)
                    && backend.equals(other.backend)
                    && shardNum == other.shardNum;
            }
            return false;
        }
    }
}
