package ru.yandex.dispatcher.consumer;

import java.io.IOException;

import java.text.ParseException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import org.apache.http.HttpHost;

import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.AsyncCallback.ChildrenCallback;
import org.apache.zookeeper.AsyncCallback.DataCallback;
import org.apache.zookeeper.AsyncCallback.StatCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.data.Stat;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.ZooDefs.Ids;
import org.apache.zookeeper.ZooKeeper;

import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncDomConsumerFactory;
import ru.yandex.logger.PrefixedLogger;

public class ZooKeeperPool {
    private static final ConcurrentHashMap<ZooHost, ConnectionChecker>
        CHECKERS = new ConcurrentHashMap<>();
    private final Map<ZooHost, ZooKeeperConnection> pool =
        new ConcurrentHashMap<>();
    private final int timeout;
    private final AsyncClient httpAsyncClient;

    public ZooKeeperPool(
        final int timeout,
        final AsyncClient httpAsyncClient)
    {
        this.timeout = timeout;
        this.httpAsyncClient = httpAsyncClient;
    }

    public ZooKeeperConnection getConnection(
        final ZooHost host,
        final PrefixedLogger logger)
    {
        if (!isAlive(host, logger, httpAsyncClient)) {
            return null;
        }
        while (true) {
            ZooKeeperConnection zkc =
                pool.computeIfAbsent(
                    host,
                    h -> new ZooKeeperConnection(h.zkAddress(), timeout, logger));
            // isConnected(), connect() and close() all synchronized, so this
            // synchronization is free of charge
            synchronized (zkc) {
                if (zkc.isConnected()) {
                    return zkc;
                } else {
                    zkc.connect(logger);
                    if (zkc.isConnected()) {
                        // Check that no one replaced our connection in the pool
                        ZooKeeperConnection oldConn =
                            pool.putIfAbsent(host, zkc);
                        if (oldConn == null || oldConn == zkc) {
                            return zkc;
                        }
                        // Someone already replaced or removed our connection,
                        // reject it and try again
                        zkc.close();
                    } else {
                        // Can't connect
                        pool.remove(host, zkc);
                        zkc.close();
                        return null;
                    }
                }
            }
        }
    }

    public int load(final ZooHost host) {
        final ConnectionChecker cc = CHECKERS.get(host);
        if (cc == null) {
            return ConnectionChecker.DEFAULT_LOAD;
        } else {
            return cc.load();
        }
    }

    public boolean leader(final ZooHost host) {
        final ConnectionChecker cc = CHECKERS.get(host);
        if (cc == null) {
            return false;
        } else {
            return cc.leader();
        }
    }

    public ZooHost getLessLoadedHost(final List<ZooHost> hosts) {
        final List<ConnectionChecker> checkers =
            new ArrayList<>(hosts.size());
        for (final ZooHost host : hosts) {
            final ConnectionChecker cc = CHECKERS.get(host);
            if (cc != null) {
                checkers.add(cc);
            }
        }
        if (checkers.size() == 0) {
            return hosts.get(0);
        }
        final float leaderOffload = checkers.size();
        Collections.sort(
            checkers,
            new Comparator<ConnectionChecker>() {
                private float load(final ConnectionChecker cc) {
                    float load = cc.load();
                    if (cc.leader()) {
                        load *= leaderOffload;
                    }
                    return load;
                }

                @Override
                public int compare(
                    final ConnectionChecker cc1,
                    final ConnectionChecker cc2)
                {
                    return Float.compare(load(cc1), load(cc2));
                }
            });
        return checkers.get(0).host();
    }

    public static boolean isAlive(
        final ZooHost host,
        final PrefixedLogger logger,
        final AsyncClient httpAsyncClient)
    {
        ConnectionChecker cc = CHECKERS.get(host);
        if (cc == null) {
            ConnectionChecker newCc =
                new ConnectionChecker(host, logger, httpAsyncClient);
            cc = CHECKERS.putIfAbsent(host, newCc);
            if (cc == null) {
                cc = newCc;
                try {
                    cc.start();
                } catch(java.lang.IllegalThreadStateException ee) {
                    //alread started
                }
            }
        }
        return cc.alive;
    }

    private static class ConnectionChecker extends Thread implements Watcher {
        private static final int MAX_FAILED_COMPUTES = 3;
        private static final int LOAD_AVG_DEPTH = 20;
        private static final int DEFAULT_LOAD =
            Integer.MAX_VALUE / LOAD_AVG_DEPTH;
        private final ZooHost host;
        private final PrefixedLogger logger;
        private final AsyncClient httpAsyncClient;
        public volatile boolean alive = true;
        private final AtomicInteger load = new AtomicInteger(Integer.MAX_VALUE);
        private final AtomicBoolean leader = new AtomicBoolean(false);
        private final String jsonStatUrl;
        private final String leaderStatUrl;
        private final BasicAsyncRequestProducerGenerator jsonGet;
        private final BasicAsyncRequestProducerGenerator leaderGet;
        private final LinkedList<Integer> loadLog = new LinkedList<>();
        private long loadSum = 0;
        private int failedComputes = 0;

        public ConnectionChecker(
            final ZooHost host,
            final PrefixedLogger logger,
            final AsyncClient httpAsyncClient)
        {
            super("ConnectionChecker-" + host);
            setDaemon(true);
            this.host = host;
            this.logger =
                logger.replacePrefix("ZooLooserPool ConnectionChecker<"
                    + host + ">:");
            this.httpAsyncClient = httpAsyncClient;
            jsonStatUrl = "/status";
            leaderStatUrl = "/stat";
            jsonGet = new BasicAsyncRequestProducerGenerator(jsonStatUrl);
            leaderGet = new BasicAsyncRequestProducerGenerator(leaderStatUrl);
        }

        public ZooHost host() {
            return host;
        }

        private final int getInt(final Map<?, ?> jsonMap, final String name)
            throws ParseException
        {
            final Object value = jsonMap.get(name);
            if (value == null || !(value instanceof Number)) {
                throw new ParseException("/status json parsing failed: "
                    + "absent or not a number \"" + name + "\" field", 0);
            }
            return ((Number) value).intValue();
        }

        private void pushLoad(final int load) {
            loadLog.add(load);
            loadSum += load;
            if (loadLog.size() > LOAD_AVG_DEPTH) {
                loadSum -= loadLog.pollFirst();
            }
            this.load.set((int) (loadSum / loadLog.size()));
        }

        private void computeLoad() {
            try {
                final Future<Object> jsonFuture = httpAsyncClient.execute(
                    host.httpHost(),
                    jsonGet,
                    JsonAsyncDomConsumerFactory.OK,
                    EmptyFutureCallback.INSTANCE);
                final Object jsonRoot = jsonFuture.get();
                if (!(jsonRoot instanceof Map)) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("/status json parsing failed: json root "
                            + "is not a map: " + jsonRoot);
                    }
                    load.set(DEFAULT_LOAD);
                    return;
                }
                final Map<?,?> statMap = (Map<?,?>) jsonRoot;
                final int activeWorkers = getInt(statMap, "active_workers");
                final int totalWorkers = getInt(statMap, "spawned_workers");
                int load = (activeWorkers * 100) / totalWorkers;
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("active_workers: " + activeWorkers
                        + ", total_workers: " + totalWorkers
                        + ", http_load: " + load);
                }

                final Future<String> stringFuture = httpAsyncClient.execute(
                    host.httpHost(),
                    leaderGet,
                    AsyncStringConsumerFactory.OK,
                    EmptyFutureCallback.INSTANCE);
                final String body = stringFuture.get();
                if (!body.equals("Not a leader")) {
                    leader.set(true);
                    pushLoad(load);
                    if (logger.isLoggable(Level.INFO)) {
                        logger.info("leader, current load: " + load
                            + ", avg load: " + this.load.get());
                    }
                } else {
                    leader.set(false);
                    pushLoad(load);
                    if (logger.isLoggable(Level.INFO)) {
                        logger.info("follower, current load: " + load
                            + ", avg load: " + this.load.get());
                    }
                }
                failedComputes = 0;
            } catch (Exception ign) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.log(
                        Level.SEVERE,
                        "Error computing server load",
                        ign);
                }
                //ignore sporadic connection loss|refuses, etc
                if (failedComputes++ > MAX_FAILED_COMPUTES) {
                    load.set(DEFAULT_LOAD);
                }
            }
        }

        public boolean leader() {
            return leader.get();
        }

        public int load() {
            return load.get();
        }

        public synchronized void run() {
            while (true) {
                try {
                    if (logger.isLoggable(Level.FINE)) {
                        logger.fine("connecting");
                    }
                    ZooKeeper zk =
                        new ZooKeeper(host.zkAddress(), 30000, this, logger);
                    int tries = 30;
                    while(true) {
                        if (!zk.getState().isAlive() || !zk.getState().isConnected()) {
                            if (alive == true) {
                                //was connected before and has been disconnected
                                if (logger.isLoggable(Level.SEVERE)) {
                                    logger.severe("disconnected: "
                                        + zk.getState());
                                }
                                alive = false;
                                zk.close();
                                Thread.sleep( 5000 );
                                break;
                            } else {
                                //connecting
                                if (tries-- == 0) {
                                    if (logger.isLoggable(Level.SEVERE)) {
                                        logger.severe("disconnected: "
                                            + zk.getState());
                                    }
                                    alive = false;
                                    zk.close();
                                    Thread.sleep( 5000 );
                                    break;
                                };
                            }
                            if (logger.isLoggable(Level.FINE)) {
                                logger.fine("connecting: "
                                    + zk.getState());
                            }
                            wait(1000);
                        } else {
                            if (logger.isLoggable(Level.FINE)) {
                                logger.fine("connected" );
                            }
                            alive = true;
                            computeLoad();
                            wait(30000);
                        }
                    }
                } catch (InterruptedException e) {
                } catch (Exception e) {
                    if (logger.isLoggable(Level.WARNING)) {
                        logger.log(
                            Level.WARNING,
                            "Failed to check connection to " + host,
                            e);
                    }
                    try {
                        wait(1000);
                    } catch (InterruptedException ex) {
                    }
                }
            }
        }

        @Override
        public synchronized void process(final WatchedEvent event) {
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe("event: " + event);
            }
            notify();
        }
    }
}

