package ru.yandex.dispatcher.consumer.logstore;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.util.zip.CRC32;
import java.util.zip.Deflater;

public class ShardEpoch {
    public final long epoch;
    private final String dataPath;
    private final String indexPath;
    private RandomAccessFile dataFile = null;
    private RandomAccessFile indexFile = null;
    private long nextFilePointer;
    private final IndexPoint indexPoint = new IndexPoint();
    private static final int BUFFER_SIZE = 2048;

    private static final ThreadLocal<DeflaterState> deflaterLocal = new ThreadLocal<DeflaterState>();

    private static final byte[] emptyIndex = emptyIndexBytes();

    private static class DeflaterState {
        public Deflater deflater;
        public CRC32 crc;
        public byte[] buffer;
    }

    public static final int POINT_SIZE = 8 + 4 + 4;
    private static class IndexPoint {
        public long dataPos;
        public int plainSize;
        public int compressedSize;
        public byte[] binary = new byte[POINT_SIZE];
        private ByteBuffer bb = ByteBuffer.wrap(binary);

        public byte[] encode() {
            bb.clear();
            bb.putLong(dataPos);
            bb.putInt(plainSize);
            bb.putInt(compressedSize);
            return binary;
        }

    }

    private final static int GZIP_MAGIC = 0x8b1f;
    private final static int TRAILER_SIZE = 8;

    private final static byte[] GZIP_HEADER = {
        (byte) GZIP_MAGIC,                // Magic number (short)
        (byte)(GZIP_MAGIC >> 8),          // Magic number (short)
        Deflater.DEFLATED,                // Compression method (CM)
        0,                                // Flags (FLG)
        0,                                // Modification time MTIME (int)
        0,                                // Modification time MTIME (int)
        0,                                // Modification time MTIME (int)
        0,                                // Modification time MTIME (int)
        0,                                // Extra flags (XFLG)
        0                                 // Operating system (OS)
    };

    private static byte[] emptyIndexBytes() {
        IndexPoint indexPoint = new IndexPoint();
        indexPoint.dataPos = -1;
        indexPoint.plainSize = 0;
        indexPoint.compressedSize = 0;
        byte[] nullPoint = indexPoint.encode();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        for (int i = 0; i < LogStore.MSGS_PER_FILE; i++) {
            try {
                baos.write(nullPoint);
            } catch (IOException ign) {
            }
        }
        return baos.toByteArray();
    }

    public ShardEpoch(final long epoch, final String dataPath,
        final String indexPath) throws IOException
    {
        this.epoch = epoch;
        this.dataPath = dataPath;
        this.indexPath = indexPath;
        dataFile = new RandomAccessFile(dataPath, "rw");
        nextFilePointer = dataFile.length();
        indexFile = new RandomAccessFile(indexPath, "rw");
        long indexLength = indexFile.length();
        if (indexLength < POINT_SIZE * LogStore.MSGS_PER_FILE && indexLength > 0) {
            if (dataFile.length() == 0) {
                //index writing was aborted by restart
                initIndex();
            } else {
                throw new IOException("Index is corrupted for " + dataPath);
            }
        } else if (indexLength == 0) {
            initIndex();
        }
    }

    private final void initIndex() throws IOException {
        indexFile.seek(0);
        indexFile.write(emptyIndex);
    }

    public synchronized void putMessage(long seq, byte[] data)
        throws IOException
    {
        dataFile.seek(nextFilePointer);
        indexPoint.compressedSize = writeCompressed(dataFile, data);
        indexPoint.dataPos = nextFilePointer;
        indexPoint.plainSize = data.length;
        indexFile.seek(seq * POINT_SIZE);
        indexFile.write(indexPoint.encode());
//        nextFilePointer += data.length;
        nextFilePointer = dataFile.getFilePointer();
    }

    public synchronized void close() throws IOException {
        if (dataFile != null) {
            dataFile.close();
            dataFile = null;
        }
        if (indexFile != null) {
            indexFile.close();
            indexFile = null;
        }
    }

    public static int writeCompressed(final RandomAccessFile file, final byte[] data)
        throws IOException
    {
        DeflaterState dState = deflaterLocal.get();
        if (dState == null) {
            dState = new DeflaterState();
            dState.deflater = new Deflater(1,true);
//            dState.deflater.setLevel(1);
            dState.buffer = new byte[BUFFER_SIZE];
            dState.crc = new CRC32();
            deflaterLocal.set(dState);
        }

        Deflater deflater = dState.deflater;
        CRC32 crc = dState.crc;
        byte[] buffer = dState.buffer;
	int deflated = GZIP_HEADER.length + TRAILER_SIZE;

	deflater.reset();
	crc.reset();

        file.write(GZIP_HEADER);

        crc.update(data);
        deflater.setInput(data, 0, data.length);
	deflater.finish();
	while(!deflater.finished()) {
	int ret;
	    ret = deflater.deflate(buffer, 0, BUFFER_SIZE);
	    deflated += ret;
	    file.write(buffer, 0, ret);
	}
	writeTrailer((int)crc.getValue(), data.length, buffer, 0);
	file.write(buffer, 0, TRAILER_SIZE);
        return deflated;
    }

    /*
     * Writes GZIP member trailer to a byte array, starting at a given
     * offset.
     */
    private static void writeTrailer(int crc, int plainSize, byte[] buf, int offset) throws IOException {
        writeInt(crc, buf, offset); // CRC-32 of uncompr. data
        writeInt(plainSize, buf, offset + 4); // Number of uncompr. bytes
    }

    /*
     * Writes integer in Intel byte order to a byte array, starting at a
     * given offset.
     */
    private static void writeInt(int i, byte[] buf, int offset) throws IOException {
        writeShort(i & 0xffff, buf, offset);
        writeShort((i >> 16) & 0xffff, buf, offset + 2);
    }

    /*
     * Writes short integer in Intel byte order to a byte array, starting
     * at a given offset
     */
    private static void writeShort(int s, byte[] buf, int offset) throws IOException {
        buf[offset] = (byte)(s & 0xff);
        buf[offset + 1] = (byte)((s >> 8) & 0xff);
    }
}
