package ru.yandex.dispatcher.consumer.logstore;

import java.io.File;
import java.io.IOException;

import ru.yandex.dispatcher.common.HttpMessage;
import ru.yandex.dispatcher.common.SerializeUtils;

public class ShardStore {
    private final String path;
    private final int shardNo;
    private final String service;
    private final String positionPath;
    private ShardEpoch currentEpoch;
    private boolean initialized = false;
    private ShardPos shardPos = null;

    public ShardStore(
        final String path,
        final int shardNo,
        final String service)
        throws IOException
    {
        this.path = path;
        this.shardNo = shardNo;
        this.service = service;
        positionPath = path + '/' + shardNo + "_current_position";
        File posFile = new File(positionPath);
        if (posFile.exists()) {
            shardPos = new ShardPos(positionPath);
        }
    }

    public static final String paddedLong(final long l) {
        return String.format("%020d", l);
    }

    public synchronized void putMessage(
        final long seq,
        final HttpMessage msg)
        throws IOException
    {
        checkInit();
        long epoch = seq / LogStore.MSGS_PER_FILE;
        if (currentEpoch == null || currentEpoch.epoch != epoch) {
            if (currentEpoch != null) {
                currentEpoch.close();
            }
            currentEpoch = new ShardEpoch(
                epoch,
                path+"/"+shardNo + "_" + paddedLong(epoch)+".dat",
                path+"/"+shardNo + "_" + paddedLong(epoch)+".idx");
        }
        currentEpoch.putMessage(
            seq % LogStore.MSGS_PER_FILE,
            msg.toByteArray(seq, shardNo, service));
        shardPos.setPos(seq);
    }

    public void checkInit() throws IOException {
        if (!initialized) {
            File root = new File(path);
            if (!root.exists()) {
                root.mkdirs();
            }
            shardPos = new ShardPos(positionPath);
            initialized = true;
        }
    }

    public synchronized long getPos() {
        if (shardPos == null) {
            return -1;
        } else {
            return shardPos.getPos();
        }
    }

    public synchronized void setPos(long pos) throws IOException {
        checkInit();
        if (shardPos == null) {
            return;
        }
        shardPos.setPos(pos);
    }
}
