package ru.yandex.dispatcher.consumer.shard;

import java.util.logging.Level;

import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;

public class FileCreator extends ShardTask
    implements AsyncCallback.StringCallback, Watcher
{
    private final String path;
    private final byte[] data;
    private final ShardTask finishCallback;

    public FileCreator(final Shard shard, final String path, final byte[] data,
        final ShardTask finishCallback) {
        super(shard);
        this.path = path;
        this.data = data;
        this.finishCallback = finishCallback;
    }

    @Override
    public void run() {
        synchronized(shard) {
            initOperId();
            createFile(path);
        }
    }

    private void createFile(String path) {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("FileCreator<"+ path +">.create: oper = " + operId);
        }
        ZooKeeper zk = shard.getZk();
        zk.create(path, data, null, CreateMode.PERSISTENT, this, null);
    }

    @Override
    public void processResult(final int rc, final String path, final Object ctx,
        final String name)
    {
        synchronized(shard) {
            if (!checkOper()) return;
            if (rc == 0) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("FileCreator<" + path + ">.create ok");
                }
                finishCallback.run();
            } else if (rc == KeeperException.Code.NODEEXISTS.intValue()) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("FileCreator<" + path +
                        ">.create node already exists");
                }
                finishCallback.run();
            } else if (rc == KeeperException.Code.NONODE.intValue()) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("FileCreator<" + path +
                        ">.create: parent path does not exists." +
                        " Trying to create.");
                }
                String parent = path.substring(0, path.lastIndexOf('/'));
                new FileCreator(shard, parent, null, this).run();
            } else if (rc == KeeperException.Code.CONNECTIONLOSS.intValue()) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("FileCreator<" + path +
                        ">.create error: connection loss");
                }
                run();
            } else {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("FileCreator<" + path +
                        ">.create unknown error: " + rc);
                }
                //restart request
                shard.reset();
            }
        }
    }

    @Override
    public void process(final WatchedEvent event) {
        synchronized(shard) {
            if (!checkOper()) return;
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("FileCreator<" + path + ">: watched event "
                    + event);
            }
            if (event.getState() == KeeperState.SyncConnected) {
                if (event.getType() == Watcher.Event.EventType.NodeCreated) {
                    finishCallback.run();
                } else {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("FileCreator<" + path
                            + ">: unhandled event " +event);
                    }
                    shard.reset();
                }
            } else if (event.getState() == KeeperState.Expired) {
                run();
            } else if (event.getState() == KeeperState.Disconnected) {
                run();
            }
        }
    }
}
