package ru.yandex.dispatcher.consumer.shard;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Future;
import java.util.logging.Level;

import org.apache.jute.BinaryInputArchive;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.MultiResponse;
import org.apache.zookeeper.OpResult;
import org.apache.zookeeper.ZooDefs;

import ru.yandex.dispatcher.common.SerializeUtils;
import ru.yandex.dispatcher.consumer.Node;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.util.timesource.TimeSource;

public class HttpNodeReader extends ShardTask
    implements FutureCallback<HttpResponse>
{
    private long currentNum;
    private boolean notifyStatus;
    private final ResetTask resetTask;
    private Future asyncHangGuard = null;

    public HttpNodeReader(final Shard shard) {
        super(shard);
        resetTask = new ResetTask(shard);
    }

    @Override
    public void run() {
        String request;
        synchronized(shard) {
            initOperId();
            request = prepareRead(shard.currentPos() + 1);
        }
        //should not be syncrhonized
        read(request);
    }

    public void runNext(final long startNum) {
        String request;
        synchronized(shard) {
            initOperId();
            request = prepareRead(startNum);
        }
        //should not be syncrhonized
        read(request);
    }

    private void read(final String request) {
        shard.httpClient().execute(
            shard.currentHost().httpHost(),
            new BasicAsyncRequestProducerGenerator(request),
            this);
    }

    private String prepareRead(final long num) {
        currentNum = num;
        notifyStatus = true;
        if (shard.lastId() - num > Shard.NOTIFY_DELAY_LATE_THRESHOLD) {
            //we are to late. do not notify on every node
            notifyStatus = (num % Shard.NOTIFY_DELAY_INTERVAL == 0);
        } else {
            long timeDiff = TimeSource.INSTANCE.currentTimeMillis() - shard.previousRead();
            if (timeDiff < Shard.NOTIFY_DELAY_TIME_THRESHOLD) {
                //do not notify on every node if rate to big
                notifyStatus = (num % Shard.NOTIFY_DELAY_INTERVAL == 0);
            }
        }
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("HttpNodeReader.read<" + num + ">: oper=" + operId);
        }
        shard.getZk();
        if (asyncHangGuard != null) {
            asyncHangGuard.cancel(false);
        }
        asyncHangGuard = Delayer.schedule(resetTask, 100000);
        return "/getData3?service=" + shard.service()
            + "&shard=" + shard.shardNo() + "&id=" + num
            + "&count=" + shard.prefetchCount();
    }

    @Override
    public void completed(final HttpResponse response) {
        if (asyncHangGuard != null) {
            asyncHangGuard.cancel(false);
            asyncHangGuard = null;
        }
        synchronized(shard) {
            if (!checkOper()) return;
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("HttpNodeReader.response received<"
                    + currentNum + ">: oper=" + operId);
            }
            shard.updatePreviousRead(TimeSource.INSTANCE.currentTimeMillis());
        }

        int code = response.getStatusLine().getStatusCode();
        if (code != 200) {
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe("HttpNodeReader.read<" + currentNum +
                    ">: http response code: " + code);
            }
            shard.reset();
            return;
        }

        HttpEntity entity = response.getEntity();

        try {
            if (entity == null || entity.getContent() == null) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("HttpNodeReader.read<" + currentNum +
                    ">: http response null entity: " + code);
                }
                shard.reset();
                return;
            }
            decodeResponse(entity.getContent());
        } catch (Exception e) {
            if (logger.isLoggable(Level.SEVERE)) {
                logger.log(Level.SEVERE, "HttpNodeReader.read<" + currentNum +
                    ">: got exception trying to decode response: ", e);
            }
            shard.reset();
        }
    }

    private void decodeResponse(final InputStream resp) throws IOException {
        BinaryInputArchive bia = BinaryInputArchive.getArchive(resp);
        MultiResponse mrsp = new MultiResponse();
        mrsp.deserialize(bia,"mrsp");
        List<Node> nodes = new ArrayList<Node>();
        Iterator<OpResult> iter = mrsp.iterator();
        long num = currentNum;
        while (iter.hasNext()) {
            OpResult op = iter.next();
            String path = shard.genPath(num);
            if (op.getType() == ZooDefs.OpCode.getData) {
                OpResult.GetDataResult gdr = (OpResult.GetDataResult) op;
                boolean notify = notifyStatus;
                if (num % Shard.NOTIFY_DELAY_INTERVAL == 0) notify = true;
                Node node = new Node(path, num, shard, notify, gdr.getData());
                node.msg = SerializeUtils.deserializeHttpMessage(node.data);
                nodes.add(node);
            } else if (op.getType() == ZooDefs.OpCode.error) {
                OpResult.ErrorResult er = (OpResult.ErrorResult) op;
                if (er.getErr() != KeeperException.Code.NONODE.intValue()) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("HttpNodeReader.read<" + currentNum +
                            ": invalid MultiResponse unhandler error type: " +
                            er.getErr());
                    }
                    reset();
                    return;
                }
                synchronized(shard) {
                    if (shard.nextReadInProgress()) {
                        shard.setNextReadInProgress(false);
                        return;
                    }
                }
                if (num > shard.firstId() && num < shard.lastId()) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("HttpNodeReader.read<" + currentNum +
                            ">: Node <" +path+ "> does not exists " +
                            "(ZooKeeper LOOSED seq:  firstId=" + shard.firstId()
                            + ", currentId=" + num + ")");
                    }
                    if (num == currentNum) {
                        //First node in the batch
                        //Try to read node using zoo proto.
                        shard.zooRead();
                        return;
                    } else {
                        //Not the first node in the batch
                        //Consume successfuly readed nodes and then resubmit
                        //request so that the LOOSED node becomes the first.
                        break;
                    }
                } else if (num < shard.firstId()) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("HttpNodeReader.read<" + currentNum +
                            ">: Node <"+path+"> does not exists " +
                            "(MISSED MODE: firstId=" + shard.firstId() +
                            ", currentId=" + num + ")");
                    }
                    if (num == currentNum) {
                        shard.findMissed();
                        return;
                    } else {
                        //same strategy as in the LOOSED block
                        break;
                    }
                } else {
                    if (num == currentNum) {
                        //First oper in multioper
                        //something went wrong
                        //drop results
                        shard.listQueue();
                        return;
                    } else {
                        break; //Stop processing multioper
                    }
                }
            }
            num++;
        }
        if (logger.isLoggable(Level.INFO)) {
            logger.info("HttpNodeReader.read<" + currentNum
                + ">: decoded " + nodes.size() + " nodes");
        }
        if (nodes.size() > 0) {
            boolean runNext = false;
            synchronized(shard) {
                if (shard.nextReadInProgress()) {
                //Previous Dispatch has not been finished yet
//                    logger.info("HttpNodeReader.read<" + currentNum
//                        + ">: nextReadInProgress=true. setting nextNodes");
                    shard.setNextNodes(nodes);
                } else {
//                    logger.info("HttpNodeReader.read<" + currentNum
//                        + ">: nextReadInProgress=false. dispatching nodes");
                    if (nodes.size() == shard.prefetchCount()) {
                    //Batch is full
                        if (logger.isLoggable(Level.INFO)) {
                            logger.info(
                                "HttpNodeReader: prefetching next nodes from: "
                                + num + ", currentNum: " + currentNum);
                        }
                        shard.setNextReadInProgress(true);
                        shard.setNextNodes(null);
                        runNext = true;
                    }
                    shard.dispatchNodes(nodes);
                }
            }
            if (runNext) {
                runNext(num);
            }
        } else {
            synchronized(shard) {
                //no nodes found
                if (shard.nextReadInProgress()) {
                    shard.setNextReadInProgress(false);
                }
            }
        }
    }

    @Override
    public void cancelled() {
        if (asyncHangGuard != null) {
            asyncHangGuard.cancel(false);
            asyncHangGuard = null;
        }
        synchronized(shard) {
            if (!checkOper()) return;
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe("HttpNodeReader.read<" + currentNum +
                    ">: http request canceled");
            }
            shard.nextHost();
            if (shard.nextReadInProgress()) {
                shard.setNextReadInProgress(false);
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("HttpNodeReader.read<" + currentNum +
                        ">: this was a cancelled prefetch read. doing nothing");
                }
            } else {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("HttpNodeReader.read<" + currentNum +
                        ">: this was a cancelled normal read. relisting");
                }
                shard.listQueue();
            }
        }
    }

    @Override
    public void failed(Exception ex) {
        if (asyncHangGuard != null) {
            asyncHangGuard.cancel(false);
            asyncHangGuard = null;
        }
        synchronized(shard) {
            if (!checkOper()) return;
            if (logger.isLoggable(Level.SEVERE)) {
                logger.log(Level.SEVERE, "HttpNodeReader.read<" + currentNum +
                    ">: http request failed: ", ex);
            }
            shard.nextHost();
            if (shard.nextReadInProgress()) {
                shard.setNextReadInProgress(false);
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("HttpNodeReader.read<" + currentNum +
                        ">: this was a failed prefetch read. doing nothing");
                }
            } else {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("HttpNodeReader.read<" + currentNum +
                        ">: this was a failed normal read. relisting");
                }
                shard.listQueue();
            }
        }
    }

    private class ResetTask extends ShardTask {
        public ResetTask(final Shard shard) {
            super(shard);
        }

        @Override
        public boolean checkOper() {
            return HttpNodeReader.this.checkOper();
        }

        @Override
        public void run() {
            shard.logger().severe("HttpNodeReader watchdog expired."
                + "Resetting shard.");
            synchronized(shard) {
                reset();
            }
        }
    }

}
