package ru.yandex.dispatcher.consumer.shard;

import java.util.HashSet;
import java.util.Set;
import java.util.function.LongConsumer;
import java.util.logging.Level;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;

import ru.yandex.dispatcher.consumer.ZooHost;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;

public class NextIdFinder extends ShardTask
    implements FutureCallback<HttpResponse>
{
    private final Set<ZooHost> hosts = new HashSet<>();
    private final int minHostQueryCount;
    private final LongConsumer foundCallback;
    private final Runnable notFoundCallback;
    private ZooHost currentHost;
    private long id;
    private long minId = -100500;

    public NextIdFinder(
        final Shard shard,
        final long operId,
        final int minHostQueryCount)
    {
        this(
            shard,
            operId,
            minHostQueryCount,
            new SetShardPositionCallback(shard),
            new ShardReadCallback(shard));
    }

    public NextIdFinder(
        final Shard shard,
        final long operId,
        final int minHostQueryCount,
        final LongConsumer foundCallback,
        final Runnable notFoundCallback)
    {
        super(shard, operId);
        this.minHostQueryCount = minHostQueryCount;
        this.foundCallback = foundCallback;
        this.notFoundCallback = notFoundCallback;
    }

    @Override
    public void run() {
        synchronized(shard) {
            // no initOperId required
            id = shard.currentPos() + 1;
            shard.getZk();
            final ZooHost zk = shard.currentHost();
            hosts.add(zk);
            readNextId(zk);
        }
    }

    private void readNextId(final ZooHost zk) {
        shard.httpClient().execute(
            zk.httpHost(),
            new BasicAsyncRequestProducerGenerator(
                "/getNextQueueChild?path=" + shard.genPath(id)),
            this);
    }

    private void tryNextHost() {
        synchronized(shard) {
            if (!checkOper()) {
                return;
            }
            if (minHostQueryCount == hosts.size()) {
                //finished server cycling
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("NextIdFinder finished cycling " + minHostQueryCount
                        + " servers: min message id: " + minId);
                }
                if (minId == -100500) {
                    notFoundCallback.run();
                } else {
                    foundCallback.accept(minId);
                }
                return;
            }
            for (int i = 0; i < shard.hostCount(); i++) {
                shard.nextHost();
                shard.getZk();
                final ZooHost zk = shard.currentHost();
                if (hosts.contains(zk)) {
                    if (logger.isLoggable(Level.FINE)) {
                        logger.fine(
                            "Skipping host: " + zk + ", already checked");
                    }
                    continue;
                } else {
                    hosts.add(zk);
                    readNextId(zk);
                    return;
                }
            }
            logger.severe("NextIdFinder: some of the server are " +
                "inavailable, suspending for 30 secs");
            Delayer.schedule(this, 30000);
        }
    }

    @Override
    public void completed(final HttpResponse response) {
        synchronized(shard) {
            if (!checkOper()) return;
        }
        int code = response.getStatusLine().getStatusCode();
        if (code == 404) {
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe("NextIdFinder<" + currentHost +
                    ">: http response code: " + code);
            }
            tryNextHost();
            return;
        }
        if (code != 200) {
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe("NextIdFinder<" + currentHost +
                    ">: http response code: " + code);
            }
            Delayer.schedule(this, 3000);
            return;
        }
        HttpEntity entity = response.getEntity();
        String content = null;

        try {
            if (entity == null || entity.getContent() == null) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("NextIdFinder<" + currentHost +
                    ">: http response null entity: " + code);
                }
                Delayer.schedule(this, 3000);
                return;
            }
            content = EntityUtils.toString(entity);
            String childNo = content.substring(content.lastIndexOf('/')+1+shard.PREFIX_LEN);
            long nextId = Long.parseLong(childNo);
            if (nextId < minId || minId == -100500) minId = nextId;
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("NextIdFinder<" + currentHost +
                    ">: nextId=" + nextId +", minId=" + minId);
            }
            tryNextHost();
        } catch (Exception e) {
            if (logger.isLoggable(Level.SEVERE)) {
                logger.log(Level.SEVERE, "NextIdFinder<" + currentHost +
                    ">: got exception trying to decode response: <" + content +
                    ">: ", e);
            }
            Delayer.schedule(this, 3000);
        }
    }

    @Override
    public void cancelled() {
        synchronized(shard) {
            if (!checkOper()) return;
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe("NextIdFinder<" + currentHost +
                    ">: http request canceled");
            }
            Delayer.schedule(this, 3000);
        }
    }

    @Override
    public void failed(Exception ex) {
        synchronized(shard) {
            if (!checkOper()) return;
            if (logger.isLoggable(Level.SEVERE)) {
                logger.log(Level.SEVERE, "NextIdFinder<" + currentHost +
                    ">: http request failed: ", ex);
            }
            Delayer.schedule(this, 3000);
        }
    }

    public static class SetShardPositionCallback implements LongConsumer {
        private final Shard shard;

        public SetShardPositionCallback(final Shard shard) {
            this.shard = shard;
        }

        @Override
        public void accept(final long pos) {
            shard.setCurrentPos(pos - 1);
            shard.httpRead();
        }
    }

    public static class ShardReadCallback implements Runnable {
        private final Shard shard;

        public ShardReadCallback(final Shard shard) {
            this.shard = shard;
        }

        @Override
        public void run() {
            shard.httpRead();
        }
    }
}
