package ru.yandex.dispatcher.consumer.shard;

import java.util.List;

import java.util.logging.Level;

import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;

public class NextMemoryIdFinder extends ShardTask
    implements AsyncCallback.ChildrenCallback, Watcher
{
    public NextMemoryIdFinder(final Shard shard) {
        super(shard);
    }

    @Override
    public void run() {
        synchronized(shard) {
            initOperId();
            list();
        }
    }

    public void list() {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("NextMemoryIdFinder.list: oper=" + operId);
        }
        ZooKeeper zk = shard.getZk();
        zk.getChildren(shard.queuePath(), this, this, null);
    }

    //ZooKeeper.getChildren handler
    @Override
    public void processResult(final int rc, final String path, final Object ctx,
        final List<String> children)
    {
        synchronized(shard) {
            logger.fine("NextMemoryIdFinder.processResult");
            if (!checkOper()) return;
            if (rc == 0) {
                if (!shard.queuePath().equals(path)) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe(
                            "NextMemoryIdFinder unhandled processResult for "
                            + "path: " + path);
                    }
                    return;
                }
                if (children == null || children.size() == 0) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("NextMemoryIdFinder children == null " +
                            "in processResult for path: " + path +
                            ". Waiting for watch trigger");
                    }
                    return;
                }
                String first = children.get(0).substring(shard.PREFIX_LEN);
                String last =
                    children.get(children.size() - 1).substring(shard.PREFIX_LEN);
                shard.setQueueIds(Long.parseLong(first), Long.parseLong(last));
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("NextMemoryIdFinder firstId: " + shard.firstId()
                        + ", lastId:" + shard.lastId());
                }
                if (shard.currentPos() < shard.lastId()) {
                    shard.setCurrentPos(shard.lastId() - 1);
                    shard.httpRead();
                }
            }
            else if (rc == KeeperException.Code.NONODE.intValue()) {
                logger.severe("NextMemoryIdFinder processResult: Node <"
                    + path + "> does not exists. Waiting");
                shard.waitForQueue();
            }
            else if (rc == KeeperException.Code.CONNECTIONLOSS.intValue()) {
                logger.severe("NextMemoryIdFinder processResult error: " +
                    "connection loss");
                shard.connloss();
            }
            else
            {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("QueueLister processResult unknown error: "
                        + rc);
                }
                //restart request
                run();
            }
        }
    }

    @Override
    public void process(final WatchedEvent event) {
        synchronized(shard) {
            if (!checkOper()) return;
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("NextMemoryIdFinder watched event " + event);
            }
            if (event.getState() == KeeperState.SyncConnected) {
                if (event.getType() ==
                    Watcher.Event.EventType.NodeChildrenChanged)
                {
                    run();
                } else {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("NextMemoryIdFinder unhandled event "
                            + event);
                    }
                    shard.reset();
                }
            } else if (event.getState() == KeeperState.Expired) {
                shard.reset();
            } else if (event.getState() == KeeperState.Disconnected) {
                shard.reset();
            }
        }
    }
}
