package ru.yandex.dispatcher.consumer.shard;

import java.util.List;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

import java.util.logging.Level;

import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;

public class QueueLister extends ShardTask
    implements AsyncCallback.ChildrenCallback, Watcher
{
    private static final long WATCHDOG_DELAY = Shard.WATCHDOG_DELAY >> 1;
    private AtomicReference<ResetTask> resetTask = new AtomicReference<>();
    private Future connChecker = null;

    public QueueLister(final Shard shard) {
        super(shard);
    }

    @Override
    public void run() {
        synchronized(shard) {
            initOperId();
            list();
        }
    }

    public void list() {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("QueueLister.list: oper=" + operId);
        }
        ZooKeeper zk = shard.getZk();
        zk.getChildren(shard.queuePath(), this, this, zk);
        scheduleReset();
    }

    public void scheduleReset() {
        final ResetTask newTask = new ResetTask(shard);
        ResetTask oldTask;
        do {
            oldTask = resetTask.get();
            if (oldTask != null) {
                oldTask.cancel();
            }
        } while (!resetTask.compareAndSet(oldTask, newTask));
        Delayer.schedule(newTask, WATCHDOG_DELAY);
    }

    public boolean cancelReset() {
        final ResetTask task = resetTask.get();
        if (task == null) {
            return true;
        }
        resetTask.compareAndSet(task, null);
        return task.cancel();
    }

    //ZooKeeper.getChildren handler
    @Override
    public void processResult(final int rc, final String path, final Object ctx,
        final List<String> children)
    {
        logger.fine("QueueLister.processResult");
        if (!cancelReset()) {
            logger.fine("Reset watchdog overtook us, discarding result");
            return;
        }
        if (rc == 0) {
            if (!shard.queuePath().equals(path)) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("QueueLister unhandled processResult for " +
                        "path: " + path);
                }
                return;
            }
            if (children == null || children.size() == 0) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("QueueLister children == null " +
                        "in processResult for path: " + path +
                        ". Waiting for watch trigger");
                }
                ZooKeeper zk = (ZooKeeper) ctx;
                connChecker = Delayer.schedule(new ConnectionChecker(zk), 10000);
                return;
            }
            synchronized(shard) {
                if (!checkOper()) return;
                String first = children.get(0).substring(shard.PREFIX_LEN);
                String last =
                    children.get(children.size() - 1).substring(shard.PREFIX_LEN);
                shard.setQueueIds(Long.parseLong(first), Long.parseLong(last));
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("QueueLister firstId: " + shard.firstId() +
                        ", lastId:" + shard.lastId());
                }
                if (connChecker != null) {
                    connChecker.cancel(false);
                    connChecker = null;
                }
            }
            if (shard.currentPos() < shard.lastId()) {
                shard.httpRead();
            } else {
                //waiting for a watch
                //schedule connection checker
                shard.unhalt();
                ZooKeeper zk = (ZooKeeper) ctx;
                if (!zk.getState().isAlive() || !zk.getState().isConnected()) {
                    logger.fine(
                        "FastCheck: connection is dead, watch event "
                        + "will be lost (possibly)");
                }
                if (!zk.childWatchRegistered(
                    shard.queuePath(),
                    QueueLister.this))
                {
                    logger.fine(
                        "FastCheck: connection is alive but watch event is "
                        + "already lost.");
                }
                connChecker = Delayer.schedule(new ConnectionChecker(zk), 10000);
            }
        } else if (rc == KeeperException.Code.NONODE.intValue()) {
            synchronized(shard) {
                if (!checkOper()) return;
            }
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe("QueueLister processResult: Node <" + path +
                    "> does not exists. Waiting");
            }
            shard.waitForQueue();
        } else if (rc == KeeperException.Code.CONNECTIONLOSS.intValue()) {
            synchronized(shard) {
                if (!checkOper()) return;
            }
            logger.severe("QueueLister processResult error: " +
                "connection loss");
            shard.connloss();
        } else {
            synchronized(shard) {
                if (!checkOper()) return;
            }
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe("QueueLister processResult unknown error: " + rc);
            }
            //restart request
            run();
        }
    }

    @Override
    public void process(final WatchedEvent event) {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("QueueLister watched event " + event);
        }
        if (!cancelReset()) {
            logger.fine("Reset watchdog overtook us, discarding event");
            return;
        }
        synchronized(shard) {
            if (connChecker == null) {
                logger.fine(
                    "connChecker=null: ConnectionChecker has already "
                    + "resetted this connection. Skipping watch event");
                return;
            }
            connChecker.cancel(false);
            connChecker = null;
            if (!checkOper()) return;
        }
        if (event.getState() == KeeperState.SyncConnected) {
            if (event.getType() ==
                Watcher.Event.EventType.NodeChildrenChanged) {
                shard.httpRead();
            } else {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("QueueLister unhandled event " + event);
                }
                shard.reset();
            }
        } else if (event.getState() == KeeperState.Expired) {
            shard.reset();
        } else if (event.getState() == KeeperState.Disconnected) {
            shard.reset();
        }
    }

    private class ConnectionChecker implements Runnable {
        private final ZooKeeper zk;

        public ConnectionChecker(final ZooKeeper zk) {
            this.zk = zk;
        }

        @Override
        public void run() {
            synchronized(shard) {
                if (connChecker == null) {
                    return;
                }
                if (!checkOper()) return;
//                logger.fine("QueueLister watcher: checking connection: " + zk);
                if (!zk.getState().isAlive() || !zk.getState().isConnected()) {
                    logger.fine(
                        "connection is dead but watch event is lost. emulating");
                    process(
                        new WatchedEvent(
                            Watcher.Event.EventType.None,
                            KeeperState.Disconnected,
                            "BADBAD"));
                } else {
                    if (!zk.childWatchRegistered(
                        shard.queuePath(),
                        QueueLister.this))
                    {
                        logger.fine(
                            "connection is alive but watch event is lost. "
                            + "emulating");
                        process(
                            new WatchedEvent(
                                Watcher.Event.EventType.None,
                                KeeperState.Disconnected,
                                "BADBAD-ALIVELOST!!!!AAAGGG"));
                    } else {
                        //reschedule
                        connChecker =
                            Delayer.schedule(new ConnectionChecker(zk), 10000);
                    }
                }
            }
        }
    }

    private static class ResetTask extends ShardTask {
        private final AtomicBoolean executed = new AtomicBoolean(false);

        public ResetTask(final Shard shard) {
            super(shard);
        }

        public boolean cancel() {
            return executed.compareAndSet(false, true);
        }

        @Override
        public boolean checkOper() {
            return true;
        }

        @Override
        public void run() {
            if (!executed.compareAndSet(false, true)) {
                return;
            }
            shard.logger().severe(
                "QueueList watchdog timed out. Resetting shard.");
            synchronized(shard) {
                reset();
            }
        }
    }
}
