package ru.yandex.dispatcher.consumer.shard;

import java.util.logging.Level;

import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.data.Stat;

public class QueuePathWaiter extends ShardTask
    implements AsyncCallback.StatCallback, Watcher
{
    public QueuePathWaiter(final Shard shard) {
        super(shard);
    }

    @Override
    public void run() {
        synchronized(shard) {
            initOperId();
            waitFor(shard.queuePath());
        }
    }

    public void waitFor(final String path) {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("QueuePathWaiter.waitFor< " + path + ">: oper="
                + operId);
        }
        ZooKeeper zk = shard.getZk();
        zk.exists(path, this, this, null);
    }

    //ZooKeeper.exists handler
    @Override
    public void processResult(final int rc, final String path, final Object ctx,
        final Stat stat)
    {
        synchronized(shard) {
            if (!checkOper()) return;
            if (rc == 0) {
                if (!shard.queuePath().equals(path)) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("QueuePathWaiter.waitFor<"
                            + shard.queuePath()
                            + ">: unhandled processResult for path: " + path
                            + ": unknown path");
                    }
                    return;
                }
                shard.listQueue();
            } else if (rc == KeeperException.Code.NONODE.intValue()) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("QueuePathWaiter.waitFor<"
                        + shard.queuePath()
                        + ">: processResult: Node <"+path+"> does not exists."
                        + " Waiting for watch trigger");
                }
            } else if (rc == KeeperException.Code.CONNECTIONLOSS.intValue()) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("QueuePathWaiter.waitFor<"
                        + shard.queuePath()
                        + ">: processResult error: connection loss");
                }
                shard.connloss();
            } else {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("QueuePathWaiter.waitFor<"
                        + shard.queuePath()
                        + ">: processResult unknown error: " + rc);
                }
                //restart request
                run();
            }
        }
    }

    @Override
    public void process(final WatchedEvent event) {
        synchronized(shard) {
            if (!checkOper()) return;
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("QueuePathWaiter.waitFor<" + shard.queuePath() +
                    ">: watched event " + event);
            }
            if (event.getState() == KeeperState.SyncConnected) {
                if (event.getType() == Watcher.Event.EventType.NodeCreated) {
                    shard.listQueue();
                } else {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("QueuePathWaiter.waitFor< " +
                            shard.queuePath() + ">: unhandled event " + event);
                    }
                }
            } else if (event.getState() == KeeperState.Expired) {
                shard.reset();
            }
            else if (event.getState() == KeeperState.Disconnected) {
                shard.reset();
            }
        }
    }
}
