package ru.yandex.dispatcher.consumer.shard;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.LongConsumer;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.zookeeper.ZooKeeper;

import ru.yandex.dispatcher.common.mappedvars.UnparkCallback;
import ru.yandex.dispatcher.consumer.BackendConsumer;
import ru.yandex.dispatcher.consumer.Node;
import ru.yandex.dispatcher.consumer.ShardsPositions;
import ru.yandex.dispatcher.consumer.StatusQueue;
import ru.yandex.dispatcher.consumer.ZooHost;
import ru.yandex.dispatcher.consumer.ZooKeeperConnection;
import ru.yandex.dispatcher.consumer.ZooKeeperConsumer;
import ru.yandex.dispatcher.consumer.ZooKeeperDSN;
import ru.yandex.dispatcher.consumer.ZooKeeperPool;
import ru.yandex.dispatcher.consumer.lock.QueueLock;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.util.timesource.TimeSource;

public class Shard //implements AsyncCallback.StatCallback, AsyncCallback.ChildrenCallback, AsyncCallback.DataCallback, AsyncCallback.VoidCallback, Watcher
{
    public static final long WATCHDOG_DELAY = 300 * 1000;
    public static final long READ_STATUS_DELAY = 20 * 1000;
    private static final long MIN_SWITCH_INTERVAL = 60 * 1000;
    public final int shardNo;
    private final ZooKeeperConsumer zc;
    private final AtomicBoolean parked = new AtomicBoolean(false);
    private final PrefixedLogger logger;
    private final String service;
    private final ZooKeeperPool zooPool;
    private final ZooKeeperDSN dsn;
    private final BackendConsumer consumer;
    private final AsyncClient httpClient;
    private final StatusQueue statusQueue;
    private final int advanceOnMissed;
    private final boolean ignoreZooPosition;
    private final boolean responseLess;
    private final long sleepInterval;
    private final QueueLock queueLock;
    private final int prefetchCount;
    private final ShardsPositions shardsPositions;
    private final String shardPath;
    private final String queuePath;
    private final String msgPathStart;
    private final String statusPath;
    private final StatusNodeReader statusNodeReader;
    private final StatusNodeWaiter statusNodeWaiter;
    private final QueueLister queueLister;
    private final QueuePathWaiter queuePathWaiter;
    private final HttpNodeReader httpNodeReader;
    private final ZooNodeReader zooNodeReader;
    private final int minNextIdFinderHostCount;
    public ZooKeeperConnection zkc;
    private ZooHost zooHost;
    private long nextAllowedServerSwitch = 0;
    private Future watchdogFuture = null;

    private long currentPos = 0;
    private long lastId;
    private long firstId;
    private volatile long operId = 0;
    private long backendRedPosition = Long.MIN_VALUE;

    private long previousRead;
    private final long watchdogDelay;
    private final long readStatusDelay;

    private volatile boolean nextReadInProgress = false;
    private volatile boolean consumeInProgress = false;
    private volatile boolean shouldReset = false;
    private volatile boolean shouldResetBatch = false;
    private volatile long    commitedBatchPos = 0;
    private volatile boolean batchStarted = false;
    private List<Node> nextNodes = null;
    private List<Node> nodesInConsume = null;
    private long lastConsumerActivity = 0;
    private int missed = 0;
    private boolean halted = false;

    private static final String prefix = "queue";
    public static final int PREFIX_LEN = prefix.length();
    public static final long NOTIFY_DELAY_LATE_THRESHOLD = 100;
    public static final long NOTIFY_DELAY_INTERVAL = 100;
    public static final long NOTIFY_DELAY_TIME_THRESHOLD = 50;

    public Shard(
        final ZooKeeperConsumer zc,
        final PrefixedLogger logger,
        final String service,
        final int num,
        final ZooKeeperPool zooPool,
        final ZooKeeperDSN dsn,
        final BackendConsumer consumer,
        final AsyncClient httpClient,
        final StatusQueue statusQueue,
        final int advanceOnMissed,
        final boolean ignoreZooPosition,
        final boolean responseLess,
        final long sleepInterval,
        final QueueLock queueLock,
        final int prefetchCount,
        final int nextIdFinderMinHostCountPct,
        final long watchdogDelay,
        final long readStatusDelay)
        throws IOException
    {
        shardNo = num;
        this.zc = zc;
        this.logger = logger;
        this.service = service;
        this.zooPool = zooPool;
        this.dsn = dsn;
        this.consumer = consumer;
        this.httpClient = httpClient;
        this.statusQueue = statusQueue;
        this.advanceOnMissed = advanceOnMissed;
        this.ignoreZooPosition = ignoreZooPosition;
        this.responseLess = responseLess;
        this.sleepInterval = sleepInterval;
        this.queueLock = queueLock;
        this.prefetchCount = prefetchCount;
        this.watchdogDelay = watchdogDelay;
        this.readStatusDelay = readStatusDelay;

        this.minNextIdFinderHostCount =
            Math.min(
                (dsn.hostCount() * nextIdFinderMinHostCountPct) / 100 + 1,
                dsn.hostCount());

        if (logger.isLoggable(Level.FINE)) {
            logger.fine("minNextIdFinderHostCount: "
                + minNextIdFinderHostCount);
            logger.fine("Watchdog delay: " + watchdogDelay);
        }

        shardsPositions = consumer.getPositions(service);
        shardPath = '/' + service + '/' + shardNo;
        queuePath = '/' + service + '/' + shardNo + "/forward";
        msgPathStart = queuePath + '/' + prefix;
        statusPath =
            '/' + service + '/' + shardNo + "/status/"
            + statusQueue.consumerName();

        statusNodeReader = new StatusNodeReader(this);
        statusNodeWaiter = new StatusNodeWaiter(this);
        queueLister = new QueueLister(this);
        queuePathWaiter = new QueuePathWaiter(this);
        httpNodeReader = new HttpNodeReader(this);
        zooNodeReader = new ZooNodeReader(this);
    }

    public String service() {
        return service;
    }

    public int shardNo() {
        return shardNo;
    }

    public PrefixedLogger logger() {
        return logger;
    }

    public long currentOperId() {
        return operId;
    }

    public long nextOperId() {
        return ++operId;
    }

    public String statusPath() {
        return statusPath;
    }

    public String queuePath() {
        return queuePath;
    }

    public long commitedBatchPos() {
        return commitedBatchPos;
    }

    public void setCommitedBatchPos(final long pos) {
        commitedBatchPos = pos;
    }

    public long currentPos() {
        return currentPos;
    }

    public void statusNodePosition(final long pos) {
        if (shardsPositions != null
            && ignoreZooPosition
            && pos != backendRedPosition
            && backendRedPosition != Long.MIN_VALUE)
        {
            if (logger.isLoggable(Level.INFO)) {
                logger.info("Updating zoo status from backend position: "
                    + currentPos);
            }
            delayNodeStatus(
                new Node(null, currentPos, this, false, null));
        } else {
            setCurrentPos(pos);
            setCommitedBatchPos(pos);
        }
    }

    public void setCurrentPos(final long pos) {
        currentPos = pos;
    }

    public boolean batchStarted() {
        return batchStarted;
    }

    public void setBatchStarted(final boolean b) {
        this.batchStarted = b;
    }

    public long firstId() {
        return firstId;
    }

    public long lastId() {
        return lastId;
    }

    public int prefetchCount() {
        return prefetchCount;
    }

    public void setQueueIds(final long firstId, final long lastId) {
        this.firstId = firstId;
        this.lastId = lastId;
    }

    public boolean isSomewhereCurrent() {
        return currentPos >= firstId;
    }

    public long previousRead() {
        return previousRead;
    }

    public void updatePreviousRead(final long pr) {
        previousRead = pr;
    }

    public boolean nextReadInProgress() {
        return nextReadInProgress;
    }

    public void setNextReadInProgress(final boolean nextReadInProgress) {
        this.nextReadInProgress = nextReadInProgress;
    }

    public void setNextNodes(final List<Node> nodes) {
        nextNodes = nodes;
    }

    private void scheduleWatchdog() {
        if (watchdogFuture != null) {
            watchdogFuture.cancel(false);
        }
        watchdogFuture =
            Delayer.schedule(new ResetTask("Watchdog"), watchdogDelay);
    }

    public boolean checkLock() {
        scheduleWatchdog();
        if (parked.get()) {
            logger.info("CheckLock: already parked");
            return false;
        }
        if (queueLock.tryLock(shardNo)) {
            logger.info("CheckLock: lock success");
            return true;
        }
        logger.info("CheckLock: parking");
        if (!parked.compareAndSet(false, true)) {
            logger.info("CheckLock: already parked");
            return false;
        }
        consumeInProgress = false;
        queueLock.park(
            shardNo,
            new UnparkCallback() {
                @Override
                public void unpark() {
                    parked.set(false);
                    logger.info("CheckLock.unparking");
                    reset();
                }
            });
        logger.info("CheckLock: parked");
        return false;
    }

    @Override
    public final int hashCode()
    {
	return shardNo + service.hashCode();
    }

    @Override
    public final boolean equals( Object o )
    {
	Shard other = (Shard)o;
	return shardNo == other.shardNo && service.equals(other.service);
    }

    public void writeNodeStatus(Node node) {
        if (!node.notifyStatus) {
            delayNodeStatus(node);
        } else {
            statusQueue.writeNodeStatus(node);
        }
    }

    public void delayNodeStatus( Node node ) {
        statusQueue.delayNodeStatus(node);
    }

    public void writeNodeError( Node node, String errorString, int code ) {
        if (!responseLess) {
            statusQueue.writeNodeError(node, errorString, code);
        }
    }

    public synchronized ZooKeeper getZk() {
        if( zkc != null && !zkc.isConnected() )
        {
            zkc.close();
            zkc = null;
        }
        if (zooHost == null) {
            final ZooHost lessLoadedHost =
                zooPool.getLessLoadedHost(dsn.hosts());
            nextAllowedServerSwitch =
                    TimeSource.INSTANCE.currentTimeMillis() + MIN_SWITCH_INTERVAL;
            if (logger.isLoggable(Level.INFO)) {
                logger.info("Trying less loaded zoolooser server: "
                    + lessLoadedHost + "/load:"
                    + zooPool.load(lessLoadedHost)
                    + ", leader: "
                    + zooPool.leader(lessLoadedHost));
            }
            dsn.setCurrentHost(lessLoadedHost);
            zooHost = lessLoadedHost;
        } else {
            zooHost = dsn.currentHost();
        }
        int t = 0;
        if (nextAllowedServerSwitch < TimeSource.INSTANCE.currentTimeMillis()) {
            if (zkc != null && zkc.isConnected()) {
                if (isSomewhereCurrent()) {
                    final ZooHost lessLoadedHost =
                        zooPool.getLessLoadedHost(dsn.hosts());
                    final long currentLoad = zooPool.load(zooHost);
                    final long lessLoad = zooPool.load(lessLoadedHost);
                    final double ratio =
                        (double) Math.min(currentLoad, lessLoad)
                        / Math.max(1.0, (double) Math.max(currentLoad, lessLoad));
                    if (!lessLoadedHost.equals(zooHost) && ratio < 0.85) {
                        nextAllowedServerSwitch =
                            TimeSource.INSTANCE.currentTimeMillis() + MIN_SWITCH_INTERVAL;
                        if (logger.isLoggable(Level.INFO)) {
                            logger.info(
                                "Switching to less loaded zoolooser server: "
                                + zooHost + "/load:" + zooPool.load(zooHost)
                                + " -> " + lessLoadedHost + "/load:"
                                + zooPool.load(lessLoadedHost));
                        }
                        dsn.setCurrentHost(lessLoadedHost);
                        zooHost = lessLoadedHost;
                        zkc.close();
                        zkc = null;
                    }
                }
            }
        }
        while (zkc == null) {
            if (t >= dsn.hostCount()) {
                t = 0;
                try {
                    Thread.sleep(sleepInterval);
                } catch (InterruptedException e) {
                }
            }
            if (zooHost == null) {
                zooHost = dsn.currentHost();
            }
            t++;
            zkc = zooPool.getConnection(zooHost, logger);
            if (logger.isLoggable(Level.INFO)) {
                logger.info("connecting to: " + zooHost);
            }
            if (zkc == null) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("getZk(): can't connect to: " + zooHost);
                }
                zooHost = dsn.nextHost();
            } else if (!zkc.isConnected()) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("getZk(): can't connect to: " + zooHost);
                }
                zkc.close();
                zkc = null;
                zooHost = dsn.nextHost();
            }
        }
        return zkc.zk;
    }

    public ZooHost currentHost() {
        return zooHost;
    }

    public int hostCount() {
        return dsn.hostCount();
    }

    public AsyncClient httpClient() {
        return httpClient;
    }

    public synchronized void resetBatch() {
        batchStarted = false;
        if (consumeInProgress) {
            shouldResetBatch = true;
        } else {
            shouldResetBatch = false;
            nextReadInProgress = false;
            nextNodes = null;
            currentPos = commitedBatchPos;
            if (!checkLock()) return;
            httpRead();
        }
    }

    public synchronized void reset()
    {
        nextOperId();
        batchStarted = false;
        if (consumeInProgress) {
            shouldReset = true;
        } else {
            shouldReset = false;
            shouldResetBatch = false;
            batchStarted = false;
            nextReadInProgress = false;
            nextNodes = null;
            if (!checkLock()) return;
            readStatus();
        }
    }

    public synchronized void hardReset() {
        nodesInConsume = null;
        consumeInProgress = false;
        reset();
    }

    public int minNextIdFinderHostCount() {
        return minNextIdFinderHostCount;
    }

    public boolean shouldReset() {
        return shouldReset;
    }

    public void readStatus()
    {
        if (!checkLock()) return;
        if (!readStatusFromConsumer()) {
            if(ignoreZooPosition) {
                logger.info("readStatus: can't read shard position from " +
                    "backend while \"ignore zookeeper position\" " + 
                    "flag is set. Delaying shard reset");
                Delayer.schedule(new ResetTask("ReadStatus delay"), readStatusDelay);
            } else {
                readStatusFromZoolooser();
            }
        }
    }

    public void listQueue() {
        if (!checkLock()) return;
        queueLister.run();
    }

    public void waitForQueue() {
        if (!checkLock()) return;
        queuePathWaiter.run();
    }

    public void waitForStatus() {
        if (!checkLock()) return;
        statusNodeWaiter.run();
    }

    public void httpRead() {
        if (!checkLock()) return;
        httpNodeReader.run();
    }

    public void zooRead() {
        if (!checkLock()) return;
        zooNodeReader.run();
    }

    public boolean consumerExpired(final List<Node> nodes) {
        lastConsumerActivity = TimeSource.INSTANCE.currentTimeMillis();
        return nodes != nodesInConsume;
    }

    public synchronized void dispatchNodes(final List<Node> nodes) {
        unhalt();
        if (!checkLock()) return;
        consumeInProgress = true;
        nodesInConsume = nodes;
        consumer.dispatchNodes(nodes, this);
    }

    public synchronized void findMissed() {
        if (!checkLock()) return;
        ShardTask task;
        if (advanceOnMissed == 100500) {
            task = new NextMemoryIdFinder(this);
        } else if (advanceOnMissed == 666) {
            task = new ZooNodeFinder(this, currentPos + 1,
                new ShardTask(this) {
                    @Override
                    public void run() {
                        currentPos++;
                        httpRead();
                    }
                });
        } else if (advanceOnMissed > 0) {
            long operId = nextOperId();
            task = new ZooNodeFinder(
                this,
                currentPos + 1,
                operId,
                new NextIdFinder(this, operId, minNextIdFinderHostCount));
        } else {
            long operId = nextOperId();
            task = new ZooNodeFinder(
                this,
                currentPos + 1,
                operId,
                new NextIdFinder(
                    this,
                    operId,
                    minNextIdFinderHostCount,
                    new TryFindCurrentPos(this, currentPos, operId),
                    () -> {}));
        }
        task.run();
    }

    public void findLoosed() {
        if (!checkLock()) return;
        new ZooNodeFinder(this, currentPos + 1, new MessageSkipper(this)).run();
    }

    private void readStatusFromZoolooser() {
        statusNodeReader.run();
    }

    private boolean readStatusFromConsumer() {
        if (shardsPositions == null) {
            return false;
        }

        logger.fine("BackendConsumer has support for shards positions reading, "
            + "trying to read position from backend");

        long pos = -1;
        try {
            pos = shardsPositions.getPosition(shardNo);
        } catch(Exception e) {
            logger.log(Level.SEVERE, "readStatus: got IOException while trying to read shard position: ", e);
            return false;
        }
        if (logger.isLoggable(Level.INFO)) {
            logger.info("Got position from backend: " + pos);
        }
        synchronized(Shard.this) {
            backendRedPosition = pos;
            currentPos = pos;
            commitedBatchPos = pos;
            readStatusFromZoolooser();
        }
        return true;
    }

    public synchronized void nextHost()
    {
        logger.info("trying next server" );
        if (zkc != null) {
            zkc.close();
            zkc = null;
        }
        zooHost = dsn.nextHost();
    }

    public synchronized void connloss() {
        if (zkc != null) {
            zkc.close();
            zkc = null;
        }
        reset();
    }

    public synchronized void halt() {
        halted = true;
        zc.shardHalted(this);
    }

    public synchronized void unhalt() {
        if (halted) {
            halted = false;
            zc.shardUnhalted(this);
        }
    }

    public synchronized void hang(Node node) {
        logger.fine("Hanging hard by user request");
        writeNodeStatus(node); //force node status update
        if (shardsPositions != null) {
            shardsPositions.updatePosition(shardNo, node.seq);
        }
        nextOperId(); //This will effectively fail all flying requests
        nextNodes = null;
        nextReadInProgress = false;
        consumeInProgress = false;
        nodesInConsume = null;
        shouldReset = false;
        shouldResetBatch = false;
        batchStarted = false;
    }

    public synchronized void finishReset() {
        consumeInProgress = false;
        nodesInConsume = null;
        if (shouldReset) {
            reset();
            return;
        }
    }


    public synchronized void nodeFinished(
        final List<Node> nodesFinished,
        final Node node)
    {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("nodeFinished: " + node.seq + ", nextInProgress: " +
                nextReadInProgress + ", nextNodes: " + nextNodes + 
                ", shouldReset: " + shouldReset + ", shouldResetBatch: "
                + shouldResetBatch);
        }
        if (nodesInConsume != nodesFinished) {
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe(
                    "BackendConsumer has finished processing expired node "
                        + "list: " + nodesFinished + ", skipping");
            }
            return;
        }
        nodesInConsume = null;
        consumeInProgress = false;
        if (shouldReset) {
            reset();
            return;
        }
        if (shouldResetBatch) {
            resetBatch();
            return;
        }
        currentPos = node.seq;
        if (!checkLock()) return;
        if (nextReadInProgress) {
            if (nextNodes != null) {
                //next batch has been already red
                final long nextBatchStart =
                    nextNodes.get(nextNodes.size() - 1).seq + 1;
                dispatchNodes(nextNodes);
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("nodeFinished: nextBatchStart: "
                        + nextBatchStart);
                }
                httpNodeReader.runNext(nextBatchStart);
                nextNodes = null;
                return;
            }
            else
            {
                nextReadInProgress = false;
            }
        }
        else
        {
            httpNodeReader.run();
        }
    }

    public String genPath(long num) {
        return msgPathStart + intToStringPadded(num);
    }

    @Override
    public String toString() {
        return service + '/' + shardNo + '/' + statusQueue.consumerName();
    }

    private final static char[] zeros =
        {'0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0',
        '0','0','0'};
    public final static String intToStringPadded(final long i) {
        StringBuilder intStr = new StringBuilder(20);
        intStr.append(i);
        intStr.insert(0, zeros, 0, 20 - intStr.length() );
        return intStr.toString();
    }

    private class ResetTask extends ShardTask {
        private final String cause;
        public ResetTask(final String cause) {
            super(Shard.this);
            this.cause = cause;
        }

        @Override
        public boolean checkOper() {
            return true;
        }

        @Override
        public void run() {
            if (logger().isLoggable(Level.INFO)) {
                logger().info(
                    cause + ": Reset delay has been reached. Resetting shard.");
            }
            synchronized(shard) {
                boolean reset = false;
                if (consumeInProgress) {
                    if ((TimeSource.INSTANCE.currentTimeMillis() - lastConsumerActivity)
                        > consumer.maxNodeTimeout())
                    {
                        reset = true;
                    } else {
                        if (logger().isLoggable(Level.INFO)) {
                            logger().info(
                                cause
                                + ": Consume in progress. Delaying watchdog.");
                        }
                        //we assuming that message execution time can be very long
                        watchdogFuture = null;
                        scheduleWatchdog();
                    }
                } else {
                    reset = true;
                }
                if (reset) {
                    if (!queueLister.checkOper()) {
                        reset();
                    } else {
                        logger().info(
                            "The last op was from QueueLister. "
                            + "Trying to relist");
                        listQueue();
                    }
                }
            }
        }
    }

    private static class TryFindCurrentPos implements LongConsumer {
        private final Shard shard;
        private final long currentPos;
        private final long operId;

        TryFindCurrentPos(
            final Shard shard,
            final long currentPos,
            final long operId)
        {
            this.shard = shard;
            this.currentPos = currentPos;
            this.operId = operId;
        }

        @Override
        public void accept(final long pos) {
            HaltTask haltTask = new HaltTask(shard, currentPos);
            if (currentPos < 0) {
                haltTask.run();
            } else {
                new ZooNodeFinder(
                    shard,
                    currentPos,
                    operId,
                    new SetShardPositionCallback(shard, pos),
                    haltTask)
                    .run();
            }
        }
    }

    private static class SetShardPositionCallback implements Consumer<Node> {
        private final Shard shard;
        private final long pos;

        SetShardPositionCallback(final Shard shard, final long pos) {
            this.shard = shard;
            this.pos = pos;
        }

        @Override
        public void accept(final Node node) {
            shard.setCurrentPos(pos - 1);
            shard.httpRead();
        }
    }

    private static class HaltTask implements Runnable {
        private final Shard shard;
        private final long currentPos;

        HaltTask(
            final Shard shard,
            final long currentPos)
        {
            this.shard = shard;
            this.currentPos = currentPos;
        }

        @Override
        public synchronized void run() {
            Logger logger = shard.logger();
            if (logger.isLoggable(Level.SEVERE)) {
                logger.severe(
                    "Can't find nodes " + currentPos
                    + " and " + (currentPos + 1)
                    + " on any queue server. HALTING");
            }
            shard.halt();
        }
    }
}
