package ru.yandex.dispatcher.consumer.shard;

import java.util.logging.Level;

import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.data.Stat;

public class StatusNodeReader extends ShardTask
    implements AsyncCallback.DataCallback, Watcher
{
    public StatusNodeReader(final Shard shard) {
        super(shard);
    }

    @Override
    public void run() {
        synchronized(shard) {
            initOperId();
            read(shard.statusPath());
        }
    }

    public void read(final String path) {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("StatusNodeReader.read: oper=" + operId);
        }
        ZooKeeper zk = shard.getZk();
        zk.getData(path, false, this, null);
    }

    //GetData handler
    @Override
    public void processResult(final int rc, final String path, final Object ctx,
        final byte[] data, final Stat stat)
    {
        synchronized(shard) {
            if (!checkOper()) return;
            if (rc == 0) {
                if (!shard.statusPath().equals(path)) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("StatusNodeReader.read<" +
                            shard.statusPath() + ">: " + 
                            "unhandled processResult for path: " + path);
                    }
                    return;
                }
                if (data == null) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("StatusNodeReader.read<" +
                            shard.statusPath() + ">: data == null " +
                            "in processResult for path: " + path);
                    }
                    return;
                }
                long pos = 0;
                String dataStr = new String(data);
                try {
                    pos = Long.parseLong(dataStr);
                }
                catch (java.lang.NumberFormatException e) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("StatusNodeReader.read<" + 
                            shard.statusPath() + ">: processResult " +
                            "can't parse long from \"" + dataStr +
                            "\" for path: " + path);
                    }
                    return;
                }
                shard.statusNodePosition(pos);
                shard.listQueue();
            } else if (rc == KeeperException.Code.NONODE.intValue()) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("StatusNodeReader processResult: Node <"
                        + path +
                        "> does not exists. Waiting");
                }
                shard.waitForStatus();
            } else if (rc == KeeperException.Code.CONNECTIONLOSS.intValue()) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("StatusNodeReader processResult error: " +
                        "connection loss");
                }
                shard.connloss();
            } else {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe(
                        "StatusNodeReader processResult unknown error: " + rc);
                }
                run();
            }
        }
    }

    @Override
    public void process(final WatchedEvent event) {
        synchronized(shard) {
            if (!checkOper()) return;
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("StatusNodeReader.read<" + shard.statusPath() +
                    ">: watched event " + event);
            }
            if (event.getState() == KeeperState.SyncConnected) {
                if (event.getType()
                    == Watcher.Event.EventType.NodeDataChanged)
                {
                    read (event.getPath());
                } else {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("StatusNodeReader unhandled event "
                            + event);
                    }
                }
            } else if (event.getState() == KeeperState.Expired) {
                run();
            } else if (event.getState() == KeeperState.Disconnected) {
                run();
            }
        }
    }
}
