package ru.yandex.dispatcher.consumer.shard;

import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.data.Stat;

import ru.yandex.dispatcher.consumer.Node;
import ru.yandex.dispatcher.consumer.ZooHost;
import ru.yandex.util.timesource.TimeSource;

public class ZooNodeFinder extends ShardTask
    implements AsyncCallback.DataCallback, Watcher
{
    private final Set<ZooHost> hosts = new HashSet<>();
    private final long num;
    private final Consumer<Node> foundCallback;
    private final Runnable notFoundCallback;
    private final boolean operIdInitialized;
    private final int minHostCount;

    public ZooNodeFinder(
        final Shard shard,
        final long num,
        final Runnable notFoundCallback)
    {
        this(shard, num, 0L, notFoundCallback);
    }

    public ZooNodeFinder(
        final Shard shard,
        final long num,
        final long operId,
        final Runnable notFoundCallback)
    {
        this(
            shard,
            num,
            operId,
            new DispatchNodeCallback(shard),
            notFoundCallback);
    }

    public ZooNodeFinder(
        final Shard shard,
        final long num,
        final long operId,
        final Consumer<Node> foundCallback,
        final Runnable notFoundCallback)
    {
        super(shard, operId);
        this.num = num;
        this.foundCallback = foundCallback;
        this.notFoundCallback = notFoundCallback;
        minHostCount = shard.minNextIdFinderHostCount();
        operIdInitialized = operId != 0L;
    }

    @Override
    public void run() {
        synchronized(shard) {
            if (!operIdInitialized) {
                initOperId();
            }
            ZooKeeper zk = shard.getZk();
            hosts.add(shard.currentHost());
            read(num, zk);
        }
    }

    private void read(final long num, ZooKeeper zk) {
        Node node = new Node(shard.genPath(num), num, shard, true, null);
        read(node, zk);
    }

    private void read(Node node, ZooKeeper zk) {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("ZooNodeFinder.read<"+ node.seq +">: oper=" + operId);
        }
        zk.getData(node.path, false, this, node);
    }

    private void tryNextHost() {
        if (minHostCount == hosts.size()) {
            if (logger.isLoggable(Level.INFO)) {
                logger.info("ZooNodeFinder: processed minNextIdFinderHostCount = "
                    + minHostCount
                    + " hosts. Assuming node does not exists.");
            }
            //all servers are processed and no node found
            notFoundCallback.run();
            return;
        }
        for(int i = 0; i < shard.hostCount(); i++) {
            shard.nextHost();
            ZooKeeper zk = shard.getZk();
            ZooHost host = shard.currentHost();
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("ZooNodeFinder: got host: " + host);
            }
            if (hosts.contains(host)) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("ZooNodeFinder: skipping host: " + host
                        + ", already checked");
                }
                continue;
            } else {
                hosts.add(shard.currentHost());
                read(num, zk);
                return;
            }
        }
        //Some of servers are not reachable
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("ZooNodeFinder: no minNextIdFinderHostCount = " + minHostCount
                + " was available, sleeping for retry");
        }
        Delayer.schedule(this, 3000);
    }

    //ZooKeeper.getData handler
    @Override
    public void processResult(final int rc, final String path, final Object ctx,
        final byte[] data, final Stat stat)
    {
        Node node = (Node)ctx;
        long num = node.seq;
        synchronized(shard) {
            if (!checkOper()) return;
            if (rc == 0) {
                shard.updatePreviousRead(TimeSource.INSTANCE.currentTimeMillis());
                if (data == null) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("ZooNodeFinder.read<" + num + 
                            ">: data == null in processResult for path: "
                            + path);
                    }
                    shard.reset();
                    return;
                }
                node.data = data;
                foundCallback.accept(node);
            } else if (rc == KeeperException.Code.NONODE.intValue()) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("ZooNodeFinder.read<" + num + ">: Node <" +
                        path + "> does not exists on server " +
                        shard.currentHost());
                }
                tryNextHost();
                return;
            } else {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("ZooNodeFinder.read<" + num +
                        ">: processResult unknown error: " + rc);
                }
                //restart operation
                Delayer.schedule(this, 3000);
            }
        }
    }

    @Override
    public void process(WatchedEvent event) {
        synchronized(shard) {
            if (!checkOper()) return;
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("ZooNodeFinder.read<" + num +
                    ">: watched event " + event);
            }
            if (event.getState() == KeeperState.SyncConnected) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("ZooNodeFinder unhandled event " + event);
                }
            } else if (event.getState() == KeeperState.Expired) {
                Delayer.schedule(this, 3000);
            } else if (event.getState() == KeeperState.Disconnected) {
                Delayer.schedule(this, 3000);
            }
        }
    }

    public static class DispatchNodeCallback implements Consumer<Node> {
        private final Shard shard;

        DispatchNodeCallback(final Shard shard) {
            this.shard = shard;
        }

        @Override
        public void accept(final Node node) {
            //DO NOT: use Collections.singletonList, List must support
            //remove() operation
            List<Node> nodes = new LinkedList<Node>();
            nodes.add(node);
            shard.dispatchNodes(nodes);
        }
    }
}
