package ru.yandex.dispatcher.consumer.shard;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;

import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.data.Stat;

import ru.yandex.dispatcher.consumer.Node;
import ru.yandex.util.timesource.TimeSource;

public class ZooNodeReader extends ShardTask
    implements AsyncCallback.DataCallback, Watcher
{
    private long currentNum;

    public ZooNodeReader(final Shard shard) {
        super(shard);
    }

    @Override
    public void run() {
        synchronized(shard) {
            initOperId();
            read(shard.currentPos() + 1);
        }
    }

    public void read(Node node) {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("ZooNodeReader.read<"+ node.seq +">: oper=" + operId);
        }
        ZooKeeper zk = shard.getZk();
        zk.getData(node.path, false, this, node);
    }

    public void read(final long num) {
        boolean notifyStatus = true;
        if (shard.lastId() - num > Shard.NOTIFY_DELAY_LATE_THRESHOLD) {
            //we are to late. do not notify on every node
            notifyStatus = (num % Shard.NOTIFY_DELAY_INTERVAL == 0);
        } else {
            long timeDiff = TimeSource.INSTANCE.currentTimeMillis() - shard.previousRead();
            if (timeDiff < Shard.NOTIFY_DELAY_TIME_THRESHOLD) {
                //do not notify on every node if rate to big
                notifyStatus = (num % Shard.NOTIFY_DELAY_INTERVAL == 0);
            }
        }
        Node node =
            new Node(shard.genPath(num), num, shard, notifyStatus, null);
        currentNum = num;
        read(node);
    }

    //ZooKeeper.getData handler
    @Override
    public void processResult(final int rc, final String path, final Object ctx,
        final byte[] data, final Stat stat)
    {
        Node node = (Node)ctx;
        long num = node.seq;
        synchronized(shard) {
            if (!checkOper()) return;
        }
        {
            if (rc == 0) {
                shard.updatePreviousRead(TimeSource.INSTANCE.currentTimeMillis());
                if (data == null) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("ZooNodeReader.read<" + num + 
                            ">: data == null in processResult for path: "
                            + path);
                    }
                    shard.setCurrentPos(num + 1);
                    run();
                    return;
                }
                node.data = data;
                //DO NOT: use Collections.singletonList, List must support
                //remove() operation
                List<Node> nodes = new LinkedList<Node>();
                nodes.add(node);
                shard.dispatchNodes(nodes);
            }
            else if (rc == KeeperException.Code.NONODE.intValue()) {
                if (num < shard.firstId()) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("ZooNodeReader.read<" + num + ">: Node <"
                            + path + "> does not exists (MISSED MODE: firstId="
                            + shard.firstId() + ", currentId=" + num + ")");
                    }
                    shard.findMissed();
                } else if( num > shard.firstId() && num < shard.lastId() ) {
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe("ZooNodeReader.read<" + num + ">: Node <"
                            + path + "> does not exists (LOOSED seq:  firstId="
                            + shard.firstId() + ", currentId=" + num + ")");
                    }
//                    read(node.seq + 1);
                    shard.findLoosed();
                } else {
                    shard.listQueue();
                }
            } else {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe("ZooNodeReader.read<" + num +
                        ">: processResult unknown error: " + rc);
                }
                //restart operation
                run();
            }
        }
    }

    @Override
    public void process(WatchedEvent event) {
        synchronized(shard) {
            if (!checkOper()) return;
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("ZooNodeReader.read<" + currentNum +
                    ">: watched event " + event);
            }
            if (event.getState() == KeeperState.SyncConnected) {
                logger.severe("ZooNodeReader unhandled event " + event);
            } else if (event.getState() == KeeperState.Expired) {
                shard.reset();
            } else if (event.getState() == KeeperState.Disconnected) {
                reset();
            }
        }
    }
}
