package ru.yandex.dispatcher.producer;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import ru.yandex.http.config.HttpTargetConfig;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;

public abstract class AbstractProducerConfigBuilder
    <T extends AbstractProducerConfigBuilder<T>>
    extends AbstractHttpProxyConfigBuilder<T>
    implements ProducerConfig
{
    private String searchMapPath = "/var/cache/yamail/searchmap.txt";
    private int syncHelperThreads;
    private int zookeeperTimeout;
    private int waitTimeout;
    private int maxGroupSize;
    private int maxGroupWeight;
    private int maxGroupDelay;
    private int maxStatusGroupSize;
    private int maxStatusGroupWeight;
    private int maxStatusGroupDelay;
    private Set<String> services;
    private Set<String> consumerServices;
    private HttpTargetConfigBuilder zooHttpTargetConfig;
    private int statusProxyFallbackDelay;
    private List<Integer> staterTimeBoundaries;

    protected AbstractProducerConfigBuilder(final ProducerConfig config) {
        super(config);
        searchMapPath(config.searchMapPath());
        syncHelperThreads(config.syncHelperThreads());
        zookeeperTimeout(config.zookeeperTimeout());
        waitTimeout(config.waitTimeout());
        maxGroupSize(config.maxGroupSize());
        maxGroupWeight(config.maxGroupWeight());
        maxGroupDelay(config.maxGroupDelay());
        maxStatusGroupSize(config.maxStatusGroupSize());
        maxStatusGroupWeight(config.maxStatusGroupWeight());
        maxStatusGroupDelay(config.maxStatusGroupDelay());
        services(config.services());
        consumerServices(config.consumerServices());
        zooHttpTargetConfig(config.zooHttpTargetConfig());
        statusProxyFallbackDelay(config.statusProxyFallbackDelay());
        staterTimeBoundaries(config.staterTimeBoundaries());
    }

    protected AbstractProducerConfigBuilder(
        final IniConfig config,
        final ProducerConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        searchMapPath = config.section("searchmap").getInputFile(
            "path",
            new File(defaults.searchMapPath())).getPath();
        IniConfig zookeeper = config.section("zookeeper");
        zookeeperTimeout = zookeeper.getIntegerDuration(
            "timeout",
            defaults.zookeeperTimeout());
        waitTimeout = zookeeper.getIntegerDuration(
            "wait-timeout",
            defaults.waitTimeout());
        maxGroupSize =
            zookeeper.getInt("max-group-size", defaults.maxGroupSize());
        maxGroupWeight =
            zookeeper.getInt("max-group-weight", defaults.maxGroupWeight());
        maxGroupDelay =
            zookeeper.getInt("max-group-delay", defaults.maxGroupDelay());
        maxStatusGroupSize =
            zookeeper.getInt("max-status-group-size", defaults.maxStatusGroupSize());
        maxStatusGroupWeight =
            zookeeper.getInt("max-status-group-weight", defaults.maxStatusGroupWeight());
        maxStatusGroupDelay =
            zookeeper.getInt("max-status-group-delay", defaults.maxStatusGroupDelay());
        syncHelperThreads =
            zookeeper.getInt(
                "sync-helper-threads",
                defaults.syncHelperThreads());
        services = zookeeper.get(
            "services",
            defaults.services(),
            new CollectionParser<>(String::trim, HashSet::new));
        consumerServices = zookeeper.get(
            "consumer-services",
            services,
            new CollectionParser<>(String::trim, HashSet::new));
        zooHttpTargetConfig =
            new HttpTargetConfigBuilder(
                config.section("zookeeper-http"),
                defaults.zooHttpTargetConfig());
        statusProxyFallbackDelay =
            zookeeper.getInt(
                "status-proxy-fallback-delay",
                defaults.statusProxyFallbackDelay());
        staterTimeBoundaries = zookeeper.get(
            "stater-time-boundaries",
            defaults.staterTimeBoundaries(),
            new CollectionParser<>(
                x -> Integer.valueOf(x.trim()),
                ArrayList::new));
    }

    @Override
    public String searchMapPath() {
        return searchMapPath;
    }

    public T searchMapPath(final String searchMapPath) {
        this.searchMapPath = searchMapPath;
        return self();
    }

    @Override
    public int syncHelperThreads() {
        return syncHelperThreads;
    }

    public T syncHelperThreads(final int syncHelperThreads) {
        this.syncHelperThreads = syncHelperThreads;
        return self();
    }

    @Override
    public int zookeeperTimeout() {
        return zookeeperTimeout;
    }

    public T zookeeperTimeout(final int zookeeperTimeout) {
        this.zookeeperTimeout = zookeeperTimeout;
        return self();
    }

    @Override
    public int waitTimeout() {
        return waitTimeout;
    }

    public T waitTimeout(final int waitTimeout) {
        this.waitTimeout = waitTimeout;
        return self();
    }

    @Override
    public int maxGroupSize() {
        return maxGroupSize;
    }

    public T maxGroupSize(final int maxGroupSize) {
        this.maxGroupSize = maxGroupSize;
        return self();
    }

    @Override
    public int maxGroupWeight() {
        return maxGroupWeight;
    }

    public T maxGroupWeight(final int maxGroupWeight) {
        this.maxGroupWeight = maxGroupWeight;
        return self();
    }

    @Override
    public int maxGroupDelay() {
        return maxGroupDelay;
    }

    public T maxGroupDelay(final int maxGroupDelay) {
        this.maxGroupDelay = maxGroupDelay;
        return self();
    }

    @Override
    public int maxStatusGroupSize() {
        return maxStatusGroupSize;
    }

    public T maxStatusGroupSize(final int maxStatusGroupSize) {
        this.maxStatusGroupSize = maxStatusGroupSize;
        return self();
    }

    @Override
    public int maxStatusGroupWeight() {
        return maxStatusGroupWeight;
    }

    public T maxStatusGroupWeight(final int maxStatusGroupWeight) {
        this.maxStatusGroupWeight = maxStatusGroupWeight;
        return self();
    }

    @Override
    public int maxStatusGroupDelay() {
        return maxStatusGroupDelay;
    }

    public T maxStatusGroupDelay(final int maxStatusGroupDelay) {
        this.maxStatusGroupDelay = maxStatusGroupDelay;
        return self();
    }

    @Override
    public Set<String> services() {
        return services;
    }

    public T services(final Set<String> services) {
        this.services = new HashSet<>(services);
        return self();
    }

    @Override
    public Set<String> consumerServices() {
        return consumerServices;
    }

    public T consumerServices(final Set<String> consumerServices) {
        this.consumerServices = new HashSet<>(consumerServices);
        return self();
    }

    @Override
    public HttpTargetConfigBuilder zooHttpTargetConfig() {
        return zooHttpTargetConfig;
    }

    public T zooHttpTargetConfig(final HttpTargetConfig zooHttpTargetConfig)
    {
        this.zooHttpTargetConfig =
            new HttpTargetConfigBuilder(zooHttpTargetConfig);
        return self();
    }

    @Override
    public int statusProxyFallbackDelay() {
        return statusProxyFallbackDelay;
    }

    public T statusProxyFallbackDelay(final int delay) {
        this.statusProxyFallbackDelay = delay;
        return self();
    }

    @Override
    public List<Integer> staterTimeBoundaries() {
        return staterTimeBoundaries;
    }

    public T staterTimeBoundaries(final List<Integer> staterTimeBoundaries) {
        this.staterTimeBoundaries = new ArrayList<>(staterTimeBoundaries);
        return self();
    }
}

