package ru.yandex.dispatcher.producer;

import java.io.Closeable;
import java.io.IOException;

import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.http.ContentTooLongException;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpRequest;

import org.apache.http.entity.ContentType;
import org.apache.http.nio.ContentDecoder;
import org.apache.http.nio.IOControl;
import org.apache.http.nio.entity.ContentBufferEntity;
import org.apache.http.nio.protocol.AbstractAsyncRequestConsumer;
import org.apache.http.nio.util.ByteBufferAllocator;
import org.apache.http.nio.util.HeapByteBufferAllocator;
import org.apache.http.nio.util.SimpleInputBuffer;
import org.apache.http.protocol.HttpContext;

public class BufferReusingAsyncRequestConsumer
    extends AbstractAsyncRequestConsumer
{
    private static final ConcurrentLinkedQueue<ReusableContentInputBuffer> BUFFERS_POOL =
        new ConcurrentLinkedQueue<>();

    private volatile HttpRequest request;
    private volatile SimpleInputBuffer buf;

    public BufferReusingAsyncRequestConsumer() {
        super();
    }

    @Override
    protected void onRequestReceived(final HttpRequest request)
        throws IOException
    {
        this.request = request;
    }

    @Override
    protected void onEntityEnclosed(
        final HttpEntity entity,
        final ContentType contentType)
        throws IOException
    {
        long len = entity.getContentLength();
        if (len > Integer.MAX_VALUE) {
            throw new ContentTooLongException("Entity content is too long: " + len);
        }
        if (len < 4096) {
            len = 4096;
        }
        buf = BUFFERS_POOL.poll();
        if (buf == null) {// || buf.capacity() < len) {
            buf =
                new ReusableContentInputBuffer((int) len, new HeapByteBufferAllocator());
        } else {
        }
        ((HttpEntityEnclosingRequest) this.request).setEntity(
                new CloseableContentBufferEntity(entity, this.buf));
    }

    @Override
    protected void onContentReceived(
            final ContentDecoder decoder, final IOControl ioctrl) throws IOException {
        this.buf.consumeContent(decoder);
    }

    @Override
    protected void releaseResources() {
        this.request = null;
        buf = null;
    }

    @Override
    protected HttpRequest buildResult(final HttpContext context) {
        return this.request;
    }

    private static class ReusableContentInputBuffer extends SimpleInputBuffer
        implements Closeable
    {
        public ReusableContentInputBuffer(
            final int buffersize,
            final ByteBufferAllocator allocator)
        {
            super(buffersize, allocator);
        }

        @Override
        public void close() {
            reset();
            BUFFERS_POOL.offer(this);
        }

        @Override
        public String toString() {
            return "cap=" + capacity();
        }
    }
}
