package ru.yandex.dispatcher.producer;

import java.security.MessageDigest;
import java.util.Collections;
import java.util.List;
import java.util.logging.Logger;
import java.util.zip.CRC32;

public class HashUtil {
    private HashUtil() {
    }

    private static final ThreadLocal<MessageDigest> hashFunc = new ThreadLocal<MessageDigest>();
    private static final ThreadLocal<CRC32> crcFunc = new ThreadLocal<CRC32>();
    private static final ThreadLocal<StringBuilder> hexBuilder = new ThreadLocal<StringBuilder>();

    private static MessageDigest getLocalDigest() throws java.security.NoSuchAlgorithmException
    {
        MessageDigest md = hashFunc.get();
        if( md == null )
        {
            md = MessageDigest.getInstance( "SHA-256" );
            hashFunc.set( md );
        }
        return md;
    }

    private static CRC32 getLocalCRC32()
    {
        CRC32 crc = crcFunc.get();
        if( crc == null )
        {
            crc = new CRC32();
            crcFunc.set( crc );
        }
        return crc;
    }

    private static StringBuilder getBuilder()
    {
        StringBuilder sb = hexBuilder.get();
        if( sb == null )
        {
            sb = new StringBuilder();
            hexBuilder.set( sb );
        }
        return sb;
    }

    public static String hash( byte[] data )
    {
        try
        {
            MessageDigest md = getLocalDigest();
            md.reset();
            md.update( data );
            byte[] digest = md.digest();
            StringBuilder sb = getBuilder();
            sb.setLength( 0 );
            for( int i = 0; i < digest.length; i++ )
            {
                sb.append( Integer.toHexString(0xFF & digest[i]) );
            }
            return sb.toString();
        }
        catch( java.security.NoSuchAlgorithmException e )
        {
            CRC32 crc = getLocalCRC32();
            crc.reset();
            crc.update( data );
            return Long.toHexString( crc.getValue() );
        }
    }

    public static String hashRand( byte[] data )
    {
        try
        {
            MessageDigest md = getLocalDigest();
            md.reset();
            md.update( data );
            long nanoTime = System.nanoTime();
            md.update( (byte)nanoTime );
            md.update( (byte)(nanoTime >> 8) );
            md.update( (byte)(nanoTime >> 16) );
            md.update( (byte)(nanoTime >> 32) );
            byte[] digest = md.digest();
            StringBuilder sb = getBuilder();
            sb.setLength( 0 );
            for( int i = 0; i < digest.length; i++ )
            {
                sb.append( Integer.toHexString(0xFF & digest[i]) );
            }
            return sb.toString();
        }
        catch( java.security.NoSuchAlgorithmException e )
        {
            CRC32 crc = getLocalCRC32();
            crc.reset();
            crc.update( data );
            long nanoTime = System.nanoTime();
            crc.update( (byte)nanoTime );
            crc.update( (byte)(nanoTime >> 8) );
            crc.update( (byte)(nanoTime >> 16) );
            crc.update( (byte)(nanoTime >> 32) );
            return Long.toHexString( crc.getValue() );
        }
    }
}
