package ru.yandex.dispatcher.producer;

import java.math.BigDecimal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.LockSupport;
import java.util.logging.Logger;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.OpResult;
import org.apache.zookeeper.Transaction;
import org.apache.zookeeper.ZooDefs.OpCode;
import org.apache.zookeeper.ZooKeeper;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.dispatcher.common.ZooCallback;
import ru.yandex.dispatcher.common.ZooException;
import ru.yandex.dispatcher.common.connection.ZooConnection;
import ru.yandex.dispatcher.common.mappedvars.ZooBigDecimalNode;
import ru.yandex.util.timesource.TimeSource;

public class MessageGrouper extends Thread {
    private static final float MS_MICROS = 1000000;
    private final ZooConnection conn;
    private final Logger logger;
    private final Producer producer;

    private final int maxGroupingTime;
    private final long maxGroupingTimeNanos;
    private final int maxGroupSize;
    private final int maxGroupWeight;

    private final ConcurrentLinkedQueue<QueueMessage> queue =
        new ConcurrentLinkedQueue<>();
    private volatile boolean stop = false;
    private final AtomicLong groupId;
    private final TimeFrameQueue<Long> writeTimesQueue;

    public MessageGrouper(
        final ZooConnection conn,
        final Logger logger,
        final Producer producer)
    {
        this.conn = conn;
        this.logger = logger;
        this.producer = producer;

        ProducerConfig config = (ProducerConfig) producer.config();
        this.maxGroupingTime = config.maxGroupDelay();
        maxGroupingTimeNanos = maxGroupingTime * 1000;
        this.maxGroupWeight = config.maxGroupWeight();
        this.maxGroupSize = config.maxGroupSize();
        groupId = new AtomicLong(TimeSource.INSTANCE.currentTimeMillis());
//        this.writeTimesQueue = producer.writeTimesQueue(conn.port());
        String ZKhost = conn.ZKhost();
        if (ZKhost.indexOf('.') >= 0 ){
            this.writeTimesQueue = producer.writeTimesQueue(ZKhost.substring(0, ZKhost.indexOf('.')));
        } else {
            this.writeTimesQueue = producer.writeTimesQueue(ZKhost);
        }
    };

    public ZooConnection connection() {
        return conn;
    }

    public void run() {
        ArrayList<QueueMessage> grouped = new ArrayList<QueueMessage>();
        while (!stop) {
            try {
                QueueMessage message = take();
                final long groupingStartTime = System.nanoTime();
                long groupingTime = 0;
                int weight = message.weight();
                grouped.add(message);
                do {
                    final long maxPoll =
                        maxGroupingTimeNanos - groupingTime;
                    message = poll(maxPoll);
                    if (message != null) {
                        grouped.add(message);
                        weight += message.weight();
                    }
                    groupingTime = System.nanoTime() - groupingStartTime;
                } while (groupingTime < maxGroupingTimeNanos
                    && weight < maxGroupWeight
                    && grouped.size() < maxGroupSize);
                sendGroup(grouped);
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("Grouping time: "
                        + ((System.nanoTime() - groupingStartTime) / MS_MICROS)
                        + " ms");
                }
                grouped.clear();
            } catch (Exception e) {
                logger.log(
                    Level.SEVERE,
                    "Unhandled exception in MessageGrouper thread",
                    e);
            }
        }
    }

    private QueueMessage poll(final long nanos) {
        QueueMessage msg = queue.poll();
        if (msg == null) {
            LockSupport.parkNanos(nanos);
            msg = queue.poll();
        }
        return msg;
    }

    private QueueMessage take() {
        QueueMessage msg = queue.poll();
        while (msg == null) {
            LockSupport.parkNanos(maxGroupingTimeNanos);
            msg = queue.poll();
        }
        return msg;
    }

    private void sendGroup(final List<QueueMessage> messages) {
        ZooKeeper zk = conn.getZk(
            new ZooCallback() {
                @Override
                public void error(ZooException e) {
                    for (QueueMessage message : messages) {
                        message.failed(e);
                    }
                }
            });
        if (zk == null) {
            return;
        }
        Transaction trans = zk.transaction();
        HashMap<ZooBigDecimalNode, BigDecimal> producerCounters = null;
        for (QueueMessage m : messages) {
            trans.create(m.path(), m.data(), m.hash(), m.mode());
            //group producer positions. if multiply messages
            //are updating one position - update only the last
            if (m.producerPositionNode() != null) {
                if (producerCounters == null) {
                    producerCounters = new HashMap<>();
                }
                BigDecimal position =
                    producerCounters.get(m.producerPositionNode());
                if (position == null
                    || position.compareTo(m.producerPosition()) < 0)
                {
                    producerCounters.put(
                        m.producerPositionNode(), m.producerPosition());
                }
            }
        }
        if (producerCounters != null) {
            for (Map.Entry<ZooBigDecimalNode, BigDecimal> entry
                : producerCounters.entrySet())
            {
                trans.setData(
                    entry.getKey().path(),
                    entry.getValue().toPlainString().getBytes(),
                    -1);
            }
        }
        final ArrayList<QueueMessage> group = new ArrayList<>(messages);
        final long newGroupId = groupId.incrementAndGet();
        trans.commit(
            new GroupingTransactionCallback(
                group,
                producerCounters,
                TimeSource.INSTANCE.currentTimeMillis(),
                newGroupId),
            null);
        int posCount = producerCounters == null ? 0 : producerCounters.size();
        final String master;
        if (conn.master()) {
            master = "(master)";
        } else {
            master = "(slave)";
        }
        if (logger.isLoggable(Level.INFO)) {
            logger.info("Sent Group: messages=" + messages.size()
                + ", positions=" + posCount + ", gid="
                + Long.toHexString(newGroupId)
                + ", server=" + conn.currentServer() + master);
        }
    }

    public void enqueueMessage(final QueueMessage message) {
        if (logger.isLoggable(Level.INFO)) {
            logger.info("Enqueuing: " + message);
        }
        queue.offer(message);
    }

    public void enqueue(final List<QueueMessage> messages) {
        for (QueueMessage message : messages) {
            if (logger.isLoggable(Level.INFO)) {
                logger.info("Enqueuing: " + message);
            }
            queue.offer(message);
        }
    }

    private void failMessages(
        final List<QueueMessage> messages,
        final Exception error)
    {
        for (QueueMessage message : messages) {
            message.failed(error);
        }
    }

    private void createPathAndRestart(
        final String path,
        final List<QueueMessage> messages)
    {
        FutureCallback<Void> cb = new FutureCallback<Void>() {
            @Override
            public void completed(Void v) {
                sendGroup(messages);
            }

            @Override
            public void failed(Exception e) {
                failMessages(messages, e);
            }

            @Override
            public void cancelled() {
            }
        };
        ZooUtil.createPath(conn, path, cb);
    }

    private void createParentAndRestart(
        final String path,
        final List<QueueMessage> messages)
    {
        final int slash = path.lastIndexOf('/');
        final String parent = path.substring(0, slash);
        createPathAndRestart(parent, messages);
    }

    private class GroupingTransactionCallback
        implements org.apache.zookeeper.AsyncCallback.MultiCallback
    {
        private final List<QueueMessage> messages;
        private final HashMap<ZooBigDecimalNode, BigDecimal> producerCounters;
        private final long startTime;
        private final long gid;

        public GroupingTransactionCallback(
            final List<QueueMessage> messages,
            final HashMap<ZooBigDecimalNode, BigDecimal> producerCounters,
            final long startTime,
            final long gid)
        {
            this.messages = messages;
            this.producerCounters = producerCounters;
            this.startTime = startTime;
            this.gid = gid;
        }

        @Override
        public void processResult(
            final int rc,
            final String path,
            final Object ctx,
            final List<OpResult> results)
        {
            KeeperException.Code code = KeeperException.Code.get(rc);
            int count = 0;
            if (results != null) {
                count = results.size();
            }
            final long time = TimeSource.INSTANCE.currentTimeMillis() - startTime;
            writeTimesQueue.accept(Producer.packTimeNCount(time, count));
            if (logger.isLoggable(Level.INFO)) {
                logger.info("Group gid: " + Long.toHexString(gid)
                    + ", process result code: " + code
                    + ", time: " + time
                    + " ms");
            }
            if (code == code.OK) {
                completed(messages, results);
            } else {
                if (results == null) {
                    final String error =
                        "Unhandled transaction results: null List<OpResult>"
                        + ", code: " + code;
                    logger.severe(error);
                    failMessages(messages,
                        new MessageSendFailedException(error));
                } else {
                    failed(messages, results);
                }
            }
        }

        private void completed(
            final List<QueueMessage> messages,
            final List<OpResult> results)
        {
            Iterator<QueueMessage> msgIter = messages.iterator();
            Iterator<OpResult> resIter = results.iterator();
            while (msgIter.hasNext()) {
                QueueMessage msg = msgIter.next();
                OpResult op = resIter.next();
                OpResult.CreateResult createRes = (OpResult.CreateResult)op;
                String returnPath = createRes.getPath();
                msg.completed(returnPath);
            }
        }

        private void failed(
            final List<QueueMessage> messages,
            final List<OpResult> results)
        {
            Iterator<QueueMessage> msgIter = messages.iterator();
            Iterator<OpResult> resIter = results.iterator();
            while (msgIter.hasNext()) {
                QueueMessage msg = msgIter.next();
                OpResult op = resIter.next();
                switch (op.getType()) {
                    case OpCode.create:
                        //even if this message is ok next following error
                        //will rollback this message
                        break;
                    case OpCode.error:
                        OpResult.ErrorResult er = (OpResult.ErrorResult)op;
                        KeeperException.Code code =
                            KeeperException.Code.get(er.getErr());
                        if (code == code.OK) {
                            //even if this message is ok next following error
                            //will rollback this message
                            break;
                        } else if (code == code.NONODE) {
                            if (logger.isLoggable(Level.SEVERE)) {
                                logger.severe("NoNode: " + msg.path()
                                    + ". Creating");
                            }
                            createParentAndRestart(msg.path(), messages);
                            return;
                        } else {
                            final String error =
                                "Transaction failed: "
                                + "Unhandled error code for msg: " + msg.path()
                                + ", rc = " + code;
                            logger.severe(error);
                            failMessages(messages,
                                new MessageSendFailedException(error));
                            return;
                        }
                    default:
                        final String error =
                            "Transaction failed: "
                            + "Unhandled AsyncCallback.MultiCallback result"
                            + " operation type: " + op.getClass().getName();
                        logger.severe(error);
                        failMessages(messages,
                                new MessageSendFailedException(error));
                        return;
                }
            }
            //next results is for Producer positions nodes
            if (resIter.hasNext() && producerCounters == null) {
                final String error =
                    "Inconsistent transaction results: "
                    + "results list size is bigger than transaction size";
                logger.severe(error);
                failMessages(messages,
                    new MessageSendFailedException(error));
            }
            Iterator<ZooBigDecimalNode> producerNodeIter =
                producerCounters.keySet().iterator();
            while (resIter.hasNext()) {
                OpResult op = resIter.next();
                ZooBigDecimalNode node = producerNodeIter.next();
                switch (op.getType()) {
                    case OpCode.create:
                        //even if this message is ok next following error
                        //will rollback this message
                        break;
                    case OpCode.error:
                        OpResult.ErrorResult er = (OpResult.ErrorResult)op;
                        KeeperException.Code code =
                            KeeperException.Code.get(er.getErr());
                        if (code == code.OK) {
                            //even if this message is ok next following error
                            //will rollback this message
                            break;
                        } else if (code == code.NONODE) {
                            if (logger.isLoggable(Level.SEVERE)) {
                                logger.severe("NoNode: " + node.path()
                                    + ". Creating");
                            }
                            createPathAndRestart(node.path(), messages);
                            return;
                        } else {
                            final String error =
                                "Transaction failed: "
                                + "Unhandled error code for position node: "
                                    + node.path()
                                + ", rc = " + code;
                            logger.severe(error);
                            failMessages(messages,
                                new MessageSendFailedException(error));
                            return;
                        }
                    default:
                        final String error =
                            "Transaction failed: "
                            + "Unhandled AsyncCallback.MultiCallback result"
                            + " operation type: " + op.getClass().getName();
                        logger.severe(error);
                        failMessages(messages,
                                new MessageSendFailedException(error));
                        return;
                }
            }
        }
    }
}
