package ru.yandex.dispatcher.producer;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.MimeTokenStream;
import org.apache.xerces.impl.io.UTF8Reader;
import org.json.simple.parser.JSONParser;

import ru.yandex.dispatcher.common.AbstractHttpMessage;
import ru.yandex.dispatcher.common.HttpGetMessage;
import ru.yandex.dispatcher.common.HttpPostMessage;
import ru.yandex.dispatcher.producer.json.DefaultFailer;
import ru.yandex.dispatcher.producer.json.HandlersManager;
import ru.yandex.dispatcher.producer.json.PrefixExtractor;
import ru.yandex.dispatcher.producer.json.RootHandler;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.NotImplementedException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.mail.mime.DefaultMimeConfig;
import ru.yandex.mail.mime.OverwritingBodyDescriptorBuilder;
import ru.yandex.mail.mime.Utf8FieldBuilder;
import ru.yandex.util.timesource.TimeSource;

public class PostDataHandler extends QueueRequestHandlerBase {
    private static final int DEFAULT_TEMP_BUFFER_SIZE = 8192;
    private static ThreadLocal<byte[]> threadLocalBuffer =
        new ThreadLocal<byte[]>();
    private static final Set<String> MULTIPART_DROP_HEADERS;

    static {
        MULTIPART_DROP_HEADERS =
            new HashSet<>(AbstractHttpMessage.DROP_HEADERS);
        MULTIPART_DROP_HEADERS.add(HttpHeaders.CONTENT_TYPE);
    }

    public PostDataHandler(final Producer producer) {
        super(producer);
    }

    @Override
    protected void fillRequest(final QueueRequest request)
        throws IOException, HttpException
    {
        if (!request.multiPart()) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request.request()).getEntity();
            byte[] postData = EntityUtils.toByteArray(entity);
            if (request.commonPrefix() == null) {
                request.setCommonPrefix(prefixFromJson(postData));
            }
            request.setPayload(postData);
        } else {
            if (request.commonPrefix() == null && request.doWait()) {
                throw new BadRequestException(
                        "wait=true is not supported "
                        + "in multiprefix batch mode");
            }
        }
    }

    @Override
    protected FutureCallback<QueueMessage> baseResponseCallback(
        final QueueRequest request,
        final FutureCallback<QueueMessage> responseCallback)
        throws IOException, HttpException
    {
        if (request.multiPart()) {
            return new MultiMessageCallback(request, responseCallback);
        } else {
            return super.baseResponseCallback(request, responseCallback);
        }
    }

    @Override
    protected void addMessages(
        final QueueRequest request)
        throws IOException, HttpException
    {
        if (!request.multiPart()) {
            HttpPostMessage msg = new HttpPostMessage(
                request.request().getRequestLine().getUri(),
                request.payload(),
                request.doWait(),
                TimeSource.INSTANCE.currentTimeMillis());
            msg.copyHeadersFrom(request.request());
            BigDecimal position = producer.getPosition(request.request());
            addMessage(request, request.commonPrefix(), msg, position);
        } else {
            if (request.commonPrefix() != null) {
                addMultiPartMessages(request);
            } else {
                addMultiPartMessagesMultiPrefix(request);
            }
        }
    }

    private long prefixFromJson(final byte[] postData)
        throws IOException, HttpException
    {
        JSONParser parser = new JSONParser();
        HandlersManager manager =
            new HandlersManager(new DefaultFailer(parser));
        PrefixExtractor extractor = new PrefixExtractor();
        manager.push(new RootHandler(manager, extractor));
        try {
            parser.parse(new UTF8Reader(new ByteArrayInputStream(postData)), manager);
            return extractor.prefix();
        } catch (org.json.simple.parser.ParseException
            | java.text.ParseException e)
        {
            throw new BadRequestException("no prefix, shard, ZooShardId "
                + "or parseable json was specified");
        }
    }

    private void addMultiPartMessages(
        final QueueRequest request)
        throws HttpException
    {
        final long currentTime = TimeSource.INSTANCE.currentTimeMillis();
        final String messagePath = producer.pathFor(request.service(),
            request.commonPrefix());
        byte[] tempBuffer = getTempBuffer();
        MimeTokenStream stream = new MimeTokenStream(
            DefaultMimeConfig.INSTANCE,
            null,
            new Utf8FieldBuilder(),
            new OverwritingBodyDescriptorBuilder());
        try {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request.request()).getEntity();
            String producerPosition = null;
            String uri = null;
            String defaultMethod = HttpPost.METHOD_NAME;
            String method = defaultMethod;
            byte[] body = null;

            stream.parseHeadless(
                entity.getContent(),
                entity.getContentType().getValue());
            EntityState state = stream.getState();
            while (state != EntityState.T_END_OF_STREAM) {
                switch (state) {
                    case T_FIELD:
                        Field field = stream.getField();
                        if (YandexHeaders.PRODUCER_POSITION
                            .equals(field.getName()))
                        {
                            producerPosition = field.getBody();
                        } else if ("prefix".equals(field.getName())
                            || "shard".equals(field.getName())
                            || YandexHeaders.ZOO_SHARD_ID.equals(field.getName()))
                        {
                            throw new BadRequestException("prefix, shard or ZooShardId "
                                + " was already specified in the main header or cgi"
                                + " parameters and is used as common prefix");
                        } else if (YandexHeaders.URI.equals(field.getName())) {
                            uri = field.getBody();
                        } else if (YandexHeaders.ZOO_HTTP_METHOD.equalsIgnoreCase(field.getName())) {
                            method = field.getBody();
                        }
                        break;
                    case T_BODY:
                        int pos = 0;
                        try (InputStream is = stream.getDecodedInputStream())
                                //stream.getBodyDescriptor().getCharset()))
                        {
                            int read = is.read(tempBuffer, pos, tempBuffer.length - pos);
                            while (read != -1) {
                                pos += read;
                                if (pos == tempBuffer.length) {
                                    tempBuffer = growTempBuffer(tempBuffer);
                                }
                                read = is.read(tempBuffer, pos, tempBuffer.length - pos);
                            }
                        }
                        body = Arrays.copyOf(tempBuffer, pos);

                        uri = null;
                        break;
                    case T_END_BODYPART:
                        BigDecimal position = null;
                        if (producerPosition != null) {
                            position = new BigDecimal(producerPosition);
                        }

                        if (uri == null) {
                            uri = request.request().getRequestLine().getUri();
                        }
                        AbstractHttpMessage msg;

                        if (HttpPost.METHOD_NAME.equalsIgnoreCase(method)) {
                            if (body == null) {
                                throw new BadRequestException(
                                    "Body is null for " + uri);
                            }
                            msg = new HttpPostMessage(
                                uri,
                                body,
                                request.doWait(),
                                currentTime);
                        } else if (HttpGet.METHOD_NAME.equalsIgnoreCase(method)) {
                            msg = new HttpGetMessage(uri, request.doWait(), currentTime);
                        } else {
                            throw new BadRequestException(
                                "Unsupported multipart http message " + method + " for " + uri);
                        }

                        msg.copyHeadersFrom(request.request(), MULTIPART_DROP_HEADERS);
                        addMessage(request, request.commonPrefix(), msg, position);

                        producerPosition = null;
                        uri = null;
                        method = defaultMethod;
                        body = null;
                        break;
                    default:
                        break;
                }
                state = stream.next();
            }
        } catch (Throwable t) {
            throw new NotImplementedException(t);
        }
    }


    private void addMultiPartMessagesMultiPrefix(
        final QueueRequest request)
        throws HttpException
    {
        final long currentTime = TimeSource.INSTANCE.currentTimeMillis();
        byte[] tempBuffer = getTempBuffer();
        MimeTokenStream stream = new MimeTokenStream(
            DefaultMimeConfig.INSTANCE,
            null,
            new Utf8FieldBuilder(),
            new OverwritingBodyDescriptorBuilder());
        try {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request.request()).getEntity();
            String producerPosition = null;
            String uri = null;
            Long prefix = null;
            String defaultMethod = HttpPost.METHOD_NAME;
            String method = defaultMethod;
            byte[] body = null;

            stream.parseHeadless(
                entity.getContent(),
                entity.getContentType().getValue());
            EntityState state = stream.getState();

            while (state != EntityState.T_END_OF_STREAM) {
                switch (state) {
                    case T_FIELD:
                        Field field = stream.getField();
                        if (YandexHeaders.PRODUCER_POSITION
                            .equals(field.getName()))
                        {
                            producerPosition = field.getBody();
                        } else if ("prefix".equals(field.getName())
                            || "shard".equals(field.getName())
                            || YandexHeaders.ZOO_SHARD_ID.equals(field.getName()))
                        {
                            prefix = Long.parseLong(field.getBody());
                        } else if (YandexHeaders.URI.equals(field.getName())) {
                            uri = field.getBody();
                        } else if (YandexHeaders.ZOO_HTTP_METHOD.equalsIgnoreCase(field.getName())) {
                            method = field.getBody();
                        }
                        break;
                    case T_BODY:
                        int pos = 0;
                        if (prefix == null) {
                            throw new BadRequestException("no prefix, shard, ZooShardId "
                                + " was specified");
                        }
                        final String messagePath =
                            producer.pathFor(request.service(), prefix);
                        try (InputStream is = stream.getDecodedInputStream())
                                //stream.getBodyDescriptor().getCharset()))
                        {
                            int read = is.read(tempBuffer, pos, tempBuffer.length - pos);
                            while (read != -1) {
                                pos += read;
                                if (pos == tempBuffer.length) {
                                    tempBuffer = growTempBuffer(tempBuffer);
                                }
                                read = is.read(tempBuffer, pos, tempBuffer.length - pos);
                            }
                        }
                        body = Arrays.copyOf(tempBuffer, pos);
                        break;
                    case T_END_BODYPART:
                        BigDecimal position = null;
                        if (producerPosition != null) {
                            position = new BigDecimal(producerPosition);
                        }

                        if (uri == null) {
                            uri = request.request().getRequestLine().getUri();
                        }
                        AbstractHttpMessage msg;

                        if (HttpPost.METHOD_NAME.equalsIgnoreCase(method)) {
                            if (body == null) {
                                throw new BadRequestException(
                                    "Body is null for " + uri);
                            }
                            msg = new HttpPostMessage(
                                uri,
                                body,
                                request.doWait(),
                                currentTime);
                        } else if (HttpGet.METHOD_NAME.equalsIgnoreCase(method)) {
                            msg = new HttpGetMessage(uri, request.doWait(), currentTime);
                        } else {
                            throw new BadRequestException(
                                "Unsupported multipart http message " + method + " for " + uri);
                        }

                        msg.copyHeadersFrom(request.request(), MULTIPART_DROP_HEADERS);
                        addMessage(request, prefix, msg, position);

                        producerPosition = null;
                        prefix = null;
                        uri = null;
                        method = defaultMethod;
                        body = null;
                        break;
                    default:
                        break;
                }
                state = stream.next();
            }
        } catch (Throwable t) {
            throw new NotImplementedException(t);
        }
    }

    private byte[] getTempBuffer() {
        byte[] buf = threadLocalBuffer.get();
        if (buf == null) {
            buf = new byte[DEFAULT_TEMP_BUFFER_SIZE];
            threadLocalBuffer.set(buf);
        }
        return buf;
    }

    private byte[] growTempBuffer(final byte[] buf) {
        byte[] newBuf = Arrays.copyOf(buf, buf.length << 1);
        threadLocalBuffer.set(newBuf);
        return newBuf;
    }

    private static class MultiMessageCallback
        implements FutureCallback<QueueMessage>
    {
        private final QueueRequest request;
        private final FutureCallback<QueueMessage> nextCallback;
        private Iterator<QueueMessage> iter = null;
        private boolean finished = false;

        public MultiMessageCallback(
            final QueueRequest request,
            final FutureCallback<QueueMessage> nextCallback)
        {
            this.request = request;
            this.nextCallback = nextCallback;
        }

        @Override
        public void completed(final QueueMessage message) {
            if (finished) {
                return;
            }
            if (iter == null) {
                iter = request.messages().iterator();
            }
            if (iter.hasNext()) {
                QueueMessage pending = iter.next();
                if (pending != message) {
                    failed(new BatchResponseError(
                        "Out of order processin: "
                        + "waiting for message " + pending
                        + " but received " + message));
                }
                if (!iter.hasNext()) {
                    finished = true;
                    nextCallback.completed(message);
                }
            }
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(Exception e) {
            if (finished) {
                return;
            }
            finished = true;
            nextCallback.failed(e);
        }
    }
}
