package ru.yandex.dispatcher.producer;

import java.io.IOException;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.nio.charset.UnsupportedCharsetException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.MethodNotSupportedException;
import org.apache.http.RequestLine;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.data.Stat;

import ru.yandex.collection.IntPair;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.dispatcher.common.ByteArrayCallback;
import ru.yandex.dispatcher.common.DelayShardMessage;
import ru.yandex.dispatcher.common.ErrorMessage;
import ru.yandex.dispatcher.common.HangShardMessage;
import ru.yandex.dispatcher.common.HttpGetMessage;
import ru.yandex.dispatcher.common.HttpMessage;
import ru.yandex.dispatcher.common.HttpPostMessage;
import ru.yandex.dispatcher.common.SerializeUtils;
import ru.yandex.dispatcher.common.StatCallback;
import ru.yandex.dispatcher.common.ZooException;
import ru.yandex.dispatcher.common.ZooNoNodeException;
import ru.yandex.dispatcher.common.connection.ZooConnection;
import ru.yandex.dispatcher.common.connection.ZooConnectionDSN;
import ru.yandex.dispatcher.common.mappedvars.ZooBigDecimalNode;
import ru.yandex.dispatcher.common.mappedvars.ZooLock;
import ru.yandex.dispatcher.common.mappedvars.ZooNodeMapping;
import ru.yandex.dispatcher.producer.statusreader.ConsumerServer;
import ru.yandex.dispatcher.producer.statusreader.StatusWatcher;
import ru.yandex.dispatcher.producer.statusreader.ZooKeeperDSN;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxyRequestHandlerAdapter;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.server.async.DelegatedHttpAsyncRequestHandler;
import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ForbiddenException;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.server.HttpServerConfigBuilder;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.timesource.TimeSource;

public class Producer
    extends HttpProxy<ImmutableProducerConfig>
    implements StatusWatcher
{
    private static final String MS_AMMM = "-ms_ammm";
    private static final BigDecimal PRODUCER_POSITION_NO_VALUE =
        new BigDecimal(-1L);
    private static final int COUNT_SHIFT = 40;
    private static final long TIME_MASK = (1L << COUNT_SHIFT) - 1;

    private final Map<String, MessageGrouper> groupers = new HashMap<>();
    private final Map<String, StatusQueue> statusQueues = new HashMap<>();
    private final Map<String, Long> zooLocksBlock = new ConcurrentHashMap<>();
    private final Map<String, Status[]> servicesStatus =
        new ConcurrentHashMap<>();
    private Map<String, ZooConnection> varMapConnections =
        new ConcurrentHashMap<>();

    private final ConcurrentHashMap<
        String,
        ConcurrentHashMap<String, ZooNodeMapping>> mappedNodes =
            new ConcurrentHashMap<>();

    private final ConcurrentHashMap<String, ZooLock> zooLocks =
        new ConcurrentHashMap<>();

    private final List<ZooConnection> zooConnections = new ArrayList<>();
//    private final ConcurrentLinkedQueue<ZooLock> ephemeralLocks =
//        new ConcurrentLinkedQueue<>();

    private final DelayQueue<ConsumerWaiter> consumerWaiters =
        new DelayQueue<>();

//    private final ConcurrentHashMap<
//        Integer,
//        TimeFrameQueue<Long>> writeTimesFrameQueues = new ConcurrentHashMap<>();
//    private final ConcurrentHashMap<
//        Integer,
//        TimeFrameQueue<Long>> statusWriteTimesFrameQueues =
//            new ConcurrentHashMap<>();
    private final ConcurrentHashMap<
        String,
        TimeFrameQueue<Long>> writeTimesFrameQueues = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<
        String,
        TimeFrameQueue<Long>> statusWriteTimesFrameQueues =
        new ConcurrentHashMap<>();

    private final ConsumerWaitersExpirer consumerWaitersExpirer;
    private final int waitTimeout;
    private final SearchMap searchMap;
    private final ConsumerServer statusConsumer;
    private final StatusProxyHandler statusProxyHandler;
    private final AsyncClient httpClient;
    private final int[] staterTimeBoundaries;

    //TODO: remove ULGI fast fix
    private final ThreadPoolExecutor requestParseExecutor;
//    private final Request

    public Producer(final ImmutableProducerConfig config)
        throws IOException, ConfigException
    {
        super(config);
        waitTimeout = config.waitTimeout();

        final int timeout = config.zookeeperTimeout();

        try {
            searchMap = new SearchMap(
                config.searchMapPath(),
                logger().replacePrefix("SearchMap"));
            statusConsumer = new ConsumerServer(
                config.searchMapPath(),
                1,
                timeout,
                this,
                logger().replacePrefix("StatusConsumer-"),
                config.consumerServices());
        } catch (java.text.ParseException e) {
            throw new IOException("SearchMap parse error", e);
        }

        requestParseExecutor =
            new ThreadPoolExecutor(
                config.syncHelperThreads(),
                config.syncHelperThreads(),
                1,
                TimeUnit.HOURS,
                new LifoWaitBlockingQueue<Runnable>(config.connections()),
                new ThreadPoolExecutor.AbortPolicy());

        httpClient = client("ZooHttpClient", config.zooHttpTargetConfig());

        List<Integer> staterTimeBoundaries = config.staterTimeBoundaries();
        int size = staterTimeBoundaries.size();
        this.staterTimeBoundaries = new int[size];
        for (int i = 0; i < size; ++i) {
            this.staterTimeBoundaries[i] =
                staterTimeBoundaries.get(i).intValue();
        }

        for (String addr: searchMap.getZooKeepers()) {
            if (!config.services().isEmpty()) {
                boolean skip = true;
                for (SearchMap.Interval interval :
                        searchMap.getZooKeeperIntervals(addr))
                {
                    if (config.services().contains(interval.service())) {
                        skip = false;
                    }
                }
                if (skip) {
                    if (logger().isLoggable(Level.INFO)) {
                        logger().info("Skipping queue for zk: " + addr
                            + " no allowed services found");
                    }
                    continue;
                }
            }
            if (logger().isLoggable(Level.INFO)) {
                logger().info("Starting queue for zk: " + addr);
            }
            ZooConnection conn =
                ZooConnection.createConnection(
                    httpClient,
                    new ZooConnectionDSN(addr),
                    timeout,
                    true,
                    logger().replacePrefix("ZooReadConnection<" + addr + ">"));
            zooConnections.add(conn);
            varMapConnections.put(addr, conn);
            if (logger().isLoggable(Level.INFO)) {
                logger().info("Created varmap connection: " + addr
                    + ", conn=" + conn);
            }

            conn =
                ZooConnection.createConnection(
                    httpClient,
                    new ZooConnectionDSN(addr),
                    timeout,
                    true,
                    logger().replacePrefix("ZooWriteConnection<" + addr + ">"));
            if (logger().isLoggable(Level.INFO)) {
                logger().info("Created grouper connection: " + addr
                    + ", conn=" + conn);
            }
            conn.preferMaster(true);
            zooConnections.add(conn);
            MessageGrouper mg = new MessageGrouper(
                conn,
                logger().replacePrefix("MessageGrouper<" + conn + ">"),
                this);
            groupers.put(addr, mg);


            conn =
                ZooConnection.createConnection(
                    httpClient,
                    new ZooConnectionDSN(addr),
                    timeout,
                    true,
                    logger().replacePrefix("ZooStatusWriteConnection<" + addr + ">"));
            if (logger().isLoggable(Level.INFO)) {
                logger().info("Created status queue connection: " + addr
                    + ", conn=" + conn);
            }
            conn.preferMaster(true);
            zooConnections.add(conn);
            StatusQueue sq =
                new StatusQueue(
                    conn,
                    logger().replacePrefix("StatusQueue<" + conn + ">"),
                    this);
            statusQueues.put(addr, sq);

            for (SearchMap.Interval interval: searchMap.getZooKeeperIntervals(addr)) {
                if (!config.services().isEmpty()
                    && !config.services().contains(interval.service()))
                {
                    if (logger().isLoggable(Level.INFO)) {
                        logger().info("Skipping service <"
                            + interval.service()
                            + "> for zk <" + addr + ">: "
                            + "service is not allowed");
                    }
                    continue;
                }
                if (!servicesStatus.containsKey(interval.service())) {
                    servicesStatus.put(interval.service(), new Status[interval.max() + 1]);
                } else {
                    Status[] statuses = servicesStatus.get(interval.service());
                    if (statuses.length < interval.max() + 1) {
                        servicesStatus.put(interval.service(),
                            new Status[interval.max() + 1]);
                    }
                }
            }
        }

        for (Map.Entry<String, Status[]> entry : servicesStatus.entrySet()) {
            Status[] statuses = entry.getValue();
            for (int i = 0; i < statuses.length; i++) {
                statuses[i] = new Status(-1, null);
            }
        }

        statusProxyHandler = new StatusProxyHandler(this);

        register(
            new Pattern<>("", true),
            new PostDataHandler(this),
            RequestHandlerMapper.POST);
        register(
            new Pattern<>("", true),
            new GetDataHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_updateConsumerPosition", false),
            new UpdateConsumerPositionHandler(this),
            RequestHandlerMapper.GET);

        register(
            new Pattern<>("/_delay_shard", false),
            new DelayShardHandler(this),
            RequestHandlerMapper.GET);

        register(
            new Pattern<>("/_stop_shard", false),
            new StopShardHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_hang_shard", false),
            new StopShardHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_status", false),
            new StatusHandler(),
            RequestHandlerMapper.GET);

        register(
            new Pattern<>("/_getNodeData", false),
            new GetNodeDataHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_setNodeData", false),
            new SetNodeDataHandler(),
            RequestHandlerMapper.GET);

        register(
            new Pattern<>("/_lock", false),
            new DelegatedHttpAsyncRequestHandler<HttpRequest>(
                new ProxyRequestHandlerAdapter(
                    new LockHandler(),
                    this),
                this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_unlock", false),
            new DelegatedHttpAsyncRequestHandler<HttpRequest>(
                new ProxyRequestHandlerAdapter(
                    new UnlockHandler(),
                    this),
                this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_producer_position", false),
            new DelegatedHttpAsyncRequestHandler<HttpRequest>(
                new ProxyRequestHandlerAdapter(
                    new ProducerPositionHandler(),
                    this),
                this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_producer_drop_position", false),
            new DelegatedHttpAsyncRequestHandler<HttpRequest>(
                new ProxyRequestHandlerAdapter(
                    new ProducerDropPositionHandler(),
                    this),
                this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_producer_lock", false),
            new DelegatedHttpAsyncRequestHandler<HttpRequest>(
                new ProxyRequestHandlerAdapter(
                    new ProducerLockHandler(),
                    this),
                this,
                requestParseExecutor),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_producer_unlock", false),
            new ProducerUnlockHandler(), RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_producer_list", false),
            new ProducerListHandler(), RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_refresh_lock", false),
            new RefreshLockHandler(), RequestHandlerMapper.GET);

        SyncStatusHandler syncStatusHandler = new SyncStatusHandler();
        consumerWaitersExpirer = new ConsumerWaitersExpirer();

        registerStater(
            new WritesStater(
                writeTimesFrameQueues,
                statusWriteTimesFrameQueues));
    }

    public AsyncClient httpClient() {
        return httpClient;
    }

    public SearchMap searchMap() {
        return this.searchMap;
    }

    protected static long packTimeNCount(final long time, final int count) {
        return time | ((long) count << COUNT_SHIFT);
    }

    protected static long unpackTime(final long packed) {
        return packed & TIME_MASK;
    }

    protected static int unpackCount(final long packed) {
        return (int) (packed >>> COUNT_SHIFT);
    }

    private <E extends Exception> void statTimings(
        final StatsConsumer<? extends E> statsConsumer,
        final TimeFrameQueue<Long> queue,
        final String prefix)
        throws E
    {
        long[] timeDistribution = new long[staterTimeBoundaries.length + 1];
        int timeCounter = 0;
        int totalCount = 0;
        loopTimes:
        for (long packed: queue) {
            long time = unpackTime(packed);
            int count = unpackCount(packed);
            timeCounter++;
            totalCount += count;
            for (int boundaryNumber = 0;
                    boundaryNumber < staterTimeBoundaries.length;
                    ++boundaryNumber)
            {
                if (time < staterTimeBoundaries[boundaryNumber]) {
                    ++timeDistribution[boundaryNumber];
                    continue loopTimes;
                }
            }
            ++timeDistribution[staterTimeBoundaries.length];
        }
        for (int i = 0; i < staterTimeBoundaries.length; ++i) {
            statsConsumer.stat(
                prefix + "-time-less-" + staterTimeBoundaries[i] + MS_AMMM,
                timeDistribution[i]);
        }
        statsConsumer.stat(
            prefix + "-time-greater-"
            + staterTimeBoundaries[staterTimeBoundaries.length - 1] + MS_AMMM,
            timeDistribution[timeDistribution.length - 1]);
        statsConsumer.stat(prefix + "-times-counter_ammm", timeCounter);
        statsConsumer.stat(prefix + "-total-items-count_ammm", totalCount);
        statsConsumer.stat(prefix + "-total-items-count_axxx", totalCount);
    }

//    public TimeFrameQueue<Long> writeTimesQueue(final int zooPort) {
//        TimeFrameQueue<Long> queue = writeTimesFrameQueues.get(zooPort);
//        if (queue == null) {
//            TimeFrameQueue<Long> newQueue =
//                new TimeFrameQueue<>(config.metricsTimeFrame());
//            queue = writeTimesFrameQueues.putIfAbsent(zooPort, newQueue);
//            if (queue == null) {
//                queue = newQueue;
//            }
//        }
//        return queue;
//    }
//
//    public TimeFrameQueue<Long> statusWriteTimesQueue(final int zooPort) {
//        TimeFrameQueue<Long> queue = statusWriteTimesFrameQueues.get(zooPort);
//        if (queue == null) {
//            TimeFrameQueue<Long> newQueue =
//                new TimeFrameQueue<>(config.metricsTimeFrame());
//            queue = statusWriteTimesFrameQueues.putIfAbsent(zooPort, newQueue);
//            if (queue == null) {
//                queue = newQueue;
//            }
//        }
//        return queue;
//    }

    public TimeFrameQueue<Long> writeTimesQueue(final String zooHost) {
        TimeFrameQueue<Long> queue = writeTimesFrameQueues.get(zooHost);
        if (queue == null) {
            TimeFrameQueue<Long> newQueue =
                new TimeFrameQueue<>(config.metricsTimeFrame());
            queue = writeTimesFrameQueues.putIfAbsent(zooHost, newQueue);
            if (queue == null) {
                queue = newQueue;
            }
        }
        return queue;
    }

    public TimeFrameQueue<Long> statusWriteTimesQueue(final String zooHost) {
        TimeFrameQueue<Long> queue = statusWriteTimesFrameQueues.get(zooHost);
        if (queue == null) {
            TimeFrameQueue<Long> newQueue =
                new TimeFrameQueue<>(config.metricsTimeFrame());
            queue = statusWriteTimesFrameQueues.putIfAbsent(zooHost, newQueue);
            if (queue == null) {
                queue = newQueue;
            }
        }
        return queue;
    }

    public BigDecimal getPosition(final HttpRequest request)
        throws HttpException
    {
        Header h = request.getFirstHeader(YandexHeaders.PRODUCER_POSITION);
        if (h != null && h.getValue() != null) {
            try {
                return new BigDecimal(h.getValue());
            } catch (NumberFormatException e) {
            throw new ServerException(HttpStatus.SC_BAD_REQUEST,
                "Can't parse <producer-position> header:"
                + " invalid number: " + e.getMessage(), e);
            }
        }
        return null;
    }

    public ZooBigDecimalNode getPositionNode(final HttpRequest request,
        final String service, final long prefix)
        throws HttpException
    {
        Header h = request.getFirstHeader("producer-name");
        if (h != null && h.getValue() != null) {
            return getPositionNode(service, h.getValue(), prefix);
        }
        return null;
    }

    public ZooBigDecimalNode getPositionNode(final String service,
        final String producerName, final long prefix)
        throws HttpException
    {
        if (producerName == null) {
            return null;
        }
        String varName = '/' + service + "/producer_position_" + producerName;
        ZooBigDecimalNode node = getBigDecimalNode(service, varName, prefix);
        return node;
    }

    private ZooBigDecimalNode getPositionNode(final String service,
        final String producerName, final String zk)
        throws HttpException
    {
        String varName = '/' + service + "/producer_position_" + producerName;
        ZooBigDecimalNode node = getBigDecimalNode(service, varName, zk);
        return node;
    }

    private ConcurrentMap<String, ZooNodeMapping> getNodeMappings(
        final String service, final String varName)
    {
        final String key = service + '_' + varName;
        ConcurrentHashMap<String, ZooNodeMapping> serverMap =
            mappedNodes.get(key);
        if (serverMap == null) {
            ConcurrentHashMap<String, ZooNodeMapping> newServerMap =
                new ConcurrentHashMap<String, ZooNodeMapping>();
            serverMap = mappedNodes.putIfAbsent(key, newServerMap);
            if (serverMap == null) {
                serverMap = newServerMap;
            }
        }
        return serverMap;
    }

    private ZooNodeMapping getNodeMapping(
        final String service,
        final String varName, final long prefix)
        throws HttpException
    {
        ConcurrentMap<String, ZooNodeMapping> serverMap =
            getNodeMappings(service, varName);
        String zkServer = searchMap.getZooKeeper(service, prefix);
        ZooNodeMapping map = serverMap.get(zkServer);
        if (map == null) {
            ZooConnection conn = varMapConnections.get(zkServer);
            if (conn == null) {
                throw new ServiceUnavailableException(
                    "No zookeeper mapped for " + varName + '@' + service);
            }
            ZooNodeMapping newMap = new ZooNodeMapping(conn, varName);
            map = serverMap.putIfAbsent(zkServer, newMap);
            if (map == null) {
                map = newMap;
                map.map();
            }
        }
        return map;
    }

    private ZooBigDecimalNode getBigDecimalNode(
        final String service,
        final String varName,
        final String zkServer)
        throws HttpException
    {
        ConcurrentMap<String, ZooNodeMapping> serverMap =
            getNodeMappings(service, varName);
        ZooNodeMapping map = serverMap.get(zkServer);
        if (map == null) {
            ZooConnection conn = varMapConnections.get(zkServer);
            if (conn == null) {
                throw new ServiceUnavailableException(
                    "No zookeeper mapped for " + varName + '@' + service);
            }
            ZooNodeMapping newMap = new ZooBigDecimalNode(conn, varName);
            newMap.setAutoCreate(true);
            newMap.setPersistent(true);
            newMap.setLogger(logger().replacePrefix("BigDecimalNode<" + varName
                + '@' + service + '>'));
            map = serverMap.putIfAbsent(zkServer, newMap);
            if (map == null) {
                map = newMap;
                map.map();
            }
        } else {
            if (!(map instanceof ZooBigDecimalNode)) {
                throw new HttpException(
                    "Variable <" + varName + '@' + service
                        + "> is not of BigDecimal type");
            }
        }
        return (ZooBigDecimalNode) map;
    }

    private ZooBigDecimalNode getBigDecimalNode(
        final String service,
        final String varName,
        final long prefix)
        throws HttpException
    {
        String zkServer = searchMap.getZooKeeper(service, prefix);
        return getBigDecimalNode(service, varName, zkServer);
    }

    private ZooLock createLock(
        final String service,
        final String name,
        final Logger logger,
        final boolean autoRemove)
        throws HttpException
    {
        String zkServer = searchMap.getZooKeeper(service, name.hashCode());
        if (logger.isLoggable(Level.INFO)) {
            logger.info(
                "GetLock: name=" + name + ", code=" + name.hashCode()
                + ", server=" + zkServer);
        }
        ZooConnection conn = varMapConnections.get(zkServer);
        if (conn == null) {
            throw new ServiceUnavailableException(
                "No zookeeper mapped for " + name + '@' + service);
        }
        final ZooLock lock = new ZooLock(conn, name, 30000);
        if (autoRemove) {
            lock.addExpireCallback(
                new LockExpireCallback(service, name, lock));
        }
        return lock;
    }

    private void removeLock(
        final String service,
        final String name,
        final ZooLock lock)
    {
        final String lockName = service + '_' + name;
        zooLocks.remove(lockName, lock);
    }

    private ZooLock getLock(
        final String service,
        final String name,
        final boolean autoExpire,
        final boolean autoRemove)
        throws HttpException
    {
        String lockName = service + '_' + name;
        ZooLock lock = zooLocks.get(lockName);
        if (lock == null) {
            ZooLock newLock = createLock(service, name, logger(), autoRemove);
            lock = zooLocks.putIfAbsent(lockName, newLock);
            if (lock == null) {
                lock = newLock;
                lock.setAutoExpire(autoExpire);
                lock.setLogger(logger().replacePrefix("ZooLock<" + name + '>'));
            }
        }
        return lock;
    }

    private void checkLock(
        final HttpRequest request,
        final String service,
        final boolean autoRemove)
        throws HttpException
    {
        Header h = request.getFirstHeader("lockid");
        if (h != null) {
            String lockId = h.getValue();
            checkLock(service, lockId, autoRemove);
        }
    }

    private void checkLock(
        final String service,
        final String lockId,
        final boolean autoRemove)
        throws HttpException
    {
        int sep = lockId.indexOf('@');
        String lockName = lockId.substring(0, sep);
        String id = lockId.substring(sep + 1);
        String varName = '/' + service + "/producer_lock_" + lockName;
        ZooLock lock = getLock(service, varName, true, autoRemove);
        try {
            if (!lock.isLocked()) {
                throw new ForbiddenException("Can't obtain lock <" + lockName + '>');
            } else {
                if (!id.equals(lock.id())) {
                    throw new ForbiddenException("Lock expired <" + lockName + '>');
                }
            }
        } catch (ZooException e) {
            throw new ForbiddenException("ZooLock failed", e);
        }
    }

    public void parseRequestAsync(
        final QueueRequestHandlerBase.ParseRequestCallback callback)
    {
        try {
            requestParseExecutor.execute(new RequestParseTask(this, callback));
        } catch (RejectedExecutionException e) {
            callback.failed(
                new HttpException(
                    "Request parsing executor exception", e));
        }
    }

    public QueueRequest parseRequest(final ProxySession session)
        throws IOException, HttpException
    {
        Logger logger = session.logger();
        CgiParams cgiParams = session.params();
        HttpRequest request = session.request();

        String prefixStr = null;
        boolean doWait = false;
        Long prefix = null;

        String service = null;
        Header h = request.getFirstHeader(YandexHeaders.SERVICE);
        if (h != null && h.getValue() != null) {
            service = h.getValue();
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("Got service name from headers: " + service);
            }
        }
        if (service == null) {
            service = cgiParams.getString("service");
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("Got service name from uri: " + service);
            }
        }

        h = request.getFirstHeader(YandexHeaders.ZOO_SHARD_ID);
        if (h != null) {
            prefixStr = h.getValue();
        }
        if (prefixStr == null) {
            h = request.getFirstHeader("shard");
            if (h != null) {
                prefixStr = h.getValue();
            }
        }
        if (prefixStr == null) {
            prefixStr = cgiParams.getString("shard", null);
        }
        if (prefixStr == null) {
            prefixStr = cgiParams.getString("prefix", null);
        }

        h = request.getFirstHeader("wait");
        if (h != null && h.getValue() != null) {
            doWait = Boolean.parseBoolean(h.getValue());
        } else {
            doWait = cgiParams.getBoolean("wait", false);
        }

        int waitTimeout = cgiParams.getInt("timeout", this.waitTimeout);

        if (prefixStr != null) {
            prefix = Long.parseLong(prefixStr);
        }

        checkLock(request, service, false);

        ContentType contentType = null;

        Header contentTypeHeader = request.getFirstHeader(HTTP.CONTENT_TYPE);
        if (contentTypeHeader != null) {
            try {
                contentType = ContentType.parse(contentTypeHeader.getValue());
            } catch (org.apache.http.ParseException
                | UnsupportedCharsetException e)
            {
                throw new BadRequestException("Bad content-type header");
            }
        }

        final boolean multiPart;
        if (contentType != null
            && contentType.getMimeType().equals("multipart/mixed"))
        {
            multiPart = true;
        } else {
            multiPart = false;
        }

        h = request.getFirstHeader("producer-name");
        final String producerName;
        if (h != null && h.getValue() != null) {
            producerName = h.getValue();
        } else {
            producerName = null;
        }

        return new QueueRequest(
            session,
            request,
            logger,
            service,
            producerName,
            prefix,
            doWait,
            waitTimeout,
            multiPart,
            cgiParams);
    }

    public boolean checkPosition(
        final ZooBigDecimalNode node,
        final BigDecimal position,
        final long prefix)
        throws HttpException
    {
        BigDecimal currentPos = node.cachedValue();

        if (currentPos == null) {
            try {
                currentPos = node.value();
            } catch (ZooException e) {
                throw new ServerException(
                    HttpStatus.SC_INTERNAL_SERVER_ERROR,
                    "Can't query producer position counter",
                    e);
            }
        } else {
            if (logger().isLoggable(Level.FINE)) {
                logger().fine(
                    "CheckPosition: desired=" + position
                        + ", cached=" + currentPos.toPlainString());
            }
        }

        if (currentPos != null) {
            if (logger().isLoggable(Level.FINE)) {
                logger().fine(
                    "CheckPosition: longValue="
                        + currentPos.toPlainString());
            }
            if (currentPos.compareTo(position) >= 0) {
                return false;
            }
        } else {
            try {
                if (!node.exists() || node.data() == null)
//                    || (node.data() != null && node.data() == -1))
                {
                    final BigDecimal initialValue =
                        position.subtract(new BigDecimal(1));
                    if (logger().isLoggable(Level.FINE)) {
                        logger().fine("CheckPosition: setBigDecimalValue="
                            + initialValue.toPlainString());
                    }
                    node.setValue(initialValue);
                }
            } catch (ZooException e) {
                throw new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR,
                    "Can't create producer position "
                    + "counter with initial value of "
                    + (position.subtract(new BigDecimal(1)).toPlainString()),
                    e);
            }
        }
        return true;
    }

    public void updatePositions(final StatusMessage message)
        throws HttpException
    {
        String zooKeeper =
            searchMap.getZooKeeper(message.service(),
                message.shard());
        if (zooKeeper == null) {
            throw new MethodNotSupportedException(
                "No routes found for service: " + message.service()
                + " and prefix " + message.shard());
        }

        final StatusQueue queue = statusQueues.get(zooKeeper);
        queue.enqueue(message);
    }

    //TODO: make it multimessage capable
    private void httpCheckDuplicateAndSend(
        final QueueRequest queueRequest,
        final FutureCallback<QueueMessage> responseCallback,
        final MessageGrouper grouper)
    {
        final HttpHost zooHost =
            grouper.connection().currentServer().httpHost();
        final int shard =
            (int) (queueRequest.commonPrefix() % SearchMap.SHARDS_COUNT);
        final QueueMessage msg = queueRequest.messages().get(0);
        final String hash = msg.hash();
        BasicAsyncRequestProducerGenerator request =
            new BasicAsyncRequestProducerGenerator("/getPathByHash?service="
                + queueRequest.service() + "&shard=" + shard
                + "&hash=" + hash
                + "&no404"
                + "&check-disk=" + (!queueRequest.memoryCheckOnly()));

        httpClient.execute(
            zooHost,
            request,
            new StatusCheckAsyncResponseConsumerFactory<IntPair<String>>(
                x -> x < HttpStatus.SC_BAD_REQUEST
                || x == HttpStatus.SC_NOT_FOUND,
                new StatusCodeAsyncConsumerFactory<String>(
                    AsyncStringConsumerFactory.INSTANCE)),
            queueRequest.session().listener()
                .createContextGeneratorFor(httpClient),
            new HttpCheckDupCallback(
                queueRequest,
                responseCallback,
                grouper,
                msg));
    }

    public void sendMessages(
        final QueueRequest request,
        final FutureCallback<QueueMessage> responseCallback)
        throws HttpException
    {
        if (request.commonPrefix() != null) {
            String zooKeeper =
                searchMap.getZooKeeper(
                    request.service(),
                    request.commonPrefix());
            if (zooKeeper == null) {
                throw new MethodNotSupportedException(
                    "No routes found for service: " + request.service()
                    + " and prefix " + request.commonPrefix());
            }


            for (QueueMessage message : request.messages()) {
                message.setCallback(responseCallback);
            }

            MessageGrouper grouper = groupers.get(zooKeeper);
            if (request.checkDuplicate() && request.messages().size() == 1) {
                final QueueMessage msg = request.messages().get(0);
                httpCheckDuplicateAndSend(
                    request,
                    responseCallback,
                    grouper);
                return;
            }

            grouper.enqueue(request.messages());
        } else {
            sendMultiPrefix(request, responseCallback);
        }
    }

    private void sendMultiPrefix(
        final QueueRequest request,
        final FutureCallback<QueueMessage> responseCallback)
        throws HttpException
    {
        for (QueueMessage message : request.messages()) {
            //check all messages are valid
            String zooKeeper =
                searchMap.getZooKeeper(request.service(), message.prefix());
            if (zooKeeper == null) {
                throw new MethodNotSupportedException(
                    "No routes found for service: " + request.service()
                    + " and prefix " + message.prefix());
            }
            message.setCallback(responseCallback);
        }
        sendMultiPrefixBatch(request, responseCallback, 0);
    }

    private void sendMultiPrefixBatch(
        final QueueRequest request,
        final FutureCallback<QueueMessage> responseCallback,
        final int batchStart)
    {
        QueueMessage first = request.messages().get(batchStart);

        String firstZooKeeper =
            searchMap.getZooKeeper(request.service(),
                first.prefix());

        if (request.logger().isLoggable(Level.FINE)) {
            request.logger().fine("Grouping batch for zoo: " + firstZooKeeper);
        }

        List<QueueMessage> messages = request.messages();
        int batchEnd;

        for (batchEnd = batchStart; batchEnd < messages.size(); batchEnd++) {
            QueueMessage message = messages.get(batchEnd);
            String nextZooKeeper =
                searchMap.getZooKeeper(request.service(),
                    message.prefix());
            if (!nextZooKeeper.equals(firstZooKeeper)) {
                break;
            }
        }
        List<QueueMessage> batch = messages.subList(batchStart, batchEnd);

        MessageGrouper grouper = groupers.get(firstZooKeeper);

        if (batchEnd != messages.size()) {
            if (request.logger().isLoggable(Level.FINE)) {
                request.logger().fine("Setting nextBatchStart to " + batchEnd);
            }
            QueueMessage lastBatchMessage = batch.get(batch.size() - 1);
            lastBatchMessage.setCallback(
                new NextBatchCallbackProxy(
                    request,
                    responseCallback,
                    batchEnd));
        }
        if (request.logger().isLoggable(Level.FINE)) {
            request.logger().fine("Grouped " + batch.size() + " messages");
        }
        grouper.enqueue(batch);
    }

    public String pathFor(final String service, final long prefix) {
        int shard = SearchMap.getShard(prefix);
        return '/' + service + '/' + shard + "/forward/queue";
    }

    public void checkMessageVersion(
        final QueueRequest request,
        final QueueMessage message,
        final FutureCallback<QueueMessage> callback)
    {
        String zooKeeper =
            searchMap.getZooKeeper(request.service(),
                request.commonPrefix());
        ZooConnection conn = varMapConnections.get(zooKeeper);
        final StatCallback statCallback = new StatCallback() {
            @Override
            public void dataImpl(final Stat stat) {
                message.version(stat.getVersion());
                callback.completed(message);
            }
            @Override
            public void errorImpl(final ZooException e){
                callback.failed(e);
            }
            @Override
            public void dataChangedImpl(){
            }
        };
        conn.exists(message.path(), statCallback);
    }

    public void readBackendResponse(
        final QueueRequest request,
        final QueueMessage message,
        final FutureCallback<QueueMessage> callback)
    {
        String zooKeeper =
            searchMap.getZooKeeper(request.service(),
                request.commonPrefix());
        ZooConnection conn = varMapConnections.get(zooKeeper);
        final ByteArrayCallback dataCallback = new ByteArrayCallback() {
            @Override
            public void dataImpl(final byte[] data) {
                try {
                    final HttpMessage response =
                        SerializeUtils.deserializeHttpMessage(data);
                    if (!(response instanceof ErrorMessage)) {
                        callback.failed(
                            new BadGatewayException(
                                "Unhandled backend response message type: "
                                + response.getClass().getName()));
                    } else {
                        message.response((ErrorMessage)response);
                        callback.completed(message);
                    }
                } catch (Exception e) {
                    callback.failed(
                        new BadGatewayException(
                            "Unhandled backend response decoding exception",
                            e));
                }
            }
            @Override
            public void errorImpl(final ZooException e){
                callback.failed(e);
            }
            @Override
            public void dataChangedImpl(){
            }
        };
        conn.data(message.path(), dataCallback, false);
    }

    public void waitConsumer(
        final QueueRequest request,
        final QueueMessage message,
        final FutureCallback<QueueMessage> callback)
    {
        Status[] serviceShards = servicesStatus.get(request.service());
        int shard = SearchMap.getShard(request.commonPrefix());
        Status shardStatus = serviceShards[shard];
        synchronized(shardStatus) {
            if (shardStatus.currentId >= message.queueId()) {
                callback.completed(message);
            } else {
                ConsumerWaiter cw =
                    new ConsumerWaiter(
                        message,
                        request.requestStartTime(),
                        request.timeout(),
                        callback);
                shardStatus.waiters.add(cw);
                consumerWaiters.offer(cw);
            }
        }
    }

    @Override
    public void start() throws IOException {
        super.start();
        httpClient.start();
        consumerWaitersExpirer.start();
        for (ZooConnection zc : zooConnections) {
            zc.connectAsync();
        }
        for (MessageGrouper grouper : groupers.values()) {
            grouper.start();
        }
        for (StatusQueue queue : statusQueues.values()) {
            queue.start();
        }
        statusConsumer.init();
    }

    private static IniConfig genConfig( int port, int workers, int timeout ) throws IOException, ConfigException
    {
        return new IniConfig( new StringReader( "port=" + port + "\nworkers.min=" + workers + "\nbacklog=10000" + "\ntimeout=" + timeout + "\nconnections=" + (workers*20) ) );
    }

    @Override
    public void notifyStatus( String service, int shard, String backend, long queueId )
    {
        Status[] serviceShards = servicesStatus.get( service );
        Status shardStatus = serviceShards[shard];
        synchronized( shardStatus )
        {
            if( shardStatus.currentId < queueId )
            {
                shardStatus.currentId = queueId;
                shardStatus.backend = backend;
                Iterator<ConsumerWaiter> waiters = shardStatus.waiters.iterator();
                while (waiters.hasNext()) {
                    ConsumerWaiter waiter = waiters.next();
                    if (waiter.message.queueId() <= queueId) {
                        waiter.completed();
                        waiters.remove();
                    }
                }
            }
        }
    }

    private abstract class GetRequestHandlerBase implements HttpRequestHandler {
        private StringEntity describeException(Exception e)
            throws UnsupportedEncodingException
        {
            StringBuilderWriter sbw = new StringBuilderWriter();
            e.printStackTrace(sbw);
            return new StringEntity(sbw.toString());
        }

        @Override
        public void handle(HttpRequest request, HttpResponse response,
            HttpContext context) throws HttpException, IOException
        {
            try {
                handle(
                    new CgiParams(request),
                    request,
                    response,
                    (Logger) context.getAttribute(LOGGER));
            } catch (InterruptedException e) {
                response.setStatusCode(HttpStatus.SC_GATEWAY_TIMEOUT);
                response.setEntity(describeException(e));
            } catch (KeeperException e) {
                response.setStatusCode(HttpStatus.SC_SERVICE_UNAVAILABLE);
                response.setEntity(describeException(e));
            } catch (ParseException e) {
                response.setStatusCode(HttpStatus.SC_BAD_REQUEST);
                response.setEntity(describeException(e));
            }
        }

        public abstract void handle(CgiParams cgiParams, HttpRequest request, HttpResponse response, Logger logger)
            throws HttpException, IOException, InterruptedException,
                KeeperException, ParseException;

    }

    private class SyncStatusHandler extends GetRequestHandlerBase {
        @Override
        public void handle(CgiParams cgiParams, HttpRequest request, HttpResponse response, Logger logger)
            throws HttpException, IOException, InterruptedException,
                KeeperException, ParseException
        {
            final String service = cgiParams.getString("service");
            if (!config().consumerServices().isEmpty()
                && !config().consumerServices().contains(service))
            {
                final ZoolooserStatusParserCallback zooCallback =
                    statusProxyHandler.sendProxyRequests(
                        logger,
                        request,
                        cgiParams,
                        null);
                final HttpResponse zooResponse = zooCallback.get();
                response.setStatusLine(zooResponse.getStatusLine());
                response.setEntity(zooResponse.getEntity());
//                response.setLocale(zooResponse.getLocale());
                return;
            }
            boolean cached = false;
            boolean allowCached = cgiParams.getBoolean("allow_cached", false);
            String[] freshBackend = statusConsumer.getFreshestBackends( cgiParams.getString("service"), cgiParams.getLong("prefix") );
            if( freshBackend == null )
            {
                if (!allowCached) {
                    throw new ServerException( 500, "No authoritative status. Disconnected from quorum. ");
                }
                cached = true;
                freshBackend = statusConsumer.getFreshestBackendsCached( cgiParams.getString("service"), cgiParams.getLong("prefix") );
                if( freshBackend == null )
                {
                    throw new ServerException( 500, "No data in cache. Disconnected from quorum. ");
                }
            }
            StringBuilder sb = new StringBuilder();
            for (String fb: freshBackend) {
                sb.append(fb);
                sb.append('\n');
            }
            if (cached) {
                response.setStatusCode(HttpStatus.SC_NON_AUTHORITATIVE_INFORMATION);
            }
            response.setEntity(new StringEntity(new String(sb)));
        }

    }

    private class StatusHandler implements ProxyRequestHandler {
        @Override
        public void handle(final ProxySession session)
            throws HttpException, IOException
        {
            CgiParams cgiParams = session.params();
            final String service = cgiParams.getString("service");
            if (!config().consumerServices().isEmpty()
                && !config().consumerServices().contains(service))
            {
                statusProxyHandler.handle(session);
                return;
            }
            boolean cached = false;
            boolean allowCached = cgiParams.getBoolean("allow_cached", false);
            String[] freshBackend =
                statusConsumer.getFreshestBackends(
                    cgiParams.getString("service"),
                    cgiParams.getLong("prefix"));
            if (freshBackend == null) {
                if (!allowCached) {
                    session.handleException(
                        HttpExceptionConverter.toHttpException(
                            new ServerException(
                                HttpStatus.SC_INTERNAL_SERVER_ERROR,
                                "No authoritative status. "
                                + "Disconnected from quorum")));
                } else {
                    cached = true;
                    freshBackend = statusConsumer.getFreshestBackendsCached(
                        cgiParams.getString("service"),
                        cgiParams.getLong("prefix"));
                    if (freshBackend == null) {
                        session.handleException(
                            HttpExceptionConverter.toHttpException(
                                new ServerException(
                                    HttpStatus.SC_INTERNAL_SERVER_ERROR,
                                    "No data in cache. "
                                    + "Disconnected from quorum")));
                    }
                }
            }
            StringBuilder sb = new StringBuilder();
            for (String fb: freshBackend) {
                sb.append(fb);
                sb.append('\n');
            }
            int code = HttpStatus.SC_OK;
            if (cached) {
                code = HttpStatus.SC_NON_AUTHORITATIVE_INFORMATION;
            }
            session.response(code, new String(sb));
        }
    }

    private class GetNodeDataHandler implements ProxyRequestHandler {
        @Override
        public void handle(final ProxySession session)
            throws HttpException, IOException
        {
            CgiParams cgiParams = session.params();
            String service = cgiParams.getString("service");
            String varName = cgiParams.getString("path");
            Long prefix = cgiParams.getLong("shard", null);
            if (prefix == null) {
                prefix = cgiParams.getLong("prefix");
            }
            ZooNodeMapping map = getNodeMapping(service, varName, prefix);
            session.response(
                HttpStatus.SC_OK,
                new ByteArrayEntity(map.data()));
        }

    }

    private class SetNodeDataHandler implements ProxyRequestHandler {
        @Override
        public void handle(final ProxySession session)
            throws HttpException, IOException
        {
            CgiParams cgiParams = session.params();
            String service = cgiParams.getString("service");
            String varName = cgiParams.getString("path");
            Long prefix = cgiParams.getLong("shard", null);
            if (prefix == null) {
                prefix = cgiParams.getLong("prefix");
            }
            byte[] data = cgiParams.getString("data", "").getBytes();
            ZooNodeMapping map = getNodeMapping(service, varName, prefix);
            logger().info("SET DATA START");
            map.setData(data);
            logger().info("SET DATA OK");
            session.response(HttpStatus.SC_OK, "OK");
        }
    }

    private class LockHandler implements ProxyRequestHandler {
        @Override
        public void handle(final ProxySession session)
            throws HttpException, IOException
        {
            final CgiParams cgiParams = session.params();
            final String service = cgiParams.getString("service");
            final String name = cgiParams.getString("name");
            final int timeout = cgiParams.getInt("timeout", 30000);
            final boolean force = cgiParams.getBoolean("force", false);
            final String id = cgiParams.getString("id", null);
            final String lockName = '/' + service + "/lock-" + name;
            final ZooLock lock =
                createLock(service, lockName, session.logger(), true);
            if (id != null) {
                lock.id(id);
            }
            if (timeout != 0) {
                lock.setTimeout(timeout);
                lock.setAutoUpdateOnSync(true);
                lock.setAutoExpire(true);
            } else {
                lock.setAutoExpire(false);
            }
            if (force) {
                lock.forceLock(false);
            }
            final boolean locked = lock.lockAndRefresh(false);
            if (!locked) {
                removeLock(service, lockName, lock);
                session.response(HttpStatus.SC_FORBIDDEN, lock.lockData());
            } else {
                session.response(HttpStatus.SC_OK, lock.id());
            }
        }

    }

    private class UnlockHandler implements ProxyRequestHandler {
        @Override
        public void handle(final ProxySession session)
            throws HttpException, IOException
        {
            final CgiParams cgiParams = session.params();
            final String service = cgiParams.getString("service");
            final String name = cgiParams.getString("name");
            final boolean force = cgiParams.getBoolean("force", false);
            final String id = cgiParams.getString("id", null);
            final String lockName = '/' + service + "/lock-" + name;
            final ZooLock lock =
                createLock(service, lockName, session.logger(), true);
            if (id != null) {
                lock.id(id);
            }
            if (!lock.isLocked(false) && !force) {
                session.response(
                    HttpStatus.SC_FORBIDDEN,
                    "lock was not held, can't unlock, lockData="
                    + lock.lockData());
                return;
            } else {
                try {
                    if (force) {
                        lock.forceUnlock();
                    } else {
                        lock.unlock();
                    }
                    session.response(HttpStatus.SC_OK, "OK");
                } catch (ZooNoNodeException e) {
                    //it is ok
                } catch (ZooException e) {
                    throw new ServerException(
                        HttpStatus.SC_INTERNAL_SERVER_ERROR,
                        "Unlock failed",
                        e);
                }
            }
        }
    }

    private class ProducerPositionHandler implements ProxyRequestHandler {
        @Override
        public void handle(final ProxySession session)
            throws HttpException, IOException
        {
            CgiParams cgiParams = session.params();
            String service = cgiParams.getString("service");
            String varName = cgiParams.getString("producer-name");
            boolean truncate = cgiParams.getBoolean("truncate", true);
            Long prefix = cgiParams.getLong("shard", null);
            if (prefix == null) {
                prefix = cgiParams.getLong("prefix", null);
            }
            if (session.logger().isLoggable(Level.FINE)) {
                session.logger().fine("ProducerPositionHandler: prefix="
                    + prefix);
            }
            final BigDecimal responseValue;
            if (prefix == null) {
                Set<String> zooKeepers =
                    searchMap.getServiceZooKeepers(service);
                BigDecimal maxValue = PRODUCER_POSITION_NO_VALUE;
                for (String zk : zooKeepers) {
                    final long start = TimeSource.INSTANCE.currentTimeMillis();
                    if (session.logger().isLoggable(Level.FINE)) {
                        session.logger().fine(
                            "Getting position <" + varName + "> from: " + zk);
                    }
                    ZooBigDecimalNode node =
                        getPositionNode(service, varName, zk);
                    BigDecimal value = node.value();
                    if (session.logger().isLoggable(Level.FINE)) {
                        session.logger().fine("got pos for <" + varName
                            + "> in: " + (TimeSource.INSTANCE.currentTimeMillis() - start)
                            + " ms");
                    }
                    if (value == null) {
                        continue;
                    }
                    if (maxValue.compareTo(value) < 0) {
                        maxValue = value;
                    }
                }
                responseValue = maxValue;
            } else {
                ZooBigDecimalNode node =
                    getPositionNode(service, varName, prefix);
                BigDecimal value = node.value();
                if (value == null) {
                    responseValue = PRODUCER_POSITION_NO_VALUE;
                } else {
                    responseValue = value;
                }
            }
            final String response;
            if (truncate) {
                response = Long.toString(responseValue.longValue());
            } else {
                response = responseValue.toPlainString();
            }
            session.response(HttpStatus.SC_OK, response);
        }
    }

    private class ProducerDropPositionHandler implements ProxyRequestHandler {
        @Override
        public void handle(final ProxySession session)
            throws HttpException, IOException
        {
            CgiParams cgiParams = session.params();
            String service = cgiParams.get(
                "service",
                NonEmptyValidator.INSTANCE);
            String producerName = cgiParams.get(
                "producer-name",
                NonEmptyValidator.INSTANCE);
            int positionsCount = cgiParams.get(
                "positions-count",
                PositiveIntegerValidator.INSTANCE);
            int lockTimeout = cgiParams.get(
                "session-timeout",
                10000,
                PositiveIntegerValidator.INSTANCE);
            Logger logger = session.logger();
            String lockName = '/' + service + "/producer_lock_" + producerName;
            try (ZooLock lock = createLock(service, lockName, logger, false)) {
                lock.setTimeout(lockTimeout);
                lock.forceLock(true);
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("Lock " + lockName + " obtained " + lock.id());
                }
                for (ZooConnection conn: varMapConnections.values()) {
                    if (logger.isLoggable(Level.INFO)) {
                        logger.info("Dropping positions for connection: " + conn);
                    }
                    for (int i = 0; i < positionsCount; ++i) {
                        String path =
                            '/' + service + "/producer_position_"
                            + producerName + ':' + i;
                        if (logger.isLoggable(Level.INFO)) {
                            logger.info("Removing path: " + path);
                        }
                        new ZooNodeMapping(conn, path).delete();
                    }
                }
            } catch (ZooException e) {
                throw new ServiceUnavailableException(e);
            }
            session.response(HttpStatus.SC_OK);
        }
    }

    private class ProducerLockHandler implements ProxyRequestHandler {
        @Override
        public void handle(final ProxySession session)
            throws HttpException, IOException
        {
            CgiParams cgiParams = session.params();
            String service = cgiParams.getString("service");
            String prodName = cgiParams.getString("producer-name");
            int lockTimeout = cgiParams.getInt("session-timeout", 10000);
            String varName = '/' + service + "/producer_lock_" + prodName;
            Long blockTime = zooLocksBlock.get(service + "_" + varName);
            if (blockTime != null && blockTime > TimeSource.INSTANCE.currentTimeMillis()) {
                session.handleException(
                    HttpExceptionConverter.toHttpException(
                        new HttpException(
                            "Producer lock failed: blocked until: "
                                + blockTime)));
            } else {
                ZooLock lock = getLock(service, varName, true, false);
                lock.setTimeout(lockTimeout);
                if (lock.lock(false)) {
                    session.response(
                        HttpStatus.SC_OK,
                        prodName + '@' + lock.id());
                } else {
                    session.handleException(
                        new HttpException("Producer lock failed"));
                }
            }
        }
    }

    private class ProducerUnlockHandler implements ProxyRequestHandler {
        @Override
        public void handle(final ProxySession session)
            throws HttpException, IOException
        {
            CgiParams cgiParams = session.params();
            String service = cgiParams.getString("service");
            String prodName = cgiParams.getString("producer-name");
            int blockTimeout = cgiParams.getInt("timeout", 10000);
            String varName = '/' + service + "/producer_lock_" + prodName;
            ZooLock lock = getLock(service, varName, true, false);

            zooLocksBlock.put(
                service + '_' + varName,
                TimeSource.INSTANCE.currentTimeMillis() + blockTimeout);

            lock.unlock();
            session.response(HttpStatus.SC_OK, prodName + '@' + lock.id());
        }
    }

    private class ProducerListHandler implements ProxyRequestHandler {
        @Override
        public void handle(
            final ProxySession session)
            throws HttpException, IOException
        {
            CgiParams cgiParams = session.params();
            String service = cgiParams.getString("service");
            StringBuilder sb = new StringBuilder();
            for (Map.Entry<String, ZooLock> entry : zooLocks.entrySet()) {
                String name = entry.getKey();
                String prefix = service + '_';
                if (name.startsWith(prefix)) {
                    ZooLock lock = entry.getValue();
                    if (lock.isLocked()) {
                        sb.append(name.substring(prefix.length()));
                        sb.append('\n');
                    }
                }
            }
            session.response(HttpStatus.SC_OK, new String(sb));
        }
    }

    private class RefreshLockHandler implements ProxyRequestHandler {
        @Override
        public void handle(final ProxySession session)
            throws HttpException, IOException
        {
            CgiParams cgiParams = session.params();
            String service = cgiParams.getString("service");
            String lockId = cgiParams.getString("lockid");
            checkLock(service, lockId, true);
            session.response(HttpStatus.SC_OK);
        }
    }

    private static class Status
    {
        public long currentId = -1;
        public String backend;
        public List<ConsumerWaiter> waiters;

        public Status( long queueId, String backend )
        {
            this.currentId = queueId;
            this.backend = backend;
            waiters = new LinkedList<>();
        }
    }

    private static class ConsumerWaiter implements Delayed {
        public final QueueMessage message;
        public final long startTime;
        public final int timeout;
        public final FutureCallback<QueueMessage> callback;
        private boolean completed = false;
        public ConsumerWaiter(
            final QueueMessage message,
            final long startTime,
            final int timeout,
            final FutureCallback<QueueMessage> callback)
        {
            this.message = message;
            this.startTime = startTime;
            this.timeout = timeout;
            this.callback = callback;
        }

        @Override
        public long getDelay(final TimeUnit unit) {
            final long elapsed =
                TimeSource.INSTANCE.currentTimeMillis() - startTime;
            final long delayMs = timeout - elapsed;
            return unit.convert(delayMs, TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(final Delayed other) {
            return Long.compare(
                getDelay(TimeUnit.MILLISECONDS),
                other.getDelay(TimeUnit.MILLISECONDS));
        }

        public synchronized void expire() {
            if (!completed) {
                callback.failed(new ConsumerWaitTimeout(
                    "Timeout waiting for consuming message: "
                    + message));
            }
        }

        public synchronized void completed() {
            completed = true;
            callback.completed(message);
        }
    }

    private class ConsumerWaitersExpirer extends Thread {
        public ConsumerWaitersExpirer() {
            super("WaitExpirer");
            setDaemon(true);
        }

        @Override
        public void run() {
            while (true) {
                try {
                    ConsumerWaiter cw = consumerWaiters.take();
                    cw.expire();
                } catch (Exception e) {
                    logger().log(
                        Level.SEVERE,
                        "ConsumerWaitersExpirer unhandled exception",
                        e);
                }
            }
        }
    }

    private class NextBatchCallbackProxy
        implements FutureCallback<QueueMessage>
    {
        private final QueueRequest request;
        private final FutureCallback<QueueMessage> nextCallback;
        private final int nextBatchStart;

        public NextBatchCallbackProxy(
            final QueueRequest request,
            final FutureCallback<QueueMessage> nextCallback,
            final int nextBatchStart)
        {
            this.request = request;
            this.nextCallback = nextCallback;
            this.nextBatchStart = nextBatchStart;
        }

        @Override
        public void completed(final QueueMessage message) {
            nextCallback.completed(message);
            if (request.logger().isLoggable(Level.FINE)) {
                request.logger().fine("Finished batch");
            }
            sendMultiPrefixBatch(request, nextCallback, nextBatchStart);
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(Exception e) {
            if (request.logger().isLoggable(Level.FINE)) {
                request.logger().fine("Batch failed");
            }
            nextCallback.failed(e);
        }
    }

    private class HttpCheckDupCallback
        implements FutureCallback<IntPair<String>>
    {
        private final QueueRequest request;
        private final FutureCallback<QueueMessage> responseCallback;
        private final MessageGrouper grouper;
        private final QueueMessage msg;

        HttpCheckDupCallback(
            final QueueRequest request,
            final FutureCallback<QueueMessage> responseCallback,
            final MessageGrouper grouper,
            final QueueMessage msg)
        {
            this.request = request;
            this.responseCallback = responseCallback;
            this.grouper = grouper;
            this.msg = msg;
        }

        @Override
        public void completed(final IntPair<String> result) {
            if (request.logger().isLoggable(Level.INFO)) {
                request.logger().info("Duplicate message early check success: "
                    + "hash=" + msg.hash() + " -> " + result);
            }
            int code = result.first();
            if (code != HttpStatus.SC_OK) {
                failed(null);
            } else {
                if (request.failOnDuplicate()) {
                    msg.failed(
                        new KeeperException.NodeExistsException(result.second()));
                } else {
                    msg.completed(result.second());
                }
            }
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(Exception e) {
            grouper.enqueueMessage(msg);
        }
    }

    private static class RequestParseTask implements Runnable {
        private final Producer producer;
        private final QueueRequestHandlerBase.ParseRequestCallback callback;
        public RequestParseTask(
            final Producer producer,
            final QueueRequestHandlerBase.ParseRequestCallback callback)
        {
            this.producer = producer;
            this.callback = callback;
        }

        @Override
        public void run() {
            try {
                callback.completed(producer.parseRequest(callback.session()));
            } catch (HttpException e) {
                callback.failed(e);
            } catch (Exception e) {
                callback.failed(
                    new HttpException(
                        "Unhandled exception while parsing request", e));
            }
        }
    }

    private class LockExpireCallback implements Runnable {
        private final String service;
        private final String lockName;
        private final ZooLock lock;

        public LockExpireCallback(
            final String service,
            final String lockName,
            final ZooLock lock)
        {
            this.service = service;
            this.lockName = lockName;
            this.lock = lock;
        }

        @Override
        public void run() {
            removeLock(service, lockName, lock);
            lock.setAutoExpire(false);
            lock.unmap();
        }
    }

//    private class WritesStater implements Stater {
//        private final Map<Integer, TimeFrameQueue<Long>> writeTimesFrameQueues;
//        private final Map<Integer, TimeFrameQueue<Long>>
//            statusWriteTimesFrameQueues;
//
//        WritesStater(
//            final Map<Integer, TimeFrameQueue<Long>> writeTimesFrameQueues,
//            final Map<Integer, TimeFrameQueue<Long>>
//                statusWriteTimesFrameQueues)
//        {
//            this.writeTimesFrameQueues = writeTimesFrameQueues;
//            this.statusWriteTimesFrameQueues = statusWriteTimesFrameQueues;
//        }
//
//        @Override
//        public <E extends Exception> void stats(
//            final StatsConsumer<? extends E> statsConsumer)
//            throws E
//        {
//            for (Map.Entry<Integer, TimeFrameQueue<Long>> entry
//                : writeTimesFrameQueues.entrySet())
//            {
//                statTimings(
//                    statsConsumer,
//                    entry.getValue(),
//                    "zoo-write-" + entry.getKey());
//            }
//            for (Map.Entry<Integer, TimeFrameQueue<Long>> entry
//                : statusWriteTimesFrameQueues.entrySet())
//            {
//                statTimings(
//                    statsConsumer,
//                    entry.getValue(),
//                    "zoo-status-write-" + entry.getKey());
//            }
//        }
//    }

    private class WritesStater implements Stater {
        private final Map<String, TimeFrameQueue<Long>> writeTimesFrameQueues;
        private final Map<String, TimeFrameQueue<Long>>
            statusWriteTimesFrameQueues;

        WritesStater(
            final Map<String, TimeFrameQueue<Long>> writeTimesFrameQueues,
            final Map<String, TimeFrameQueue<Long>>
                statusWriteTimesFrameQueues)
        {
            this.writeTimesFrameQueues = writeTimesFrameQueues;
            this.statusWriteTimesFrameQueues = statusWriteTimesFrameQueues;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            for (Map.Entry<String, TimeFrameQueue<Long>> entry
                : writeTimesFrameQueues.entrySet())
            {
                statTimings(
                    statsConsumer,
                    entry.getValue(),
                    "zoo-write-" + entry.getKey());
            }
            for (Map.Entry<String, TimeFrameQueue<Long>> entry
                : statusWriteTimesFrameQueues.entrySet())
            {
                statTimings(
                    statsConsumer,
                    entry.getValue(),
                    "zoo-status-write-" + entry.getKey());
            }
        }
    }
}

