package ru.yandex.dispatcher.producer;

import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.nio.charset.UnsupportedCharsetException;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.RequestLine;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.MimeConfig;
import org.apache.james.mime4j.stream.MimeTokenStream;
import org.apache.xerces.impl.io.UTF8Reader;
import org.apache.zookeeper.CreateMode;
import org.json.simple.parser.JSONParser;

import ru.yandex.dispatcher.common.HttpMessage;
import ru.yandex.dispatcher.common.HttpPostMessage;
import ru.yandex.dispatcher.common.SerializeUtils;
import ru.yandex.dispatcher.common.mappedvars.ZooBigDecimalNode;
import ru.yandex.dispatcher.producer.json.DefaultFailer;
import ru.yandex.dispatcher.producer.json.HandlersManager;
import ru.yandex.dispatcher.producer.json.PrefixExtractor;
import ru.yandex.dispatcher.producer.json.RootHandler;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.parser.uri.CgiParams;

public abstract class QueueRequestHandlerBase
    implements HttpAsyncRequestHandler<HttpRequest>
{
    protected final Producer producer;
    protected int skipped = 0;

    public QueueRequestHandlerBase(final Producer producer) {
        this.producer = producer;
    }

    @Override
    public BufferReusingAsyncRequestConsumer processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new BufferReusingAsyncRequestConsumer();
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        handle(new BasicProxySession(producer, exchange, context));
    }

    private void handle(final ProxySession session) {
        producer.parseRequestAsync(new ParseRequestCallback(session, this));
    }

    private static void freeResources(final ProxySession session) {
        if (session.request() instanceof HttpEntityEnclosingRequest) {
            final HttpEntity entity =
                ((HttpEntityEnclosingRequest) session.request()).getEntity();
            if (entity instanceof Closeable) {
                try {
                    ((Closeable) entity).close();
                } catch (IOException e) {
                    session.logger().log(
                        Level.SEVERE,
                        "Can't free HttpRequest buffer",
                        e);
                }
            }
        }
    }

    private void handle(
        final ProxySession session,
        final QueueRequest queueRequest)
        throws IOException, HttpException
    {
//    private void handle(final ProxySession session)
//        throws HttpException, IOException
//    {
//        QueueRequest queueRequest = producer.parseRequest(session);

        fillRequest(queueRequest);

        FutureCallback<QueueMessage> responseCallback =
            new HttpResponseCallback(session, queueRequest);

        if (queueRequest.doWait()) {
            responseCallback =
                new ConsumerWaitingCallback(producer, queueRequest,
                    responseCallback);
        }

        responseCallback = baseResponseCallback(queueRequest, responseCallback);

        addMessages(queueRequest);

        if (queueRequest.messages().size() == 0) {
            if (skipped == 0) {
                session.handleException(
                    new BadRequestException("No messages to store"));
            } else {
                session.handleException(
                    HttpExceptionConverter.toHttpException(
                        new ServerException(
                            HttpStatus.SC_CONFLICT,
                            "Message with this position was "
                            + "already stored")));
            }
        } else {
            producer.sendMessages(queueRequest, responseCallback);
        }
    }

    protected abstract void addMessages(
        final QueueRequest request)
        throws IOException, HttpException;

    protected FutureCallback<QueueMessage> baseResponseCallback(
        final QueueRequest request,
        final FutureCallback<QueueMessage> nextCallback)
        throws IOException, HttpException
    {
        return new SendMessageCallback(nextCallback);
    }

    protected void fillRequest(final QueueRequest request)
        throws IOException, HttpException
    {
        if (request.commonPrefix() == null) {
            throw new BadRequestException("no prefix, shard or ZooShardId "
                + " was specified");
        }
    }

    protected void addMessage(
        final QueueRequest request,
        long prefix,
        final HttpMessage httpMsg,
        final BigDecimal position)
        throws IOException, HttpException
    {
        if (request.logger().isLoggable(Level.FINEST)) {
            request.logger().finest("AddMessage: " + prefix
                + ", msg:" + httpMsg);
        }
        ZooBigDecimalNode positionNode = producer.getPositionNode(
            request.service(), request.producerName(), prefix);
        if (positionNode != null && position != null) {
            if (!producer.checkPosition(positionNode, position, prefix)) {
                if (request.logger().isLoggable(Level.FINE)) {
                    request.logger().fine(
                        "Skipping message request: "
                        + "message with position="
                        + position.toPlainString()
                        + " has been already stored");
                }
                skipped++;
                return;
            }
        }
        byte[] data = SerializeUtils.serializeHttpMessage(httpMsg);
        final CreateMode mode;
        if (request.checkDuplicate()) {
            if (request.memoryCheckOnly()) {
                if (request.failOnDuplicate()) {
                    mode = CreateMode.QUEUE_FAIL_ON_DUPLICATE_MEM_ONLY;
                } else {
                    mode = CreateMode.QUEUE_SUCCESS_ON_DUPLICATE_MEM_ONLY;
                }
            } else {
                if (request.failOnDuplicate()) {
                    mode = CreateMode.QUEUE_FAIL_ON_DUPLICATE;
                } else {
                    mode = CreateMode.QUEUE_SUCCESS_ON_DUPLICATE;
                }
            }
        } else {
            mode = CreateMode.QUEUE;
        }
        final String hash;
        if (request.hash() == null) {
            hash = HashUtil.hashRand(data);
        } else {
            hash = request.hash();
        }
        QueueMessage message = new QueueMessage(
            producer.pathFor(request.service(), prefix),
            hash,
            data,
            prefix,
            mode,
            request);
        if (positionNode != null && position != null) {
            message.setProducerPositionNode(positionNode);
            message.setProducerPosition(position);
        }
        request.addMessage(message);
    }

    private static class MultiMessageCallback
        implements FutureCallback<QueueMessage>
    {
        private final QueueRequest request;
        private final FutureCallback<QueueMessage> nextCallback;
        private Iterator<QueueMessage> iter = null;
        private boolean finished = false;

        public MultiMessageCallback(
            final QueueRequest request,
            final FutureCallback<QueueMessage> nextCallback)
        {
            this.request = request;
            this.nextCallback = nextCallback;
        }

        @Override
        public void completed(final QueueMessage message) {
            if (finished) {
                return;
            }
            if (iter == null) {
                iter = request.messages().iterator();
            }
            if (iter.hasNext()) {
                QueueMessage pending = iter.next();
                if (pending != message) {
                    failed(new BatchResponseError(
                        "Out of order processin: "
                        + "waiting for message " + pending
                        + " but received " + message));
                }
                if (!iter.hasNext()) {
                    finished = true;
                    nextCallback.completed(message);
                }
            }
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(Exception e) {
            if (finished) {
                return;
            }
            finished = true;
            nextCallback.failed(e);
        }
    }

    private static class SendMessageCallback
        implements FutureCallback<QueueMessage>
    {
        private final FutureCallback<QueueMessage> nextCallback;
        public SendMessageCallback(
            FutureCallback<QueueMessage> nextCallback)
        {
            this.nextCallback = nextCallback;
        }

        @Override
        public void completed(final QueueMessage message) {
            nextCallback.completed(message);
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(Exception e) {
            nextCallback.failed(e);
        }
    }

    private static class HttpResponseCallback
        implements FutureCallback<QueueMessage>
    {
        private final ProxySession session;
        private final QueueRequest request;
        public HttpResponseCallback(
            final ProxySession session,
            final QueueRequest request)
        {
            this.session = session;
            this.request = request;
        }

        @Override
        public void completed(final QueueMessage message) {
            if (message.response() != null) {
                session.response(
                    message.response().getHttpCode(),
                    message.response().getErrorString());
            } else {
                session.response(HttpStatus.SC_OK,
                    Long.toString(message.queueId()));
            }
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(Exception e) {
            session.handleException(HttpExceptionConverter.toHttpException(e));
        }
    }

    private static class ConsumerWaitingCallback
        implements FutureCallback<QueueMessage>
    {
        private final Producer producer;
        private final QueueRequest request;
        private final FutureCallback<QueueMessage> nextCallback;
        public ConsumerWaitingCallback(
            final Producer producer,
            final QueueRequest request,
            final FutureCallback<QueueMessage> nextCallback)
        {
            this.producer = producer;
            this.request = request;
            this.nextCallback = nextCallback;
        }

        @Override
        public void completed(final QueueMessage message) {
            if (request.logger().isLoggable(Level.FINE)) {
                request.logger().fine("Waiting for consumer for "
                    + message.path());
            }
            final MessageVersionCheckCallback versionCallback =
                new MessageVersionCheckCallback(
                    producer,
                    request,
                    nextCallback);
            producer.waitConsumer(request, message, versionCallback);
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(Exception e) {
            nextCallback.failed(e);
        }
    }

    private static class MessageVersionCheckCallback
        implements FutureCallback<QueueMessage>
    {
        private final Producer producer;
        private final QueueRequest request;
        private final FutureCallback<QueueMessage> nextCallback;
        public MessageVersionCheckCallback(
            final Producer producer,
            final QueueRequest request,
            final FutureCallback<QueueMessage> nextCallback)
        {
            this.producer = producer;
            this.request = request;
            this.nextCallback = nextCallback;
        }

        @Override
        public void completed(final QueueMessage message) {
            if (request.logger().isLoggable(Level.FINE)) {
                request.logger().fine("Checking version for " + message.path());
            }
            final ResponseReaderCallback responseCallback =
                new ResponseReaderCallback(producer, request, nextCallback);
            producer.checkMessageVersion(request, message, responseCallback);
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(Exception e) {
            nextCallback.failed(e);
        }
    }

    private static class ResponseReaderCallback
        implements FutureCallback<QueueMessage>
    {
        private final Producer producer;
        private final QueueRequest request;
        private final FutureCallback<QueueMessage> nextCallback;
        public ResponseReaderCallback(
            final Producer producer,
            final QueueRequest request,
            final FutureCallback<QueueMessage> nextCallback)
        {
            this.producer = producer;
            this.request = request;
            this.nextCallback = nextCallback;
        }

        @Override
        public void completed(final QueueMessage message) {
            if (request.logger().isLoggable(Level.FINE)) {
                request.logger().fine("Version for " + message.path() + " = "
                    + message.version());
            }
            if (message.version() != 0) {
                producer.readBackendResponse(request, message, nextCallback);
            } else {
                nextCallback.completed(message);
            }
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(Exception e) {
            nextCallback.failed(e);
        }
    }

    static class ParseRequestCallback
        implements FutureCallback<QueueRequest>
    {
        private final ProxySession session;
        private final QueueRequestHandlerBase requestHandler;
        public ParseRequestCallback(
            final ProxySession session,
            final QueueRequestHandlerBase requestHandler)
        {
            this.session = session;
            this.requestHandler = requestHandler;
        }

        public ProxySession session() {
            return session;
        }

        @Override
        public void completed(final QueueRequest request) {
            try {
                requestHandler.handle(session, request);
            } catch (Exception e) {
                session.handleException(
                    HttpExceptionConverter.toHttpException(e));
            } finally {
                freeResources(session);
            }
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(Exception e) {
            freeResources(session);
            session.handleException(HttpExceptionConverter.toHttpException(e));
        }
    }
}
