package ru.yandex.dispatcher.producer;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import ru.yandex.util.string.StringUtils;

public class SearchMap {
    public static final int SHARDS_COUNT = 65534;
    public static final int DEFAULT_JSON_INDEX_PORT = 8082;
    public static final int DEFAULT_JSON_SEARCH_PORT = 8080;

    public class Interval {
        private final String zk;
        private final String addr;
        private final String hostname;
        private final int indexPort;
        private final int searchPort;
        private final int queueIdPort;
        private final int min;
        private final int max;
        private final String service;
        private final boolean searchable;
        private final String lockType;
        private final String lockName;
        private final String targetHost;
        private final int targetTimeout;
        private final String tag;
        private final String consumerName;

        public Interval(
            final String zk,
            final String hostname,
            final String addr,
            final int indexPort,
            final int searchPort,
            final int queueIdPort,
            final String service,
            final boolean searchable,
            final int min,
            final int max,
            final String lockType,
            final String lockName,
            final String targetHost,
            final int targetTimeout,
            final String tag,
            final String consumerName)
        {
            this.zk = intern(zk);
            this.addr = intern(addr);
            this.hostname = intern(hostname);
            this.indexPort = indexPort;
            this.searchPort = searchPort;
            this.queueIdPort = queueIdPort;
            this.min = min;
            this.max = max;
            this.service = intern(service);
            this.searchable = searchable;
            this.lockType = intern(lockType);
            this.lockName = intern(lockName);
            this.targetHost = intern(targetHost);
            this.targetTimeout = targetTimeout;
            this.tag = intern(tag);
            this.consumerName = intern(consumerName);
        }

        private String intern(final String string) {
            if (string == null) {
                return null;
            }
            return string.intern();
        }

        public boolean match(int value) {
            return value >= min && value <= max;
        }

	public int min()
	{
	    return min;
	}

	public int max()
	{
	    return max;
	}

        public String zk() {
            return zk;
        }

        public String addr() {
            return addr;
        }

        public String hostname()
        {
            return hostname;
        }

        public boolean searchable()
        {
            return searchable;
        }

	public int indexPort()
	{
	    return indexPort;
	}

        public String targetHost() {
            return targetHost;
        }

        public int targetTimeout() {
            return targetTimeout;
        }

	public int searchPort()
	{
	    return searchPort;
	}

        public int queueIdPort() {
            return queueIdPort;
        }

        public String service() {
            return service;
        }

        public String lockType() {
            return lockType;
        }

        public String lockName() {
            return lockName;
        }

        public String consumerName() {
            String name = consumerName;
            if (name == null) {
                name = targetHost;
            }
            if (name == null) {
                name = hostname;
            }
            return name;
        }

        public String tag() {
            return tag;
        }

        @Override
        public String toString() {
            return "ZooKeepers[" + zk + "]:" + addr + ",shards:" + min + "-"
                + max;
        }
    }

    private Map<String, List<Interval>> map =
        new HashMap<String, List<Interval>>();

    private Map<String, Set<String>> backends =
        new HashMap<String, Set<String>>();

    private Map<String, List<Interval>> backendsToIntervals =
        new HashMap<String, List<Interval>>();

    private HashMap<String, String> zkdedupMap = new HashMap<>();

    public SearchMap(String fileName, Logger logger)
        throws FileNotFoundException, IOException, ParseException
    {
        BufferedReader reader = new BufferedReader(new InputStreamReader(
            new FileInputStream(fileName)));
        try {
            String line;
            for (int i = 1; (line = reader.readLine()) != null; ++i) {
                if (line.isEmpty() || line.charAt(0) == '#') {
                    continue;
                }

                String[] parts = line.split(" ");
                if (parts.length != 2) {
                    logger.severe( "Wrong number of space separated components: " + line + ", line=" +  i );
                }

                String host = null;
                String indexPort = null;
                String searchPort = null;
                String queueIdPort = null;
                String zk = null;
                String shards = null;
                String lockType = null;
                String lockName = "default";
                String targetHost = null;
                String tag = null;
                String consumerName = null;
                int targetTimeout = -1;
                boolean searchable = true;
                for (String param: parts[1].split(",")) {
                    String[] values = param.split(":", 2);
                    if (values[0].equals("host")) {
                        host = values[1];
                    } else if (values[0].equals("json_indexer_port")) {
                        indexPort = values[1];
                    } else if (values[0].equals("json_search_port")) {
                        searchPort = values[1];
                    } else if (values[0].equals("queue_id_port")) {
                        queueIdPort = values[1];
                    } else if (values[0].equals("zk")) {
                        zk = deduparseZk(values[1]);
                    } else if (values[0].equals("skip_search")) {
                        if (values[1].equalsIgnoreCase("yes")) {
                            searchable = false;
                        }
                    } else if (values[0].equals("tag")) {
                        tag = values[1];
                    } else if (values[0].equals("lock")) {
                        lockType = values[1];
                    } else if (values[0].equals("lockname")) {
                        lockName = values[1];
                    } else if (values[0].equals("target_host")) {
                        targetHost = values[1];
                    } else if (values[0].equals("consumer_name")) {
                        consumerName = values[1];
                    } else if (values[0].equals("shards")) {
                        shards = values[1];
                    } else if (values[0].equals("target_timeout")) {
                        targetTimeout = Integer.parseInt(values[1]);
                    }
                }
                if (host == null) {
                    logger.severe("host is not set: " + line + ", line=" + i);
		    continue;
                } else if (zk == null) {
//                    logger.severe("zk is not set: " + line + ", line=" +  i);
		    continue;
                } else if (shards == null) {
                    logger.severe("shards is not set: " + line + ", line=" + i);
                } else if (indexPort == null) {
//                    logger.severe(
//                        "json_index_port is not set, using default " + DEFAULT_JSON_INDEX_PORT + ": " + line + ", line=" + i);
//		    continue;
                } else if (searchPort == null) {
//                    logger.severe(
//                        "json_search_port is not set, using default " + DEFAULT_JSON_SEARCH_PORT + ": " + line + ", line=" + i);
//		    continue;
		}

                List<Interval> intervals = map.get(parts[0]);
                if (intervals == null) {
                    intervals = new ArrayList<Interval>();
                    map.put(parts[0], intervals);
                }
                String minMax[] = shards.split("-", 2);

                List<Interval> biList = backendsToIntervals.get(host);
                if (biList == null) {
                    biList = new ArrayList<Interval>();
                    backendsToIntervals.put(host, biList);
                }

                String addrLine;
                try {
		    addrLine = host;// + ":" + indexPort;
		    int indexPortParsed =
		        (indexPort == null ? DEFAULT_JSON_INDEX_PORT : Integer.parseInt(indexPort));
		    int queueIdPortParsed =
		        (queueIdPort != null ? Integer.parseInt(queueIdPort) : indexPortParsed);
		    Interval interval = new Interval(
                        zk,
                        host,
                        addrLine,
                        indexPortParsed,
                        ( searchPort == null ? DEFAULT_JSON_SEARCH_PORT : Integer.parseInt(searchPort) ),
                        queueIdPortParsed,
                        parts[0],
                        searchable,
                        Integer.parseInt(minMax[0]),
                        Integer.parseInt(minMax[1]),
                        lockType,
                        lockName,
                        targetHost,
                        targetTimeout,
                        tag,
                        consumerName);
                    intervals.add(interval);
                    biList.add(interval);
                } catch (Exception exc) {
//                    ParseException e =
//                        new ParseException("Failed to parse line:` " + line, i);
//                    e.initCause(exc);
//                    throw e;
                    logger.severe( "Failed to parse line: " + line + ", lineno: " + i + " : " + exc.getMessage() );
                    continue;
                }

                Set<String> backend = backends.get(addrLine);
                if (backend == null) {
                    backend = new HashSet<String>();
                    backends.put(addrLine, backend);
                }
                backend.add(zk);
            }
        } finally {
            reader.close();
        }
    }

    public String deduparseZk(final String zk) {
        String ret = zkdedupMap.get(zk);
        if (ret == null) {
            final String[] zkArray = zk.split("\\|");
            final Set<String> zkSet = new HashSet<>();
            final List<String> zkList = Arrays.asList(zkArray);
            Collections.sort(zkList);
            zkSet.addAll(zkList);
            ret = StringUtils.join(zkSet, ',');
            zkdedupMap.put(zk, ret);
        }
        return ret;
    }

    public Set<String> getZooKeepers(String backend) {
        return backends.get(backend);
    }

    public Set<String> getZooKeepers() {
        Set<String> result = new HashSet<String>();
        for (Map.Entry<String, List<Interval>> entry: map.entrySet()) {
            for (Interval interval: entry.getValue()) {
                result.add(interval.zk());
            }
        }
        return result;
    }

    public Set<String> getZooKeeperAddrs(String zk) {
        Set<String> result = new HashSet<String>();
        for (Map.Entry<String, List<Interval>> entry: map.entrySet()) {
            for (Interval interval: entry.getValue()) {
                if (interval.zk().equals(zk)) {
                    result.add(interval.addr());
                }
            }
        }
        return result;
    }

    public List<Interval> getBackendIntervals(String hostname) {
        return backendsToIntervals.get(hostname);
    }

    public Set<Interval> getZooKeeperIntervals(String zk) {
        Set<Interval> result = new HashSet<Interval>();
        for (Map.Entry<String, List<Interval>> entry: map.entrySet()) {
            for (Interval interval: entry.getValue()) {
                if (interval.zk().equals(zk)) {
                    result.add(interval);
                }
            }
        }
        return result;
    }

    public Set<String> getZooKeepers(String name, long prefix)
    {
        Set<String> result = new HashSet<String>();
        List<Interval> list = map.get(name);
        if (list != null) {
            int shard = (int)(prefix % SHARDS_COUNT);
            for (Interval interval: list) {
                if (interval.match(shard)) {
                    result.add(interval.zk());
                }
            }
        }
        return result;
    }

    public Set<String> getServiceZooKeepers(String service)
    {
        Set<String> result = new HashSet<String>();
        List<Interval> list = map.get(service);
        if (list != null) {
            for (Interval interval: list) {
                result.add(interval.zk());
            }
        }
        return result;
    }

    public boolean isBackendSearchable(String service, long prefix, String backend)
    {
        List<Interval> list = backendsToIntervals.get(backend);
        if (list != null) {
            int shard = (int)(prefix % SHARDS_COUNT);
            for (Interval interval: list) {
                if (interval.match(shard) && interval.service().equals(service)) {
                    return interval.searchable();
                }
            }
        }
        return false;
    }

    public String getZooKeeper(String name, long prefix)
    {
        List<Interval> list = map.get(name);
        if (list != null) {
            int shard = (int)Math.abs(prefix % SHARDS_COUNT);
            for (Interval interval: list) {
                if (interval.match(shard)) {
                    return interval.zk();
                }
            }
        }
        return null;
    }

    public static int getShard( long prefix )
    {
        return (int)(prefix % SHARDS_COUNT);
    }

/*    public Map<String, Set<String>> getAddrs(String name, long prefix)
    {
        Map<String, Set<String>> result =
            new HashMap<String, Set<String>>();
        List<Interval> list = map.get(name);
        if (list != null) {
            int shard = (int)(prefix % SHARDS_COUNT);
            for (Interval interval: list) {
                if (interval.match(shard)) {
                    Set<String> addrs = result.get(interval.zk());
                    if (addrs == null) {
                        addrs = new HashSet<String>();
                        result.put(interval.zk(), addrs);
                    }
                    addrs.add(interval.addr());
                }
            }
        }
        return result;
    }
*/
    public Set<String> getAddrs(String name, long prefix)
    {
        Set<String> result =
            new HashSet<String>();
        List<Interval> list = map.get(name);
        if (list != null) {
            int shard = (int)(prefix % SHARDS_COUNT);
            for (Interval interval: list) {
                if (interval.match(shard)) {
                    result.add(interval.addr());
                }
            }
        }
        return result;
    }

    public Set<String> getHosts(String name, long prefix)
    {
        Set<String> result =
            new HashSet<String>();
        List<Interval> list = map.get(name);
        if (list != null) {
            int shard = Math.abs((int)(prefix % SHARDS_COUNT));
            for (Interval interval: list) {
                if (interval.match(shard)) {
                    result.add(interval.hostname());
                }
            }
        }
        return result;
    }


    @Override
    public String toString() {
        return map.toString();
    }

    public static void main(String[] args)
        throws FileNotFoundException, IOException, ParseException
    {
        if (args.length < 1 || args.length > 3) {
            System.err.println("Usage:\n"
                + "ru.yandex.dispatcher.producer.SearchMap <searchmap file>\n"
                + "ru.yandex.dispatcher.producer.SearchMap <searchmap file> "
                    + "<backend>\n"
                + "ru.yandex.dispatcher.producer.SearchMap <searchmap file> "
                    + "<service> <prefix>");
            return;
        }
        SearchMap map = new SearchMap(args[0], Logger.getAnonymousLogger());
        System.out.println("SearchMap is:");
        for (Map.Entry<String, List<Interval>> entry: map.map.entrySet()) {
            System.out.println("Database " + entry.getKey()
                    + " has the following routes: " + entry.getValue());
        }

        System.out.println("\nZooKeepers list:");
        for (String zk: map.getZooKeepers()) {
            System.out.println("\t" + zk + ":");
            for (String addr: map.getZooKeeperAddrs(zk)) {
                System.out.println("\t\t" + addr);
            }
        }
/*
        if (args.length == 2) {
            System.out.println("\nBackend " + args[1]
                + " is served by clusters: " + map.getZooKeepers(args[1]));
        } else if (args.length == 3) {
            System.out.println("\nAddresses for service " + args[1]
                + " and prefix " + args[2] + " are:");
            Map<String, Set<String>> list =
                map.getAddrs(args[1], Long.parseLong(args[2]));
            for (Map.Entry<String, Set<String>> entry:
                list.entrySet())
            {
                System.out.println("\tZooKeepers " + entry.getKey()
                    + " stores queues for " + entry.getValue());
            }
        }
*/
    }
}

