package ru.yandex.dispatcher.producer;

import java.io.IOException;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.MethodNotSupportedException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;

import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.json.async.consumer.JsonAsyncDomConsumerFactory;

import ru.yandex.parser.uri.CgiParams;

public class StatusProxyHandler implements ProxyRequestHandler {
    protected Producer producer;
    private final AsyncClient httpClient;

    public StatusProxyHandler(final Producer producer) {
        this.producer = producer;
        this.httpClient = producer.httpClient();
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        final CgiParams cgiParams = session.params();
        sendProxyRequests(
            session.logger(),
            session.request(),
            cgiParams,
            new HttpResponseCallback(session));
    }

    public ZoolooserStatusParserCallback sendProxyRequests(
        final Logger logger,
        final HttpRequest request,
        final CgiParams cgiParams,
        final FutureCallback<HttpResponse> nextCallback)
        throws HttpException, IOException
    {
        final StatusProxyRequest statusRequest =
            new StatusProxyRequest(logger, cgiParams);

        final String path =
            "/_status?service=" + statusRequest.service()
            + "&prefix=" + statusRequest.shard()
            + "&all=true&json-type=normal";
        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(path);

        final List<HttpHost> hostList = new ArrayList<>();
        final String zooKeeperString =
            producer.searchMap().getZooKeeper(
                statusRequest.service(),
                statusRequest.shard());
        if (zooKeeperString == null) {
            throw new MethodNotSupportedException(
                "No routes found for service: " + statusRequest.service()
                + " and prefix " + statusRequest.shard());
        }
        for (final String zkHost : zooKeeperString.split("[\\|,]")) {
            final HttpHost host = httpHostFromZk(zkHost);
            hostList.add(host);
        }

        final ZoolooserStatusParserCallback zooCallback =
            new ZoolooserStatusParserCallback(
                statusRequest,
                producer.searchMap(),
                nextCallback);

        httpClient.executeWithDelay(
            hostList,
            generator,
            producer.config().statusProxyFallbackDelay(),
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.ANY_GOOD,
                JsonAsyncDomConsumerFactory.INSTANCE),
                zooCallback
            );
        return zooCallback;
    }

    private static HttpHost httpHostFromZk(final String zk) {
        String[] cols = zk.split(":");
        if (cols.length == 2) {
            final String portStr = cols[1];
            final int slash = portStr.indexOf('/');
            final int httpPort;
            if (slash == -1) {
                httpPort = Integer.parseInt("8" + portStr.substring(1));
            } else {
                httpPort = Integer.parseInt(portStr.substring(slash + 1));
            }
            return new HttpHost(cols[0], httpPort);
        }
        throw new IllegalArgumentException("Invalid zkHost: " + zk);
    }

    private static class HttpResponseCallback
        implements FutureCallback<HttpResponse>
    {
        private final ProxySession session;
        public HttpResponseCallback(final ProxySession session) {
            this.session = session;
        }

        @Override
        public void completed(final HttpResponse response) {
            session.response(response);
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(Exception e) {
            session.handleException(HttpExceptionConverter.toHttpException(e));
        }
    }
}
