package ru.yandex.dispatcher.producer;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.LockSupport;
import java.util.logging.Logger;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.OpResult;
import org.apache.zookeeper.Transaction;
import org.apache.zookeeper.ZooDefs.OpCode;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.dispatcher.common.ZooCallback;
import ru.yandex.dispatcher.common.ZooException;
import ru.yandex.dispatcher.common.connection.ZooConnection;
import ru.yandex.dispatcher.common.mappedvars.ZooLongNode;
import ru.yandex.util.timesource.TimeSource;

public class StatusQueue extends Thread {
    private final ZooConnection conn;
    private final Logger logger;
    private final Producer producer;

    private final int maxGroupingTime;
    private final long maxGroupingTimeNanos;
    private final int maxGroupSize;
    private final int maxGroupWeight;

    private final ConcurrentLinkedQueue<StatusMessage> queue =
        new ConcurrentLinkedQueue<>();
    private volatile boolean stop = false;
    private final TimeFrameQueue<Long> writeTimesQueue;

    public StatusQueue(
        final ZooConnection conn,
        final Logger logger,
        final Producer producer)
    {
        this.conn = conn;
        this.logger = logger;
        this.producer = producer;

        ProducerConfig config = (ProducerConfig) producer.config();
        this.maxGroupingTime = config.maxStatusGroupDelay();
        maxGroupingTimeNanos = maxGroupingTime * 1000;
        this.maxGroupWeight = config.maxStatusGroupWeight();
        this.maxGroupSize = config.maxStatusGroupSize();
//        this.writeTimesQueue = producer.statusWriteTimesQueue(conn.port());
        String ZKhost = conn.ZKhost();
        if (ZKhost.indexOf('.') >= 0 ){
            this.writeTimesQueue = producer.statusWriteTimesQueue(ZKhost.substring(0, ZKhost.indexOf('.')));
        } else {
            this.writeTimesQueue = producer.statusWriteTimesQueue(ZKhost);
        }
    }

    public void run() {
        ArrayList<StatusMessage> grouped = new ArrayList<StatusMessage>();
        while (!stop) {
            try {
                StatusMessage message = take();
                final long groupingStartTime = System.nanoTime();
                long groupingTime = 0;
                int weight = message.weight();
                grouped.add(message);
                do {
                    final long maxPoll =
                        maxGroupingTimeNanos - groupingTime;
                    message = poll(maxPoll);
                    if (message != null) {
                        grouped.add(message);
                        weight += message.weight();
                    }
                    groupingTime = System.nanoTime() - groupingStartTime;
                } while (groupingTime < maxGroupingTimeNanos
                    && weight < maxGroupWeight
                    && grouped.size() < maxGroupSize);
                sendGroup(grouped);
                if (logger.isLoggable(Level.INFO)) {
                    logger.info("Grouping time: "
                        + (System.nanoTime() - groupingStartTime));
                }
                grouped.clear();
            } catch (Exception e) {
                logger.log(
                    Level.SEVERE,
                    "Unhandled exception in MessageGrouper thread",
                    e);
            }
        }
    }

    private StatusMessage poll(final long nanos) {
        StatusMessage msg = queue.poll();
        if (msg == null) {
            LockSupport.parkNanos(nanos);
            msg = queue.poll();
        }
        return msg;
    }

    private StatusMessage take() {
        StatusMessage msg = queue.poll();
        while (msg == null) {
            LockSupport.parkNanos(maxGroupingTimeNanos);
            msg = queue.poll();
        }
        return msg;
    }

    private void sendGroup(final List<StatusMessage> messages) {
        ZooKeeper zk = conn.getZk(
            new ZooCallback() {
                 @Override
                public void error(ZooException e) {
                    for (StatusMessage message : messages) {
                        message.failed(e);
                    }
                }
            });
        if (zk == null) {
            return;
        }
        Transaction trans = zk.transaction();
        for (StatusMessage m : messages) {
            trans.setData(m.path(), m.data(), -1);
        }
        trans.commit(
            new GroupingTransactionCallback(
                new ArrayList<>(messages)),
            null);
        if (logger.isLoggable(Level.INFO)) {
            logger.info("Sent Group: statuses=" + messages.size());
        }
    }

    private void enqueueMessage(final StatusMessage message) {
        queue.offer(message);
    }

    public void enqueue(final List<StatusMessage> messages) {
        for (StatusMessage message : messages) {
            if (logger.isLoggable(Level.FINEST)) {
                logger.finest("Enqueuing: " + message);
            }
            enqueueMessage(message);
        }
    }

    public void enqueue(final StatusMessage message) {
        if (logger.isLoggable(Level.INFO)) {
            logger.info("Enqueuing: " + message);
        }
        enqueueMessage(message);
    }

    private void failMessages(
        final List<StatusMessage> messages,
        final Exception error)
    {
        for (StatusMessage message : messages) {
            message.failed(error);
        }
    }

    private void createPathAndRestart(
        final String path,
        final List<StatusMessage> messages)
    {
        FutureCallback<Void> cb = new FutureCallback<Void>() {
            @Override
            public void completed(Void v) {
                sendGroup(messages);
            }

            @Override
            public void failed(Exception e) {
                failMessages(messages, e);
            }

            @Override
            public void cancelled() {
            }
        };
        ZooUtil.createPath(conn, path, cb);
    }

    private void createStatusAndRestart(
        final String path,
        final List<StatusMessage> messages)
    {
        createPathAndRestart(path, messages);
    }

    private class GroupingTransactionCallback
        implements org.apache.zookeeper.AsyncCallback.MultiCallback
    {
        private final List<StatusMessage> messages;
        private final long startTime;

        public GroupingTransactionCallback(final List<StatusMessage> messages) {
            this.messages = messages;
            startTime = TimeSource.INSTANCE.currentTimeMillis();
        }

        @Override
        public void processResult(
            final int rc,
            final String path,
            final Object ctx,
            final List<OpResult> results)
        {
            KeeperException.Code code = KeeperException.Code.get(rc);
            long time = TimeSource.INSTANCE.currentTimeMillis() - startTime;
            int count = 0;
            if (results != null) {
                count = results.size();
            }
            writeTimesQueue.accept(Producer.packTimeNCount(time, count));
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("StatusGroup process result code: " + code
                    + ", time: " + time);
            }
            if (code == code.OK) {
                completed(messages, results);
            } else {
                if (results == null) {
                    final String error =
                        "Unhandled transaction results: null List<OpResult>"
                        + ", code: " + code;
                    if (logger.isLoggable(Level.SEVERE)) {
                        logger.severe(error);
                    }
                    failMessages(messages,
                        new MessageSendFailedException(error));
                } else {
                    failed(messages, results);
                }
            }
        }

        private void completed(
            final List<StatusMessage> messages,
            final List<OpResult> results)
        {
            final Iterator<StatusMessage> msgIter = messages.iterator();
            final Iterator<OpResult> resIter = results.iterator();
            while (msgIter.hasNext()) {
                final StatusMessage msg = msgIter.next();
                final OpResult op = resIter.next();
                final OpResult.SetDataResult statRes = (OpResult.SetDataResult)op;
                final Stat stat = statRes.getStat();
                msg.completed(stat.toString());
            }
        }

        private void failed(
            final List<StatusMessage> messages,
            final List<OpResult> results)
        {
            final Iterator<StatusMessage> msgIter = messages.iterator();
            final Iterator<OpResult> resIter = results.iterator();
            while (msgIter.hasNext()) {
                final StatusMessage msg = msgIter.next();
                final OpResult op = resIter.next();
                switch (op.getType()) {
                    case OpCode.setData:
                        //even if this message is ok next following error
                        //will rollback this message
                        break;
                    case OpCode.error:
                        final OpResult.ErrorResult er =
                            (OpResult.ErrorResult)op;
                        final KeeperException.Code code =
                            KeeperException.Code.get(er.getErr());
                        if (code == code.OK) {
                            //even if this message is ok next following error
                            //will rollback this message
                            break;
                        } else if (code == code.NONODE) {
                            if (logger.isLoggable(Level.SEVERE)) {
                                logger.severe("NoNode: " + msg.path()
                                    + ". Creating");
                            }
                            createStatusAndRestart(msg.path(), messages);
                            return;
                        } else {
                            final String error =
                                "Transaction failed: "
                                + "Unhandled error code for msg: " + msg.path()
                                + ", rc = " + code;
                            if (logger.isLoggable(Level.SEVERE)) {
                                logger.severe(error);
                            }
                            failMessages(messages,
                                new MessageSendFailedException(error));
                            return;
                        }
                    default:
                        final String error =
                            "Transaction failed: "
                            + "Unhandled AsyncCallback.MultiCallback result"
                            + " operation type: " + op.getClass().getName();
                        if (logger.isLoggable(Level.SEVERE)) {
                            logger.severe(error);
                        }
                        failMessages(messages,
                                new MessageSendFailedException(error));
                        return;
                }
            }
            //next results if for Producer positions nodes
            if (resIter.hasNext()) {
                final String error =
                    "Inconsistent transaction results: "
                    + "results list size is bigger than transaction size";
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.severe(error);
                }
                failMessages(messages,
                    new MessageSendFailedException(error));
            }
        }
    }
}
