package ru.yandex.dispatcher.producer;

import java.io.IOException;

import java.text.ParseException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.HttpVersion;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHttpResponse;

import ru.yandex.http.util.ServerException;

import ru.yandex.io.StringBuilderWriter;

import ru.yandex.json.writer.JsonWriter;

public class ZoolooserStatusParserCallback implements FutureCallback<Object> {
    private static final HttpResponse REQUEST_CANCELLED =
        new BasicHttpResponse(
            HttpVersion.HTTP_1_1,
            HttpStatus.SC_SERVICE_UNAVAILABLE,
            "request was cancelled");
    private final ThreadLocal<StringBuilderWriter> sbWriterLocal =
        new ThreadLocal<StringBuilderWriter>() {
            @Override
            public StringBuilderWriter initialValue() {
                return new StringBuilderWriter();
            }
        };
    private final StatusProxyRequest request;
    private final SearchMap searchMap;
    private final FutureCallback<HttpResponse> nextCallback;
    private final AtomicReference<HttpResponse> response =
        new AtomicReference<>(null);

    public ZoolooserStatusParserCallback(
        final StatusProxyRequest request,
        final SearchMap searchMap,
        final FutureCallback<HttpResponse> nextCallback)
    {
        this.request = request;
        this.searchMap = searchMap;
        this.nextCallback = nextCallback;
    }

    @Override
    public void completed(final Object domJson) {
        try {
            response(parseJson(domJson));
            if (nextCallback != null) {
                nextCallback.completed(response.get());
            }
        } catch (Exception e) {
            response(responseFromException(e));
            if (nextCallback != null) {
                nextCallback.failed(e);
            }
        }
    }

    private HttpResponse parseJson(final Object jsonRoot)
        throws ParseException, IOException
    {
        if (!(jsonRoot instanceof List)) {
            throw new ParseException("/_status json parsing failed: json root "
            + "is not a list: " + jsonRoot, 0);
        }
        final List<Status> statuses = new ArrayList<>();
        final List<?> statList = (List<?>) jsonRoot;
        for (Object elem : statList) {
            if (!(elem instanceof Map)) {
                throw new ParseException(
                    "/_status json parsing failed: array element is not a map: "
                    + elem, 0);
            }
            final Map<?,?> status = (Map<?,?>) elem;
            for (Map.Entry<?,?> entry : status.entrySet()) {
                final String host = (String) entry.getKey();
                final long pos = ((Number) entry.getValue()).longValue();
                if (!request.searchable()
                    || searchMap.isBackendSearchable(
                        request.service(),
                        request.shard(),
                        host))
                {
                    statuses.add(new Status(host, pos));
                }
            }
        }
        return generateResponse(statuses);
    }

    private HttpResponse generateResponse(final List<Status> statuses)
        throws IOException
    {
        if (statuses.size() == 0) {
            final Set<String> searchMapHosts =
                searchMap.getHosts(request.service(), request.shard());
            for (final String host : searchMapHosts) {
                statuses.add(new Status(host, -1));
            }
        }
        final StringBuilderWriter sbw = sbWriterLocal.get();
        sbw.clear();
        if (request.jsonType() == null) {
            if (statuses.size() > 0) {
                Collections.sort(statuses);
                final Status top = statuses.get(0);
                for (Status status : statuses) {
                    if (status.compareTo(top) != 0 && !request.all()) {
                        break;
                    }
                    sbw.append(status.host());
                    if (request.all()) {
                        sbw.append(' ');
                        sbw.append(Long.toString(status.pos()));
                    }
                    sbw.append('\n');
                }
            }
        } else {
            final JsonWriter jsonWriter = request.jsonType().create(sbw);
            Collections.sort(statuses);
            jsonWriter.startArray();
            for (Status status : statuses) {
                jsonWriter.startObject();
                jsonWriter.key(status.host());
                jsonWriter.value(status.pos());
                jsonWriter.endObject();
            }
            jsonWriter.endArray();
            jsonWriter.close();
        }
        final HttpResponse response = new  BasicHttpResponse(
            HttpVersion.HTTP_1_1,
            HttpStatus.SC_OK,
            "OK");
        response.setEntity(new StringEntity(sbw.toString()));
        return response;
    }

    @Override
    public void cancelled() {
        response(REQUEST_CANCELLED);
        if (nextCallback != null) {
            nextCallback.completed(REQUEST_CANCELLED);
        }
    }

    @Override
    public void failed(Exception e) {
        request.logger().log(
            Level.SEVERE,
            "Failed to get positions from zooloosers",
            e);
        if (request.allowCached()) {
            try {
                response(generateResponse(new ArrayList<Status>()));
                if (nextCallback != null) {
                    nextCallback.completed(response.get());
                }
            } catch (IOException ie) {
                request.logger().log(
                    Level.SEVERE,
                    "Failed to generate fake response",
                    ie);
                ie.addSuppressed(e);
                response(responseFromException(ie));
                if (nextCallback != null) {
                    nextCallback.failed(ie);
                }
            }
        } else {
            response(responseFromException(e));
            if (nextCallback != null) {
                nextCallback.failed(e);
            }
        }
    }

    private HttpResponse responseFromException(final Exception e) {
        return new BasicHttpResponse(
            HttpVersion.HTTP_1_1,
            HttpStatus.SC_INTERNAL_SERVER_ERROR,
            e.getMessage());
    }

    public synchronized HttpResponse get() {
        while (response.get() == null) {
            try {
                wait();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        return response.get();
    }

    private synchronized void response(final HttpResponse response) {
        this.response.set(response);
        notify();
    }

    private static class Status implements Comparable<Status> {
        private final String host;
        private final long pos;

        public Status(final String host, final long pos) {
            this.host = host;
            this.pos = pos;
        }

        public String host() {
            return host;
        }

        public long pos() {
            return pos;
        }

        @Override
        public int compareTo(final Status other) {
            return Long.compare(other.pos, pos);
        }
    }
}
