package ru.yandex.dispatcher.producer.statusreader;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.ParseException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.dispatcher.producer.SearchMap;

public class ConsumerServer
{
    private final SearchMap searchMap;
    private final Map<String,StatusConsumer> consumers = new HashMap<String, StatusConsumer>();
    private StatusWatcher statusWatcher;
    private PrefixedLogger logger;
    private int workers;
    private int timeout;
    private final Set<String> services;

    public ConsumerServer(String searchMapFile, int workers, int timeout, StatusWatcher watcher, PrefixedLogger logger, final Set<String> services)
        throws FileNotFoundException, IOException, ParseException
    {
        searchMap = new SearchMap(searchMapFile, logger);
        this.logger = logger;
        statusWatcher = watcher;
        this.workers = workers;
        this.timeout = timeout;
        this.services = services;
    }

    public ConsumerServer(SearchMap searchMap, int workers, int timeout, StatusWatcher watcher, PrefixedLogger logger, final Set<String> services)
        throws FileNotFoundException, IOException, ParseException
    {
        this.searchMap = searchMap;
        this.logger = logger;
        statusWatcher = watcher;
        this.workers = workers;
        this.timeout = timeout;
        this.services = services;
    }

    public void init() {
        Thread starter = new Thread("ConsumerServerBootstrap") {
            @Override
            public void run() {
                try {
                    ZooKeeperPool.setStaticLogger( logger );

                    for (String addr: searchMap.getZooKeepers())
                    {
                        if (!services.isEmpty()) {
                            boolean skip = true;
                            for (SearchMap.Interval interval :
                                searchMap.getZooKeeperIntervals(addr))
                            {
                                if (services.contains(interval.service())) {
                                    skip = false;
                                }
                            }
                            if (skip) {
                                logger.info(
                                    "StatusConsumer: "
                                    + "Skipping queue for zk: " + addr
                                    + " no allowed services found");
                                continue;
                            }
                        }
                        consumers.put(addr,
                            new StatusConsumer(
                                ConsumerServer.this,
                                addr,
                                searchMap,
                                workers,
                                timeout,
                                logger,
                                services));
                    }
                } catch (KeeperException | InterruptedException | IOException e) {
                    throw new RuntimeException("StatusConsumer init failed", e);
                }
                logger.info("StatusConsumer.Initialized");
            }
        };
        starter.start();
    }

    public String[] getFreshestBackends( String service, long prefix )
    {
        String zooKeeper = searchMap.getZooKeeper( service, prefix );
        if( zooKeeper == null )
        {
    	    logger.severe( "ConsumerServer.getFreshestBackends no zooKeepers can be found for: " + service + "/" + prefix );
    	    return null;
    	}
        StatusConsumer sc = consumers.get( zooKeeper );
        if( sc == null )
        {
    	    logger.severe( "ConsumerServer.getFreshestBackends no StatusConsumer can be found for: " + service + "/" + prefix );
    	    return null;
    	}
        return sc.getFreshestBackends( service, prefix );
    }

    public String[] getFreshestBackendsCached( String service, long prefix )
    {
        String zooKeeper = searchMap.getZooKeeper( service, prefix );
        if( zooKeeper == null )
        {
    	    logger.severe( "ConsumerServer.getFreshestBackends no zooKeepers can be found for: " + service + "/" + prefix );
    	    return null;
    	}
        StatusConsumer sc = consumers.get( zooKeeper );
        if( sc == null )
        {
    	    logger.severe( "ConsumerServer.getFreshestBackends no StatusConsumer can be found for: " + service + "/" + prefix );
    	    return null;
    	}
        return sc.getFreshestBackendsCached( service, prefix );
    }

    public void run()
    {
	synchronized(this)
	{
	    try
	    {
		wait();
	    }
	    catch( java.lang.InterruptedException ign )
	    {
	    }
	}
    }

    public static void main(String... args)
        throws FileNotFoundException, IOException, InterruptedException,
            KeeperException, ParseException, Exception
    {
        if (args.length == 0) {
            System.err.println("Usage: "
                + "ru.yandex.dispatcher.consumer.ConsumerServer "
                + "<searchmap file> [workers [timeout]]");
            return;
        }
        int workers = 10;
        int timeout = 50000;
        if (args.length > 1) {
            workers = Integer.parseInt(args[1]);
        }
        if (args.length > 2) {
            timeout = Integer.parseInt(args[2]);
        }
//        Logger.init( "status_consumer.log" );
        final ConsumerServer server =
            new ConsumerServer(args[0], workers, timeout, null, null, null);
//        server.start();

        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
        	synchronized(server)
        	{
        	    server.notify();
        	}
//                server.interrupt();
            }
        });

	server.run();
    }

    public void notifyStatus( String service, int shard, String backend, long queueId )
    {
        if( statusWatcher != null )
            statusWatcher.notifyStatus( service, shard, backend, queueId );
    }

}