package ru.yandex.dispatcher.producer.statusreader;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.HashSet;
import java.util.TreeMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Iterator;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.AsyncCallback.ChildrenCallback;
import org.apache.zookeeper.AsyncCallback.DataCallback;
import org.apache.zookeeper.AsyncCallback.StatCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.data.Stat;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.ZooDefs.Ids;
import org.apache.zookeeper.ZooKeeper;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.dispatcher.producer.SearchMap;

public class StatusConsumer
{
    private String zkAddr;
    private SearchMap searchMap;
    private int workers;
    private int timeout;
    private ConsumerServer parent;
    private PrefixedLogger logger;

    private Map<String, ServiceConsumer> services = new HashMap<String, ServiceConsumer>();

    StatusConsumer( ConsumerServer parent, String zkAddr, SearchMap searchMap, int workers, int timeout, PrefixedLogger logger, final Set<String> allowedServices )
        throws IOException, InterruptedException, KeeperException
    {
	this.zkAddr = zkAddr;
	this.searchMap = searchMap;
	this.workers = workers;
	this.timeout = timeout;
	this.parent = parent;

        for( SearchMap.Interval interval: searchMap.getZooKeeperIntervals(zkAddr) )
        {
            if (!allowedServices.isEmpty()
                && !allowedServices.contains(interval.service()))
            {
                logger.info("StatusConsumer: Skipping service <"
                    + interval.service()
                    + "> for zk <" + zkAddr + ">: "
                    + "service is not allowed");
                continue;
            }
            String service = interval.service();
            ServiceConsumer sc = services.get( service );
            if( sc == null )
            {
                sc = new ServiceConsumer( this, searchMap, service, zkAddr, workers, timeout, logger.replacePrefix("StatusConsumer-" + service) );
                services.put( service, sc );
            }
            sc.expandShards( interval.min(), Math.min(interval.max(), SearchMap.SHARDS_COUNT - 1) );
        }
        for( Map.Entry<String,ServiceConsumer> entry : services.entrySet() )
        {
            entry.getValue().start();
        }
        logger.info("StatusConsumer<"+zkAddr+"> Initialized");
    }

    public String[] getFreshestBackends( String service, long prefix )
    {
        ServiceConsumer sc = services.get( service );
        if( sc == null )
        {
	    logger.severe( "StatusConsumer.getFreshestBackends no ServiceConsumer can be found for: " + service + "/" + prefix );
    	    return null;
    	}
        try
        {
    	    return sc.getFreshestBackends( searchMap.getShard(prefix) );
    	}
    	catch( NewShardException e )
    	{
    	    Set<String> backends = searchMap.getAddrs( service, prefix );
    	    if( backends != null ) return backends.toArray( new String[backends.size()] );
    	    logger.severe( "StatusConsumer.getFreshestBackends no Backends can be found for newShard: " + service + "/" + prefix );
    	    return null;
//    	    return searchMap.getBackendsArray(service, prefix);
    	}
    }

    public String[] getFreshestBackendsCached( String service, long prefix )
    {
        ServiceConsumer sc = services.get( service );
        if( sc == null )
        {
	    logger.severe( "StatusConsumer.getFreshestBackends no ServiceConsumer can be found for: " + service + "/" + prefix );
    	    return null;
    	}
        try
        {
    	    return sc.getFreshestBackendsCached( searchMap.getShard(prefix) );
    	}
    	catch( NewShardException e )
    	{
    	    Set<String> backends = searchMap.getAddrs( service, prefix );
    	    if( backends != null ) return backends.toArray( new String[backends.size()] );
    	    logger.severe( "StatusConsumer.getFreshestBackends no Backends can be found for newShard: " + service + "/" + prefix );
    	    return null;
//    	    return searchMap.getBackendsArray(service, prefix);
    	}
    }

    public void notifyStatus( String service, int shardNo, String backend, long queueId )
    {
        parent.notifyStatus( service, shardNo, backend, queueId );
    }
}

