package ru.yandex.dispatcher.producer.statusreader;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.HashSet;
import java.util.TreeMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Iterator;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.AsyncCallback.ChildrenCallback;
import org.apache.zookeeper.AsyncCallback.DataCallback;
import org.apache.zookeeper.AsyncCallback.StatCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.data.Stat;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.ZooDefs.Ids;
import org.apache.zookeeper.ZooKeeper;

import ru.yandex.logger.PrefixedLogger;
import ru.yandex.util.timesource.TimeSource;

public class ZooKeeperPool extends Thread
{
    private ConcurrentHashMap<String,ZooKeeperConnection> pool = new ConcurrentHashMap<String,ZooKeeperConnection>();
    private static final ConcurrentHashMap<String,ConnectionChecker> checkersMap = new ConcurrentHashMap<String,ConnectionChecker>();
    private ServiceConsumer consumer;
    private static PrefixedLogger staticLogger = null;
    private int num;
    private int timeout;
    private PrefixedLogger logger;
    private final boolean autoSync;
    public ZooKeeperPool( ServiceConsumer consumer, int num, int timeout, PrefixedLogger logger, boolean autoSync )
    {
        this.consumer = consumer;
        this.num = num;
        this.timeout = timeout;
        this.logger = logger;
        this.autoSync = autoSync;
        start();
    }

    public boolean autoSync() {
        return autoSync;
    }

    public ZooKeeperConnection getConnection( String zkAddr )
    {
        if( !isAlive(zkAddr) ) return null;
        ZooKeeperConnection zkc;
        synchronized( pool )
        {
            zkc = pool.get( zkAddr );
            if( zkc == null )
            {
                zkc = new ZooKeeperConnection( zkAddr, consumer, num, timeout, logger );
                pool.put( zkAddr, zkc );
            }
            if( zkc.isConnected() )
            {
                return zkc;
            }
            zkc.connect();
            if( !zkc.isConnected() )
            {
                zkc.close();
                pool.remove( zkAddr );
                return null;
            }
            return zkc;
        }
    }

    public void run()
    {
        while( true )
        {
            try
            {
                for( Map.Entry<String, ZooKeeperConnection> entry : pool.entrySet() )
                {
                    ZooKeeperConnection zkc = entry.getValue();
                    if( zkc.lastUpdate() + 500 < TimeSource.INSTANCE.currentTimeMillis() )
                    {
                        if (autoSync) {
                            zkc.sync();
                        }
                    }
                }

                Thread.sleep( 500 );
            }
            catch( Exception e )
            {
                e.printStackTrace();
            }
        }
    }

    public final static boolean isAlive( String addr )
    {
        ConnectionChecker cc = checkersMap.get( addr );
        if( cc == null )
        {
            ConnectionChecker newCc = new ConnectionChecker( addr );
            cc = checkersMap.putIfAbsent( addr, newCc );
            if( cc == null )
            {
                cc = newCc;
                try
                {
                    cc.start();
                }
                catch( java.lang.IllegalThreadStateException ee )
                {
                    //alread started
                }
            }
        }
        return cc.alive;
    }

    public final static void setStaticLogger( PrefixedLogger logger )
    {
        staticLogger = logger;
    }

    private static class ConnectionChecker extends Thread implements Watcher
    {
    private String addr;
    public volatile boolean alive = true;
    private PrefixedLogger logger;

        public ConnectionChecker( String addr )
        {
            this.addr = addr;
            setName( "Checker-" + addr );
            if( staticLogger != null )
            {
                logger = staticLogger.replacePrefix("ZooConnectionChecker-" + addr);
            }
        }

        public synchronized void run()
        {
            while( true )
            {
                try
                {
                    ZooKeeper zk = new ZooKeeper( addr, 30000, this, logger );
                    while( true )
                    {
                        wait( 30000 );
                        if( !zk.getState().isAlive() || !zk.getState().isConnected() )
                        {
                            alive = false;
                            if( logger != null ) logger.fine( "disconnected" );
                            zk.close();
                            Thread.sleep( 5000 );
                            break;
                        }
                        else
                        {
                            if( logger != null ) logger.fine( "connected" );
                            alive = true;
                        }
                    }
                }
                catch( Exception e )
                {
                }
            }
        }

        @Override
        public synchronized void process(WatchedEvent event)
        {
            notify();
        }
    }

}
