/**
 * Licensed to the zk1931 under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.  You may obtain a copy of the
 * License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.github.zk1931.jzab;

import com.google.protobuf.TextFormat;
import java.io.IOException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.HashSet;

import java.util.logging.Level;
import java.util.logging.Logger;

import com.github.zk1931.jzab.proto.ZabMessage;
import com.github.zk1931.jzab.proto.ZabMessage.Message;
import com.github.zk1931.jzab.proto.ZabMessage.Message.MessageType;

/**
 * Follower.
 */
class Follower extends Participant {

  public Follower(
    final java.util.logging.Logger logger,
    ParticipantState participantState,
    StateMachine stateMachine,
    ZabConfig config)
  {
    super(logger, participantState, stateMachine, config);
    filter = new FollowerFilter(logger, messageQueue, election);
//    MDC.put("state", "following");
  }

  @Override
  protected void changePhase(Phase phase) throws IOException {
    this.currentPhase = phase;
    if (phase == Phase.DISCOVERING) {
//      MDC.put("phase", "discovering");
      if (stateChangeCallback != null) {
        stateChangeCallback.followerDiscovering(this.electedLeader);
      }
      if (failCallback != null) {
        failCallback.followerDiscovering();
      }
    } else if (phase == Phase.SYNCHRONIZING) {
//      MDC.put("phase", "synchronizing");
      if (stateChangeCallback != null) {
        stateChangeCallback
        .followerSynchronizing(persistence.getProposedEpoch());
      }
      if (failCallback != null) {
        failCallback.followerSynchronizing();
      }
    } else if (phase == Phase.BROADCASTING) {
//      MDC.put("phase", "broadcasting");
      if (stateChangeCallback != null) {
        stateChangeCallback
        .followerBroadcasting(persistence.getAckEpoch(),
                              getAllTxns(),
                              persistence.getLastSeenConfig());
      }
      if (failCallback != null) {
        failCallback.followerBroadcasting();
      }
    } else if (phase == Phase.FINALIZING) {
//      MDC.put("phase", "finalizing");
      stateMachine.recovering(pendings);
      if (persistence.isInStateTransfer()) {
        // If the participant goes back to recovering phase in state
        // trasferring mode, we need to explicitly undo the state transferring.
        persistence.undoStateTransfer();
      }
      // Closes the connection to leader.
      if (this.electedLeader != null) {
        this.transport.clear(this.electedLeader);
      }
    }
  }

  /**
   * Starts from joining some one who is in cluster..
   *
   * @param peer the id of server who is in cluster.
   * @throws Exception in case something goes wrong.
   */
  @Override
  public void join(String peer) throws Exception {
    logger.warning("Follower joins in.");
    try {
      logger.warning("Query leader from " + peer);
      Message query = MessageBuilder.buildQueryLeader();
      MessageTuple tuple = null;
      while (true) {
        // The joiner might gets started before the cluster gets started.
        // Instead of reporting failure immediately, the follower will retry
        // joining the cluster until joining the cluster successfully.
        try {
          sendMessage(peer, query);
          tuple = filter.getExpectedMessage(MessageType.QUERY_LEADER_REPLY,
                                            peer, config.getTimeoutMs());
          break;
        } catch (TimeoutException ex) {
          long retryInterval = 1;
          logger.warning("Timeout while contacting leader, going to retry after "
            + retryInterval + " second.");
          // Waits for 1 second.
          Thread.sleep(retryInterval * 1000);
        }
      }
      this.electedLeader = tuple.getMessage().getReply().getLeader();
      logger.warning("Got current leader " + this.electedLeader);

      /* -- Synchronizing phase -- */
      while (true) {
        try {
          joinSynchronization();
          break;
        } catch (TimeoutException | BackToElectionException ex) {
          logger.warning("Timeout("
            + getSyncTimeoutMs()
            + " ms) in synchronizing, retrying. Last zxid : "
            + persistence.getLatestZxid());
          transport.clear(electedLeader);
          clearMessageQueue();
        }
      }
      // See if it can be restored from the snapshot file.
      restoreFromSnapshot();
      // Delivers all transactions in log before entering broadcasting phase.
      deliverUndeliveredTxns();

      /* -- Broadcasting phase -- */
      // Initialize the vote for leader election.
      this.election.specifyLeader(this.electedLeader);
      changePhase(Phase.BROADCASTING);
      accepting();
    } catch (InterruptedException e) {
      logger.warning("Participant is canceled by user.");
      throw e;
    } catch (TimeoutException e) {
      logger.warning("Didn't hear message from "
        + this.electedLeader + " for " + this.config.getTimeoutMs()
        + " milliseconds. Going back to leader election.");
      if (persistence.getLastSeenConfig() == null) {
        throw new JoinFailure("Fails to join cluster.");
      }
    } catch (BackToElectionException e) {
      logger.warning("Got GO_BACK message from queue, going back to electing.");
      if (persistence.getLastSeenConfig() == null) {
        throw new JoinFailure("Fails to join cluster.");
      }
    } catch (LeftCluster e) {
      logger.warning("Exit running : " + e.getMessage());
      throw e;
    } catch (Exception e) {
      logger.log(Level.SEVERE, "Caught exception", e);
      throw e;
    } finally {
      changePhase(Phase.FINALIZING);
    }
  }

  public void follow(String leader) throws Exception {
    this.electedLeader = leader;
    try {
      /* -- Discovering phase -- */
      changePhase(Phase.DISCOVERING);
      sendProposedEpoch();
      waitForNewEpoch();

      /* -- Synchronizing phase -- */
      logger.warning("Synchronizing...");
      changePhase(Phase.SYNCHRONIZING);
      // Starts synchronizing.
      waitForSync(this.electedLeader);
      waitForNewLeaderMessage();
      waitForCommitMessage();
      // Adjusts the sync timeout based on this synchronization time.
//      adjustSyncTimeout((int)(syncTime / 1000000));

      // See if it can be restored from the snapshot file.
      restoreFromSnapshot();
      // Delivers all transactions in log before entering broadcasting phase.
      deliverUndeliveredTxns();

      /* -- Broadcasting phase -- */
      changePhase(Phase.BROADCASTING);
      accepting();
    } catch (InterruptedException e) {
      logger.warning("Participant is canceled by user.");
      throw e;
    } catch (TimeoutException e) {
      logger.warning("Didn't hear message from "
        + this.electedLeader + " for " + this.config.getTimeoutMs()
        + " milliseconds. Going back to leader election.");
    } catch (BackToElectionException e) {
      logger.warning("Got GO_BACK message from queue, going back to electing.");
    } catch (Zab.SimulatedException e) {
      logger.warning("Got SimulatedException, go back to leader election.");
    } catch (LeftCluster e) {
      logger.warning("Exit running : " + e.getMessage());
      throw e;
    } catch (Exception e) {
      logger.log(Level.SEVERE, "Caught exception", e);
      throw e;
    } finally {
      if (this.currentPhase == Phase.SYNCHRONIZING) {
        incSyncTimeout();
        logger.warning("Go back to recovery in synchronization phase, increase " +
            "sync timeout to " + getSyncTimeoutMs() + " milliseconds.");
      }
      changePhase(Phase.FINALIZING);
    }
  }

  /**
   * Sends CEPOCH message to its prospective leader.
   * @throws IOException in case of IO failure.
   */
  void sendProposedEpoch() throws IOException {
    Message message = MessageBuilder
                      .buildProposedEpoch(persistence.getProposedEpoch(),
                                          persistence.getAckEpoch(),
                                          persistence.getLastSeenConfig(),
                                          getSyncTimeoutMs());
    if (logger.isLoggable(Level.FINEST)) {
      logger.finest("Sends "
        + TextFormat.shortDebugString(message)
        + " to leader " + this.electedLeader);
    }
    sendMessage(this.electedLeader, message);
  }

  /**
   * Waits until receives the NEWEPOCH message from leader.
   *
   * @throws InterruptedException if anything wrong happens.
   * @throws TimeoutException in case of timeout.
   * @throws IOException in case of IO failure.
   */
  void waitForNewEpoch()
      throws InterruptedException, TimeoutException, IOException {
    MessageTuple tuple = filter.getExpectedMessage(MessageType.NEW_EPOCH,
                                                   this.electedLeader,
                                                   config.getTimeoutMs());
    Message msg = tuple.getMessage();
    String source = tuple.getServerId();
    ZabMessage.NewEpoch epoch = msg.getNewEpoch();
    if (epoch.getNewEpoch() < persistence.getProposedEpoch()) {
      logger.severe("New epoch " + epoch.getNewEpoch() + " from "
        + source + " is smaller than last received proposed epoch "
        + persistence.getProposedEpoch());
      throw new RuntimeException("New epoch is smaller than current one.");
    }
    // Updates follower's last proposed epoch.
    persistence.setProposedEpoch(epoch.getNewEpoch());
    // Updates follower's sync timeout.
    setSyncTimeoutMs(epoch.getSyncTimeout());
    logger.warning("Received the new epoch proposal "
        + epoch.getNewEpoch() + " from " + source);
    Zxid zxid = persistence.getLatestZxid();
    // Sends ACK to leader.
    sendMessage(this.electedLeader,
                MessageBuilder.buildAckEpoch(persistence.getAckEpoch(),
                                             zxid));
  }

  /**
   * Waits for NEW_LEADER message and sends back ACK and update ACK epoch.
   *
   * @throws TimeoutException in case of timeout.
   * @throws InterruptedException in case of interrupt.
   * @throws IOException in case of IO failure.
   */
  void waitForNewLeaderMessage()
      throws TimeoutException, InterruptedException, IOException {
    logger.warning("Waiting for New Leader message from " + this.electedLeader);
    MessageTuple tuple = filter.getExpectedMessage(MessageType.NEW_LEADER,
                                                   this.electedLeader,
                                                   config.getTimeoutMs());
    Message msg = tuple.getMessage();
    String source = tuple.getServerId();
    if (logger.isLoggable(Level.FINEST)) {
      logger.finest("Follower.wairForNewLeader Got message "
        + TextFormat.shortDebugString(msg)
        + " from " + source);
    }
    ZabMessage.NewLeader nl = msg.getNewLeader();
    long epoch = nl.getEpoch();
    Log log = persistence.getLog();
    // Sync Ack epoch to disk.
    log.sync();
    persistence.setAckEpoch(epoch);
    Message ack = MessageBuilder.buildAck(persistence.getLatestZxid());
    sendMessage(source, ack);
  }

  /**
   * Wait for a commit message from the leader.
   *
   * @throws TimeoutException in case of timeout.
   * @throws InterruptedException in case of interruption.
   * @throws IOException in case of IO failures.
   */
  void waitForCommitMessage()
      throws TimeoutException, InterruptedException, IOException {
    logger.warning("Waiting for commit message from " + this.electedLeader);
    MessageTuple tuple = filter.getExpectedMessage(MessageType.COMMIT,
                                                   this.electedLeader,
                                                   config.getTimeoutMs());
    Zxid zxid = MessageBuilder.fromProtoZxid(tuple.getMessage()
                                                  .getCommit()
                                                  .getZxid());
    Zxid lastZxid = persistence.getLatestZxid();
    // If the followers are appropriately synchronized, the Zxid of ACK should
    // match the last Zxid in followers' log.
    if (zxid.compareTo(lastZxid) != 0) {
      logger.warning("The ACK zxid " + zxid + " doesn't match last zxid "
        + lastZxid + " in log!");
      throw new RuntimeException("The ACK zxid doesn't match last zxid");
    }
  }

  void acceptingInit() throws IOException {
    this.clusterConfig = persistence.getLastSeenConfig();
    this.syncProcessor =
      new SyncProposalProcessor(persistence, transport, maxBatchSize);
    this.commitProcessor
      = new CommitProcessor(logger, stateMachine, lastDeliveredZxid, serverId,
                            transport, null, clusterConfig, electedLeader,
                            pendings);
    this.snapProcessor =
      new SnapshotProcessor(stateMachine, persistence, serverId, transport);
    // Notifies the client current configuration.
    stateMachine.following(electedLeader,
                           new HashSet<String>(clusterConfig.getPeers()));
  }

  /**
   * Entering broadcasting phase.
   *
   * @throws InterruptedException if it's interrupted.
   * @throws TimeoutException  in case of timeout.
   * @throws IOException in case of IOException.
   * @throws ExecutionException in case of exception from executors.
   */
  void accepting()
      throws TimeoutException, InterruptedException, IOException,
      ExecutionException {
    // Initialization.
    acceptingInit();
    // The last time of HEARTBEAT message comes from leader.
    long lastHeartbeatTime = System.nanoTime();
    long ackEpoch = persistence.getAckEpoch();
    try {
      while (true) {
        MessageTuple tuple = filter.getMessage(config.getTimeoutMs());
        Message msg = tuple.getMessage();
        String source = tuple.getServerId();
        // The follower only expect receiving message from leader and
        // itself(REQUEST).
        if (source.equals(this.electedLeader)) {
          lastHeartbeatTime = System.nanoTime();
        } else {
          // Checks if the leader is alive.
          long timeDiff = (System.nanoTime() - lastHeartbeatTime) / 1000000;
          if ((int)timeDiff >= this.config.getTimeoutMs()) {
            // HEARTBEAT timeout.
            logger.warning("Detects there's a timeout in waiting"
                + "message from leader " + this.electedLeader
                + ", goes back to leader electing");
            throw new TimeoutException("HEARTBEAT timeout!");
          }
          if (msg.getType() == MessageType.QUERY_LEADER) {
            logger.warning("Got QUERY_LEADER from " + source);
            Message reply = MessageBuilder.buildQueryReply(this.electedLeader);
            sendMessage(source, reply);
          } else if (msg.getType() == MessageType.ELECTION_INFO) {
            this.election.reply(tuple);
          } else if (msg.getType() == MessageType.DELIVERED) {
            // DELIVERED message should come from itself.
            onDelivered(msg);
          } else if (msg.getType() == MessageType.SNAPSHOT) {
            snapProcessor.processRequest(tuple);
          } else if (msg.getType() == MessageType.SNAPSHOT_DONE) {
            commitProcessor.processRequest(tuple);
          } else {
            logger.warning("Got unexpected message from " + source + ", ignores.");
          }
          continue;
        }
        if (msg.getType() != MessageType.HEARTBEAT && logger.isLoggable(Level.FINEST)) {
          logger.finest("Follower.accepting Got message " + msg.getType()
             + " from " + source);
//          logger.finest("Follower.accepting Got message " + TextFormat.shortDebugString(msg)
//             + " from " + source);
        }
        if (msg.getType() == MessageType.PROPOSAL) {
          Transaction txn = MessageBuilder.fromProposal(msg.getProposal());
          Zxid zxid = txn.getZxid();
          if (zxid.getEpoch() == ackEpoch) {
            onProposal(tuple);
          } else {
            logger.warning("The proposal has the wrong epoch "
                + zxid.getEpoch() + ", expecting " + ackEpoch);
            throw new RuntimeException("The proposal has wrong epoch number.");
          }
        } else if (msg.getType() == MessageType.COMMIT) {
          commitProcessor.processRequest(tuple);
        } else if (msg.getType() == MessageType.HEARTBEAT) {
          logger.finest("Got HEARTBEAT from " + source);
          // Replies HEARTBEAT message to leader.
          Message heartbeatReply = MessageBuilder.buildHeartbeat();
          sendMessage(source, heartbeatReply);
        } else if (msg.getType() == MessageType.FLUSH) {
          onFlush(tuple);
        } else {
            logger.warning("Unexpected messgae : "
                + TextFormat.shortDebugString(msg)
                + " from " + source);
        }
      }
    } finally {
      commitProcessor.shutdown();
      syncProcessor.shutdown();
      snapProcessor.shutdown();
      this.lastDeliveredZxid = commitProcessor.getLastDeliveredZxid();
      this.participantState.updateLastDeliveredZxid(this.lastDeliveredZxid);
    }
  }

  // It's possible the cluster already has a huge history. Instead of
  // joining the cluster now, we do the first synchronization to make the
  // new joined server's history 'almost' up-to-date, then issues the
  // JOIN message. Once JOIN message has been issued, the cluster might be
  // blocked (depends on whether the new configuration will have a quorum
  // of servers in broadcasting phase while the synchronization is going
  // on). If the cluster will be blocked, the length of the blocking time
  // depends on the length of the synchronization.
  private void joinSynchronization()
      throws IOException, TimeoutException, InterruptedException {
    Message sync = MessageBuilder.buildSyncHistory(persistence.getLatestZxid());
    sendMessage(this.electedLeader, sync);
    MessageTuple tuple =
      filter.getExpectedMessage(MessageType.SYNC_HISTORY_REPLY, electedLeader,
                                config.getTimeoutMs());
    Message msg = tuple.getMessage();
    // Updates the sync timeout based on leader's suggestion.
    setSyncTimeoutMs(msg.getSyncHistoryReply().getSyncTimeout());
    // Waits for the first synchronization completes.
    waitForSync(this.electedLeader);

    // Gets the last zxid in disk after first synchronization.
    Zxid lastZxid = persistence.getLatestZxid();
    logger.warning("After first synchronization, the last zxid is " + lastZxid);
    // Then issues the JOIN message.
    Message join = MessageBuilder.buildJoin(lastZxid);
    sendMessage(this.electedLeader, join);

    /* -- Synchronizing phase -- */
    changePhase(Phase.SYNCHRONIZING);
    waitForSync(this.electedLeader);
    waitForNewLeaderMessage();
    waitForCommitMessage();
    persistence.setProposedEpoch(persistence.getAckEpoch());
  }

  /**
   * A filter class for follower acts as a successor of ElectionMessageFilter
   * class. It filters and handles DISCONNECTED and PROPOSED_EPOCH messages.
   */
  class FollowerFilter extends ElectionMessageFilter {
    FollowerFilter(
        final Logger logger,
        BlockingQueue<MessageTuple> msgQueue,
        Election election)
    {
      super(logger, msgQueue, election);
    }

    @Override
    protected MessageTuple getMessage(int timeoutMs)
        throws InterruptedException, TimeoutException {
      int startMs = (int)(System.nanoTime() / 1000000);
      while (true) {
        int nowMs = (int)(System.nanoTime() / 1000000);
        int remainMs = timeoutMs - (nowMs - startMs);
        if (remainMs < 0) {
          remainMs = 0;
        }
        MessageTuple tuple = super.getMessage(remainMs);
        if (tuple.getMessage().getType() == MessageType.DISCONNECTED) {
          // Got DISCONNECTED message enqueued by onDisconnected callback.
          Message msg = tuple.getMessage();
          String peerId = msg.getDisconnected().getServerId();
          if (electedLeader != null && peerId.equals(electedLeader)) {
            // Disconnection from elected leader, going back to leader election,
            // the clearance of transport will happen in exception handlers of
            // follow/join function.
            logger.warning("Lost elected leader " + electedLeader);
            throw new BackToElectionException();
          } else {
            // Lost connection to someone you don't care, clear transport.
            logger.warning("Lost peer " + peerId);
            transport.clear(peerId);
          }
        } else if (tuple.getMessage().getType() == MessageType.PROPOSED_EPOCH) {
          // Explicitly close the connection when gets PROPOSED_EPOCH message in
          // FOLLOWING state to help the peer selecting the right leader faster.
          logger.warning("Got PROPOSED_EPOCH in FOLLOWING state. Close connection.");
          transport.clear(tuple.getServerId());
        } else {
          return tuple;
        }
      }
    }
  }
}
