/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.zookeeper;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.apache.zookeeper.KeeperException;

/***
 *  CreateMode value determines how the znode is created on ZooKeeper.
 */
public enum CreateMode {
    
    /**
     * The znode will not be automatically deleted upon client's disconnect.
     */
    PERSISTENT (0, false, false, false, false, false, false),
    /**
    * The znode will not be automatically deleted upon client's disconnect,
    * and its name will be appended with a monotonically increasing number.
    */
    PERSISTENT_SEQUENTIAL (2, false, true, false, false, false, false),
    /**
     * The znode will be deleted upon the client's disconnect.
     */
    EPHEMERAL (1, true, false, false, false, false, false),
    /**
     * The znode will be deleted upon the client's disconnect, and its name
     * will be appended with a monotonically increasing number.
     */
    EPHEMERAL_SEQUENTIAL (3, true, true, false, false, false, false),

    QUEUE (4, false, true, true, false, false, false),

    QUEUE_SUCCESS_ON_DUPLICATE (5, false, true, true, true, false, false),

    QUEUE_FAIL_ON_DUPLICATE (6, false, true, true, true, true, false),

    QUEUE_SUCCESS_ON_DUPLICATE_MEM_ONLY (7, false, true, true, true, false, true),

    QUEUE_FAIL_ON_DUPLICATE_MEM_ONLY (8, false, true, true, true, true, true);

    private static final Logger LOG = LoggerFactory.getLogger(CreateMode.class);

    private boolean ephemeral;
    private boolean sequential;
    private boolean queue;
    private final boolean checkDuplicate;
    private final boolean errorOnDuplicate;
    private final boolean memCheckOnly;
    private int flag;

    CreateMode(
        final int flag,
        final boolean ephemeral,
        final boolean sequential,
        final boolean queue,
        final boolean checkDuplicate,
        final boolean errorOnDuplicate,
        final boolean memCheckOnly)
    {
        this.flag = flag;
        this.ephemeral = ephemeral;
        this.sequential = sequential;
        this.queue = queue;
        this.checkDuplicate = checkDuplicate;
        this.errorOnDuplicate = errorOnDuplicate;
        this.memCheckOnly = memCheckOnly;
    }

    public boolean isEphemeral() { 
        return ephemeral;
    }

    public boolean isSequential() { 
        return sequential;
    }

    public boolean isQueue() { 
        return queue;
    }

    public boolean checkDuplicate() {
        return checkDuplicate;
    }

    public boolean errorOnDuplicate() {
        return errorOnDuplicate;
    }

    public boolean memCheckOnly() {
        return memCheckOnly;
    }

    public int toFlag() {
        return flag;
    }

    /**
     * Map an integer value to a CreateMode value
     */
    static public CreateMode fromFlag(int flag) throws KeeperException {
        switch(flag) {
        case 0: return CreateMode.PERSISTENT;

        case 1: return CreateMode.EPHEMERAL;

        case 2: return CreateMode.PERSISTENT_SEQUENTIAL;

        case 3: return CreateMode.EPHEMERAL_SEQUENTIAL ;

        case 4: return CreateMode.QUEUE ;

        case 5: return CreateMode.QUEUE_SUCCESS_ON_DUPLICATE ;

        case 6: return CreateMode.QUEUE_FAIL_ON_DUPLICATE ;

        case 7: return CreateMode.QUEUE_SUCCESS_ON_DUPLICATE_MEM_ONLY ;

        case 8: return CreateMode.QUEUE_FAIL_ON_DUPLICATE_MEM_ONLY ;

        default:
            LOG.error("Received an invalid flag value to convert to a CreateMode");
            throw new KeeperException.BadArgumentsException(); 
        }
    }
}
