/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.zookeeper.server;

import java.io.IOException;
import java.util.TreeSet;
import java.util.Set;

import org.apache.jute.InputArchive;
import org.apache.jute.OutputArchive;
import org.apache.jute.Record;
import org.apache.zookeeper.data.Stat;
import org.apache.zookeeper.data.StatPersisted;
//import org.apache.zookeeper.server.persistence.Chunk;
//import org.apache.zookeeper.server.persistence.ChunkedStorage;

/**
 * This class contains the data for a node in the data tree.
 * <p>
 * A data node contains a reference to its parent, a byte array as its data, an
 * array of ACLs, a stat object, and a set of its children's paths.
 * 
 */
public class DataNode extends StatPersisted implements Record {//, Comparable<DataNode> {
    /** the data for this datanode */
    byte data[];
    HashKey hash;

    private static final int BASE_OBJECT_SIZE = 130;
    public  static final int DATANODE_LONG_CVERSION = 3;

    private long size;

    boolean hasQueue;
    boolean queueChild;

    /**
     * the acl map long for this datanode. the datatree has the map
     */
//    Long acl;

    /**
     * the stat for this node that is persisted to disk.
     */
//    public StatPersisted stat;

    /**
     * the list of children for this node. note that the list of children string
     * does not contain the parent path -- just the last part of the path. This
     * should be synchronized on except deserializing (for speed up issues).
     */
    private ChildSet children = null;

    /**
     * default constructor for the datanode
     */
    DataNode() {
        // default constructor
        size = BASE_OBJECT_SIZE;
        hasQueue = false;
    }

    /**
     * create a DataNode with parent, data, acls and stat
     * 
     * @param parent
     *            the parent of this DataNode
     * @param data
     *            the data to be set
     * @param acl
     *            the acls for this node
     * @param stat
     *            the stat for this node.
     */
    public DataNode(byte[] data, StatPersisted stat) {
        setCzxid(stat.getCzxid());
        setMzxid(stat.getMzxid());
        setCtime(stat.getCtime());
        setMtime(stat.getMtime());
        setVersion(stat.getVersion());
        setCversion(stat.getCversion());
        setAversion(stat.getAversion());
        setEphemeralOwner(stat.getEphemeralOwner());
        setPzxid(stat.getPzxid());
        this.data = data;
        this.size = data == null ? 0 : data.length;
        this.size += BASE_OBJECT_SIZE;
        this.hasQueue = false;
    }

/*    @Override
    public int compareTo( DataNode other ) {
	return other.size() - size();
    }
*/
/*    @Override
    public int compareTo(Object o) {
        return super.compareTo(o);
    }
*/
    public long size() {
        long fullSize = size;
//        if (children != null) {
//            fullSize += children.sizeInBytes();
//        }
        if (hash != null) {
            fullSize += hash.size();
        }
	return fullSize;
    }

    /**
     * Method that inserts a child into the children set
     * 
     * @param child
     *            to be inserted
     * @return true if this set did not already contain the specified element
     */
    public synchronized boolean addChild(String child, long size, boolean queueChild) {
        if (children == null) {
            // let's be conservative on the typical number of children
            if (queueChild) {
                children = new QueueChildSet();
            } else {
                children = new TreeChildSet();
            }
        }
        this.size += size;
        return children.add(child);
    }

    /**
     * Method that removes a child from the children set
     * 
     * @param child
     * @return true if this set contained the specified element
     */
    public synchronized boolean removeChild(String child, long size) {
        if (children == null) {
            return false;
        }
        this.size -= size;
        return children.remove(child);
    }

    /**
     * convenience method for setting the children for this datanode
     * 
     * @param children
     */
/*
    public synchronized void setChildren(TreeSet<String> children, long size) {
        this.children = children;
        this.size = (data == null ? 0 : data.length) + size;
    }
*/
    public synchronized String getOldestChild() {
	return children.first();
    }

    public synchronized String getNewestChild() {
	return children.last();
    }

    /**
     * convenience methods to get the children
     * 
     * @return the children of this datanode
     */
    public synchronized ChildSet getChildren() {
        return children;
    }

    public synchronized long getApproximateDataSize() {
        if(null==data) return 0;
        return data.length;
    }

    synchronized public void copyStat(Stat to) {
        to.setAversion(getAversion());
        to.setCtime(getCtime());
        to.setCzxid(getCzxid());
        to.setMtime(getMtime());
        to.setMzxid(getMzxid());
        to.setPzxid(getPzxid());
        to.setVersion(getVersion());
        to.setEphemeralOwner(getEphemeralOwner());
        to.setDataLength(data == null ? 0 : data.length);
        int numChildren = 0;
        if (this.children != null) {
            numChildren = children.size();
        }
        // when we do the Cversion we need to translate from the count of the creates
        // to the count of the changes (v3 semantics)
        // for every create there is a delete except for the children still present
        to.setCversion(getCversion()*2 - numChildren);
        to.setNumChildren(numChildren);
    }

    synchronized public void deserialize(InputArchive archive, String tag)
            throws IOException {
        archive.startRecord("node");
        data = archive.readBuffer("data");
        size = data == null ? 0 : data.length;
        size += BASE_OBJECT_SIZE;
        hasQueue = archive.readBool("hasQueue");
        queueChild = archive.readBool("queueChild");
        hash = DataTree.hashKey(archive.readString("hash"));
//        stat = new StatPersisted();
        statDeserialize(archive, "statpersisted");
        archive.endRecord("node");
    }

    synchronized public void serialize(OutputArchive archive, String tag, StringBuilder sb)
            throws IOException {
        archive.startRecord(this, "node");
        archive.writeBuffer(data, "data");
        archive.writeBool(hasQueue, "hasQueue");
        archive.writeBool(queueChild, "queueChild");
        if (hash == null) {
            archive.writeString(null, "hash");
        } else {
            hash.toString(sb);
            archive.writeStringBuilder(sb, "hash");
        }
        statSerialize(archive, "statpersisted");
        archive.endRecord(this, "node");
    }

    synchronized public void serialize(OutputArchive archive, String tag)
            throws IOException {
        archive.startRecord(this, "node");
        archive.writeBuffer(data, "data");
        archive.writeBool(hasQueue, "hasQueue");
        archive.writeBool(queueChild, "queueChild");
        if (hash == null) {
            archive.writeString(null, "hash");
        } else {
            archive.writeString(hash.toString(), "hash");
        }
        statSerialize(archive, "statpersisted");
        archive.endRecord(this, "node");
    }

    public void statSerialize(OutputArchive a_, String tag)
        throws java.io.IOException
    {
        a_.startRecord(this,tag);
        a_.writeLong(getCzxid(),"czxid");
        a_.writeLong(getMzxid(),"mzxid");
        a_.writeLong(getCtime(),"ctime");
        a_.writeLong(getMtime(),"mtime");
        a_.writeInt(getVersion(),"version");
        if (a_.version() >= DATANODE_LONG_CVERSION) {
            a_.writeLong(getCversion(),"cversion");
        } else {
            a_.writeInt((int)getCversion(),"cversion");
        }
        a_.writeInt(getAversion(),"aversion");
        a_.writeLong(getEphemeralOwner(),"ephemeralOwner");
        a_.writeLong(getPzxid(),"pzxid");
        a_.endRecord(this,tag);
    }

    public void statDeserialize(InputArchive a_, String tag)
        throws java.io.IOException
    {
        a_.startRecord(tag);
        setCzxid(a_.readLong("czxid"));
        setMzxid(a_.readLong("mzxid"));
        setCtime(a_.readLong("ctime"));
        setMtime(a_.readLong("mtime"));
        setVersion(a_.readInt("version"));
        if (a_.version() >= DATANODE_LONG_CVERSION) {
            setCversion(a_.readLong("cversion"));
        } else {
            setCversion(a_.readInt("cversion"));
        }
        setAversion(a_.readInt("aversion"));
        setEphemeralOwner(a_.readLong("ephemeralOwner"));
        setPzxid(a_.readLong("pzxid"));
        a_.endRecord(tag);
    }

}
