/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.zookeeper.server;

import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.util.timesource.TimeSource;

public class DataWatchManager extends Thread {
    private static final Logger LOG = LoggerFactory.getLogger(DataTree.class);

    private static final String STOP = "\\STOP";
    private static final int SLEEP_DELAY = 20;
    private final ConcurrentHashMap<String, Set<Watcher>> pathWatchers;
    private final ConcurrentLinkedQueue<String> notifyQueue;
    private final ConcurrentHashMap<String, DataNode> dataMap;
    private volatile boolean stop = false;
    private final AtomicLong sessionId =
        new AtomicLong(TimeSource.INSTANCE.currentTimeMillis());
    private volatile boolean enabled = false;

    public DataWatchManager() {
        super("DWMngr");
        setDaemon(true);
        pathWatchers = new ConcurrentHashMap(128, 0.75f, 2);
        notifyQueue = new ConcurrentLinkedQueue();
        dataMap = new ConcurrentHashMap(128, 0.75f, 2);
    }

    public void enable() {
        enabled = true;
    }

    public void dataNotify(final String path, final DataNode node) {
        if (!enabled) {
            return;
        }
        if (dataMap.putIfAbsent(path, node) == null) {
            notifyQueue.offer(path);
        }
    }

    public void registerWatcher(final String path, final Watcher watcher) {
        if (stop) {
            watcher.close();
            return;
        }
        Set<Watcher> watchers = pathWatchers.get(path);
        if (watchers == null) {
            Set<Watcher> newWatchers = ConcurrentHashMap.newKeySet(1);
            watchers = pathWatchers.putIfAbsent(path, newWatchers);
            if (watchers == null) {
                watchers = newWatchers;
            }
        }
        if (stop) {
            watcher.close();
            return;
        }
        watchers.add(watcher);
    }

    public void unregisterWatcher(final String path, final Watcher watcher) {
        Set<Watcher> watchers = pathWatchers.get(path);
        if (watchers == null) {
            return;
        }
        watchers.remove(watcher);
    }

    @Override
    public void run() {
        while (true) {
            try {
                final String path = notifyQueue.poll();
                if (path == null) {
                    Thread.sleep(SLEEP_DELAY);
                    continue;
                }
                if (path == STOP) {
                    LOG.info("DataWatchManager stopped");
                    return;
                }
                DataNode node = dataMap.remove(path);
                byte[] data;
                long zxid;
                synchronized (node) {
                    data = node.data;
                    zxid = node.getMzxid();
                }
                Event event = new Event(path, data, zxid);
                for (String subPath = path; subPath != null;) {
                    Set<Watcher> watchers = pathWatchers.get(subPath);
                    if (watchers != null) {
                        Iterator<Watcher> iter = watchers.iterator();
                        while (iter.hasNext()) {
                            Watcher watcher = iter.next();
                            if (watcher.closed()) {
                                iter.remove();
                            } else {
                                watcher.notifyDataChange(event);
                            }
                        }
                    }
                    int slash = subPath.lastIndexOf('/');
                    if (slash != -1 && slash != 0) {
                        subPath = subPath.substring(0, slash);
                    } else {
                        subPath = null;
                    }
                }
            } catch (Throwable t) {
                t.printStackTrace();
            }
        }
    }

    public void close() {
        stop = true;
        notifyQueue.clear();
        notifyQueue.offer(STOP);
        dataMap.clear();
        for (Set<Watcher> watchers: pathWatchers.values()) {
            for (Watcher watcher: watchers) {
                watcher.close();
            }
        }
        pathWatchers.clear();
    }

    public long newSessionId() {
        return sessionId.incrementAndGet();
    }

    interface Watcher {
        void notifyDataChange(final Event event);

        void close();

        boolean closed();
    }

    static class Event {
        private final String path;
        private final byte[] data;
        private final long zxid;

        Event(final String path, final byte[] data, final long zxid) {
            this.path = path;
            this.data = data;
            this.zxid = zxid;
        }

        public String path() {
            return path;
        }

        public byte[] data() {
            return data;
        }

        public long zxid() {
            return zxid;
        }
    }
}
