package org.apache.zookeeper.server;

import java.util.Arrays;

public class HexByteHash implements HashKey {
    private static final int BASE_OBJECT_SIZE = 40;
    public final String parentPath;
    public final byte[] bytes;
    public final int hashCode;

    final static char[] digits = {
        '0' , '1' , '2' , '3' , '4' , '5' ,
        '6' , '7' , '8' , '9' , 'a' , 'b' ,
        'c' , 'd' , 'e' , 'f' , 'g' , 'h' ,
        'i' , 'j' , 'k' , 'l' , 'm' , 'n' ,
        'o' , 'p' , 'q' , 'r' , 's' , 't' ,
        'u' , 'v' , 'w' , 'x' , 'y' , 'z'
    };

    private static final byte[] UNHEX = {
        (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0,
        (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0,
        (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0,
        (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0,
        (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0,
        (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0,
        (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0,
        (byte) 0x1, (byte) 0x2, (byte) 0x3, (byte) 0x4, (byte) 0x5, (byte) 0x6,
        (byte) 0x7, (byte) 0x8, (byte) 0x9, (byte) 0, (byte) 0, (byte) 0,
        (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0xa, (byte) 0xb,
        (byte) 0xc, (byte) 0xd, (byte) 0xe, (byte) 0xf, (byte) 0, (byte) 0,
        (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0,
        (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0,
        (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0, (byte) 0,
        (byte) 0, (byte) 0, (byte) 0, (byte) 0xa, (byte) 0xb, (byte) 0xc,
        (byte) 0xd, (byte) 0xe, (byte) 0xf};

    private HexByteHash(final String parentPath, final byte[] data) {
        this.parentPath = parentPath;
        this.bytes = data;
        hashCode = 0;
    }

    private HexByteHash(final String parentPath, final String hex) {
        this.parentPath = parentPath.intern();
        bytes = getBytes(hex);
        hashCode = hex.hashCode() + parentPath.hashCode();
    }

    private HexByteHash(
        final String parentPath,
        final byte[] bytes,
        final int hashCode)
    {
        this.parentPath = parentPath;
        this.bytes = bytes;
        this.hashCode = hashCode;
    }

    public static HexByteHash parse(final String parentPath, final String hex) {
        byte[] bytes = getBytes(hex);
        if (bytes == null) {
            return null;
        }
        String parentPathInterned = parentPath.intern();
        int hashCode = hex.hashCode() + parentPathInterned.hashCode();
        return new HexByteHash(parentPathInterned, bytes, hashCode);
    }

    public static final byte charValue(final char c, final String hex) {
        switch(c) {
            case '0':
                return 0;
            case '1':
                return 1;
            case '2':
                return 2;
            case '3':
                return 3;
            case '4':
                return 4;
            case '5':
                return 5;
            case '6':
                return 6;
            case '7':
                return 7;
            case '8':
                return 8;
            case '9':
                return 9;
            case 'a':
            case 'A':
                return 10;
            case 'b':
            case 'B':
                return 11;
            case 'c':
            case 'C':
                return 12;
            case 'd':
            case 'D':
                return 13;
            case 'e':
            case 'E':
                return 14;
            case 'f':
            case 'F':
                return 15;
            default:
                throw new java.lang.NumberFormatException(hex);
        }
    }

    public static final int getLen(final String hex) {
        int i = 0;
        int b = 0;
        boolean odd = false;
        int len = hex.length();
        for(i = 0; i < len; i++) {
            byte v = UNHEX[hex.charAt(i)];
            if (!odd) {
                if (v == 0) {
                    b++;
                    continue;
                } else {
                    if (i == len - 1) {
                        b++;
                    }
                }
            } else {
                b++;
            }
            odd = !odd;
        }
//        if (odd) b++;
        return b;
    }

    public static final byte[] getBytes(final String hex) {
//        System.out.println("hexlen: " + getLen(hex));
        if (hex.length() != 64) {
            return null;
        }
        byte[] bytes = new byte[getLen(hex)];
        int i = 0;
        int b = 0;
        int len = hex.length();
        boolean odd = false;
        for(i = 0; i < len; i++) {
//            int odd = t & 0x1;
            byte v = UNHEX[hex.charAt(i)];
            if (!odd) {
                if (v == 0) {
                    bytes[b] = 0;
//                    System.out.println( Integer.toHexString(bytes[b] & 0xFF) );
                    b++;
                    continue;
                } else {
                    if (i == len - 1) {
                        bytes[b] = (byte)(v);
                    } else {
                        bytes[b] = (byte)(v << 4);
                    }
                }
            } else {
                bytes[b] |= v;
//                System.out.println( "odd: " + Integer.toHexString(bytes[b] & 0xFF) );
                b++;
            }
            odd = !odd;
        }
        return bytes;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder(bytes.length * 2);
        sb.append("hash");
        sb.append(parentPath);
        sb.append("/");
        final int charPos = 32;
        final int shift = 4;
        final int radix = 1 << shift;
        final int mask = radix - 1;
        for (int i = 0; i < bytes.length; i++) {
            final byte b = bytes[i];
            final byte hi = (byte)((b >>> shift) & mask);
            final byte lo = (byte)(b & mask);
            if (hi != 0) {
                sb.append(digits[hi]);
            }
            sb.append(digits[lo]);
        }
        return new String(sb);
    }

    public String toStringOld() {
        StringBuilder sb = new StringBuilder(bytes.length * 2);
        sb.append("hash");
        sb.append(parentPath);
        sb.append("/");
        for(int i = 0; i < bytes.length; i++) {
            sb.append(Integer.toHexString(bytes[i] & 0xFF));
        }
        return sb.toString();
    }

    public void toString(final StringBuilder sb) {
        sb.setLength(0);
        sb.append("hash");
        sb.append(parentPath);
        sb.append('/');
        final int charPos = 32;
        final int shift = 4;
        final int radix = 1 << shift;
        final int mask = radix - 1;
        for (int i = 0; i < bytes.length; i++) {
            final byte b = bytes[i];
            final byte hi = (byte)((b >>> shift) & mask);
            final byte lo = (byte)(b & mask);
            if (hi != 0) {
                sb.append(digits[hi]);
            }
            sb.append(digits[lo]);
        }
    }

    public int hashCode() {
        return hashCode;
    }

    public boolean equals(Object o) {
        if (o instanceof HexByteHash) {
            HexByteHash other = (HexByteHash)o;
            if (!parentPath.equals(other.parentPath)) {
                return false;
            }
            return Arrays.equals(bytes, other.bytes);
        }
        return false;
    }

    public int size() {
        return BASE_OBJECT_SIZE * 3 + parentPath.length() * 2 +
            bytes.length;
    }

    public static void main(String[] args) {
        byte[] testData = new byte[1024];
        for (int i = 0; i < testData.length; i++) {
            testData[i] = (byte)(127 - Math.random() * 256);
        }
        final HexByteHash hash = new HexByteHash("test", testData);
        final StringBuilder sb = new StringBuilder();
        hash.toString(sb);
        final String method1 = new String(sb);
        final String method2 = hash.toString();
        if (!method1.equals(method2)) {
            System.err.println("toString methods output differs: \nm1="
                + method1 + "\nm2=" + method2);
        } else {
            System.err.println("toString methods output are equal: \nm1="
                + method1 + "\nm2=" + method2);
        }
    }

}
