package org.apache.zookeeper.server;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;
import java.util.concurrent.atomic.AtomicLong;

public class QueueChildSet implements ChildSet {
public static final String PREFIX = "queue";
private LinkedList<Interval> intervals;
private volatile long lastNum = -1;
private volatile long firstNum = -1;

    private static class Interval {
        public long start;
        public long end;

        public Interval(final long start) {
            this.start = start;
            this.end = start;
        }

        protected Interval(final long start, final long end) {
            this.start = start;
            this.end = end;
        }

        public Interval clone() {
            return new Interval(start, end);
        }
    }

    public synchronized boolean add(final String child) {
        String numberStr = child.substring(PREFIX.length());
        Long num = Long.parseLong(numberStr);
        Interval current;
        if (intervals == null || intervals.size() == 0) {
            intervals = new LinkedList<Interval>();
            current = new Interval(num);
            intervals.add(current);
            lastNum = firstNum = num;
            return true;
        } else {
            current = intervals.getLast();
        }
        if (current.end >= num) {
            return insertChild(num);
//            throw new IllegalStateException(
//                "Adding queue children with lessen number: " + num + " < " +
//                current.end);
        }
        if (current.end + 1 < num) { //hole
            current = new Interval(num);
            intervals.add(current);
        } else {
            current.end++;
        }
        lastNum = num;
        return true;
    }

    private boolean insertChild(final long num) {
        if (contains(num)) {
            return false;
        }

        Interval ceil = ceilInterval(num);
        Interval floor = floorInterval(num);
        if (ceil == null) {
            //insert into beggining
            firstNum = num;
            if (floor.start - 1 == num) {
                //expand floor
                floor.start--;
                return true;
            }
            //else add new ceil interval
            intervals.addFirst(new Interval(num));
            return true;
        }
        if (floor == null) {
            lastNum = num;
            if (ceil.end + 1 == num) {
                ceil.end++;
                return true;
            }
            intervals.addLast(new Interval(num));
            return true;
        }
        //we have ceil and floor intervals
        //check if we can merge both
        if (ceil.end + 1 == floor.start - 1) {
            ceil.end = floor.end;
            intervals.remove(floor);
            return true;
        }
        if (ceil.end + 1 == num) {
            ceil.end++;
            return true;
        }
        if (floor.start - 1 == num) {
            floor.start--;
            return true;
        }
        //insert new interval;
        Interval newInterval = new Interval(num);
        ListIterator<Interval> iter = intervals.listIterator();
        while (iter.hasNext()) {
            Interval interval = iter.next();
            if (interval == ceil) {
                iter.add(newInterval);
            }
        }
        return true;
    }

    public Interval ceilInterval(final long num) {
        Interval ceil = null;
        for (Interval interval : intervals) {
            if (interval.start > num) {
                return ceil;
            } else {
                if (interval.end < num) {
                    ceil = interval;
                }
            }
        }
        return ceil;
    }

    public Interval floorInterval(final long num) {
        for (Interval interval : intervals) {
            if (interval.start > num) {
                return interval;
            }
        }
        return null;
    }

    private boolean removeFromMid(long num) {
//        if (!contains(num)) return false;
        ListIterator<Interval> iter = intervals.listIterator();
        boolean ret = false;
        while (iter.hasNext()) {
            Interval interval = iter.next();
            if (interval.start <= num && interval.end >= num) {
                if (interval.start == interval.end) {
                    //remove whole interval
                    iter.remove();
                    ret = true;
                    break;
                }
                if (interval.start == num) {
                    interval.start++;
                    ret = true;
                    break;
                }
                if (interval.end == num) {
                    interval.end--;
                    ret = true;
                    break;
                }
                //else split interval;
                Interval newInterval = new Interval(num + 1);
                newInterval.end = interval.end;
                interval.end = num - 1;
                iter.add(newInterval);
                ret = true;
                break;
            }
        }
        if (num == lastNum) {
            lastNum = calcLast();
        }
        if (num == firstNum) {
            firstNum = calcFirst();
        }
        return ret;
    }

    public synchronized boolean remove(final String child) {
        String numberStr = child.substring(PREFIX.length());
        Long num = Long.parseLong(numberStr);
        if (intervals == null || intervals.size() == 0) return false;
        Interval first = intervals.getFirst();
        if (first.start != num) {
            return removeFromMid(num);
//            throw new IllegalStateException(
//                "Remove not first child from QueueDataNode: " + num + " <> " +
//                first.start);
        }
        first.start++;
        if (first.start > first.end) {
            intervals.remove();
        }
        if (num == lastNum) {
            lastNum = calcLast();
        }
        if (num == firstNum) {
            firstNum = calcFirst();
        }
        return true;
    }

    private long calcFirst() {
        if (intervals == null || intervals.size() == 0) {
            return -1;
        }
        final Interval first = intervals.getFirst();
        return first.start;
    }

    private long calcLast() {
        if (intervals == null || intervals.size() == 0) {
            return -1;
        }
        final Interval last = intervals.getLast();
        return last.end;
    }

    public long firstPosition() {
        return firstNum;
    }

    public long lastPosition() {
        return lastNum;
    }

    public String first() {
        final long first = firstNum;
        if (first == -1) {
            return null;
        }
        return PREFIX + PrepRequestProcessor.intToStringPadded(first);
    }

    public String last() {
        final long last = lastNum;
        if (last == -1) {
            return null;
        }
        return PREFIX + PrepRequestProcessor.intToStringPadded(last);
    }

    public synchronized String nextChild(String child) {
        if (intervals == null || intervals.size() == 0) {
            return null;
        }
        String numberStr = child.substring(PREFIX.length());
        Long num = Long.parseLong(numberStr);
        Long next = null;
        for (Interval interval : intervals) {
            if (interval.start <= num) {
                if (interval.end > num) {
                    next = num + 1;
                }
            } else {
                next = interval.start;
            }
        }
        if (next != null) {
            return PREFIX + PrepRequestProcessor.intToStringPadded(next);
        }
        return null;
    }

    public synchronized boolean contains(String child) {
        String numberStr = child.substring(PREFIX.length());
        Long num = Long.parseLong(numberStr);
        for (Interval interval : intervals) {
            if (interval.start <= num && interval.end >= num) {
                return true;
            }
        }
        return false;
    }

    public synchronized boolean contains(long num) {
        for (Interval interval : intervals) {
            if (interval.start <= num && interval.end >= num) {
                return true;
            }
        }
        return false;
    }

    public synchronized int size() {
        int size = 0;
        for (Interval interval : intervals) {
            size += interval.end - interval.start + 1;
        }
        return size;
    }

    public synchronized List<String> getList() {
        ArrayList<String> list = new ArrayList<String>();
        for (Interval interval : intervals) {
            for (long i = interval.start; i <= interval.end; i++) {
                list.add(PREFIX + PrepRequestProcessor.intToStringPadded(i));
            }
        }
        return list;
    }

    public synchronized String[] toArray() {
        List<String> list = getList();
        return list.toArray(new String[list.size()]);
    }

    @Override
    public synchronized Iterator<String> iterator() {
        final List<Interval> cloned = new ArrayList<>(intervals.size());
        for (Interval interval : intervals) {
            cloned.add(interval.clone());
        }
        return new IteratorImpl(cloned);
    }

    private class IteratorImpl implements Iterator<String> {
        private final List<Interval> intervals;
        private final Iterator<Interval> intervalIterator;
        private Interval currentInterval = null;
        private long pos = 0;

        public IteratorImpl(final List<Interval> intervals) {
            this.intervals = intervals;
            intervalIterator = this.intervals.iterator();
        }

        @Override
        public boolean hasNext() {
            if (currentInterval == null) {
                if (intervalIterator.hasNext()) {
                    currentInterval = intervalIterator.next();
                    pos = currentInterval.start;
                } else {
                    return false;
                }
            }
            if (pos <= currentInterval.end) {
                return true;
            } else {
                currentInterval = null;
                return hasNext();
            }
        }

        @Override
        public String next() {
            if (hasNext()) {
                return PREFIX + PrepRequestProcessor.intToStringPadded(pos++);
            } else {
                throw new NoSuchElementException();
            }
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }

    }

    public String dumpIntervals() {
        StringBuilder sb = new StringBuilder();
        String sep = "";
        for (Interval i : intervals) {
            sb.append(sep);
            sep = "\n";
            sb.append("{ ");
            sb.append(i.start);
            sb.append(" - ");
            sb.append(i.end);
            sb.append(" }");
        }
        sb.append(" first: ");
        sb.append(firstNum);
        sb.append("last: ");
        sb.append(lastNum);
        return sb.toString();
    }

    private static void testAdd(QueueChildSet set, long i) {
        System.out.println("adding " + i);
        set.add(PREFIX + i);
    }

    private static void testRemove(QueueChildSet set, long i) {
        System.out.println("removing " + i);
        set.remove(PREFIX + i);
    }

    public static void main(String[] args) {
        QueueChildSet set = new QueueChildSet();
        testAdd(set, 0);
        testAdd(set, 1);
        testAdd(set, 2);
        testAdd(set, 3);

        testAdd(set, 5);
        testAdd(set, 6);
        testAdd(set, 7);
        testAdd(set, 8);
        System.out.println(set.dumpIntervals());
        testAdd(set, 4);;
        System.out.println(set.dumpIntervals());

        testAdd(set, 11);
        testAdd(set, 12);
        testAdd(set, 13);
        testAdd(set, 14);
        System.out.println(set.dumpIntervals());

        testAdd(set, 9);
        System.out.println(set.dumpIntervals());

        testAdd(set, 10);
        System.out.println(set.dumpIntervals());

        testAdd(set, 18);
        testAdd(set, 19);
        testAdd(set, 20);
        testAdd(set, 21);
        System.out.println(set.dumpIntervals());

        testAdd(set, 17);
        System.out.println(set.dumpIntervals());

        testAdd(set, 15);
        System.out.println(set.dumpIntervals());

        testAdd(set, 16);
        System.out.println(set.dumpIntervals());

        testAdd(set, -10);
        System.out.println(set.dumpIntervals());

        testAdd(set, 100);
        System.out.println(set.dumpIntervals());

        testAdd(set, 100);
        System.out.println(set.dumpIntervals());

        testAdd(set, 0);
        System.out.println(set.dumpIntervals());

        testAdd(set, 18);
        System.out.println(set.dumpIntervals());

        //iterator test
        System.out.println("Iterator");
        for (String s : set) {
            System.out.println(s);
        }
        System.out.println("Iterator.done");
        System.out.println("toArray");
        for (String s : set.toArray()) {
            System.out.println(s);
        }
        System.out.println("toArray.done");


        testRemove(set, 5);
        System.out.println(set.dumpIntervals());
        testRemove(set, 4);
        System.out.println(set.dumpIntervals());
        testRemove(set, 6);
        System.out.println(set.dumpIntervals());
        testRemove(set, 0);
        System.out.println(set.dumpIntervals());
        testRemove(set, -10);
        System.out.println(set.dumpIntervals());
        testRemove(set, 18);
        System.out.println(set.dumpIntervals());
        testRemove(set, 19);
        System.out.println(set.dumpIntervals());

        testRemove(set, 1);
        testRemove(set, 2);
        testRemove(set, 3);
        System.out.println(set.dumpIntervals());

        testRemove(set, 20);
        System.out.println(set.dumpIntervals());
        testRemove(set, 21);
        System.out.println(set.dumpIntervals());

        testRemove(set, 100);
        System.out.println(set.dumpIntervals());

        testRemove(set, 100);
        System.out.println(set.dumpIntervals());
    }
}
