package org.apache.zookeeper.server;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.MethodNotSupportedException;
import org.apache.http.NameValuePair;
import org.apache.http.RequestLine;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentProducer;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.EntityTemplate;
import org.apache.http.entity.StringEntity;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.http.util.EntityUtils;
import org.apache.jute.BinaryOutputArchive;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.KeeperException.NodeExistsException;
import org.apache.zookeeper.MultiResponse;
import org.apache.zookeeper.OpResult;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.ZookeeperServerSupplier;
import org.apache.zookeeper.data.Stat;
import org.apache.zookeeper.proto.ErrorResponse;
import org.apache.zookeeper.proto.GetDataResponse;
import org.apache.zookeeper.proto.MultiHeader;
import org.apache.zookeeper.server.quorum.LeaderZooKeeperServer;
import org.apache.zookeeper.server.quorum.QuorumPeer;

import ru.yandex.charset.Encoder;
import ru.yandex.collection.Pattern;
import ru.yandex.function.OutputStreamProcessorAdapter;
import ru.yandex.function.StringBuilderVoidProcessor;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ByteArrayEntityFactory;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.server.BaseServerConfigBuilder;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.server.sync.HttpEntityEnclosingRequestHandler;
import ru.yandex.http.server.sync.HttpEntityEnclosingRequestHandlerAdapter;
import ru.yandex.stater.AlertThresholds;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.GolovanSignal;
import ru.yandex.stater.ImmutableGolovanAlertsConfig;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.timesource.TimeSource;

public class ZooHttpServer extends BaseHttpServer<ImmutableBaseServerConfig> {
    private static final int STREAM_PING_INTERVAL = 40;
    private static final int STATUS_SLEEP_DELAY = 40;
    private static final int MAGIC_SHARDS = 65534;
    private static final char[] ZEROS = {
        '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '0',
        '0', '0', '0', '0', '0'};
    private final ZKDatabase zkDb;
    private final ZookeeperServerSupplier quorumPeer;
    private final ThreadLocal<DecodableByteArrayOutputStream> streamsLocal =
        new ThreadLocal<>();
    private final ThreadLocal<StringBuilderWriter> sbWriterLocal =
        new ThreadLocal<StringBuilderWriter>() {
            @Override
            public StringBuilderWriter initialValue() {
                return new StringBuilderWriter();
            }};

    public ZooHttpServer(
        final ImmutableBaseServerConfig config,
        final ZKDatabase zkDb,
        final ZookeeperServerSupplier quorumPeer)
        throws IOException, InterruptedException,
        ParseException, ConfigException
    {
        super(config);

        this.zkDb = zkDb;
        this.quorumPeer = quorumPeer;

        register(
            new Pattern<>("/getData", false),
            new GetDataHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/getData2", false),
            new GetData2Handler(false),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/getData3", false),
            new GetData2Handler(true),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/queuelen", false),
            new QueueLenHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/getNextQueueChild", false),
            new GetNextQueueChildHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/getPathByHash", false),
            new GetPathByHashHandler(),
            RequestHandlerMapper.GET);
        HttpRequestHandler unistatHandler = register(
            new Pattern<>("/stat", false),
            new StatHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/unistat", false),
            unistatHandler,
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/cachestats", false),
            new CacheStatsHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/pushPending", false),
            new PushPendingHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_status", false),
            new QueueStatusHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_watch", false),
            new DataWatchHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/_statusStream", false),
            new StatusStreamHandler(),
            RequestHandlerMapper.GET);
        registerStater(new QuorumPeerStater());
    }

    private static class Status implements Comparable<Status> {
        private final String host;
        private final long pos;

        public Status(final String host, final long pos) {
            this.host = host;
            this.pos = pos;
        }

        public String host() {
            return host;
        }

        public long pos() {
            return pos;
        }

        @Override
        public int compareTo(final Status other) {
            return Long.compare(other.pos, pos);
        }
    }

    private static class ServerContext {
        private final CgiParams cgiParams;
        private final Logger logger;
        private final HttpRequest request;
        private final HttpResponse response;
        private final ZooKeeperServer zkServer;
        private final ZKDatabase zkDb;

        ServerContext(
            final CgiParams cgiParams,
            final Logger logger,
            final HttpRequest request,
            final HttpResponse response,
            final ZooKeeperServer zkServer,
            final ZKDatabase zkDb)
        {
            this.cgiParams = cgiParams;
            this.logger = logger;
            this.request = request;
            this.response = response;
            this.zkServer = zkServer;
            this.zkDb = zkDb;
        }

        public CgiParams cgiParams() {
            return cgiParams;
        }

        public Logger logger() {
            return logger;
        }

        public HttpRequest request() {
            return request;
        }

        public HttpResponse response() {
            return response;
        }

        public ZooKeeperServer zkServer() {
            return zkServer;
        }

        public ZKDatabase zkDb() {
            return zkDb;
        }
    }

    private abstract class ZKDbAccessingRequestHandlerBase
        implements HttpRequestHandler
    {
        private StringEntity describeException(final Exception e)
            throws UnsupportedEncodingException
        {
            StringWriter sw = new StringWriter();
            e.printStackTrace(new PrintWriter(sw));
            return new StringEntity(sw.toString());
        }

        @Override
        public void handle(
            final HttpRequest request,
            final HttpResponse response,
            final HttpContext context)
            throws HttpException, IOException
        {
            final Logger logger =
                (Logger)context.getAttribute(BaseHttpServer.LOGGER);
            try {
                final ZooKeeperServer zkServer = quorumPeer.getActiveServer();
                if (zkServer == null) {
                    throw new ServiceUnavailableException(
                        "Quorum is not running");
                }
                if (zkServer.getServerCnxnFactory() == null
                    || zkServer.getServerCnxnFactory().getZooKeeperServer() == null)
                {
                    throw new ServiceUnavailableException(
                        "Quorum is not running");
                }
                final ZKDatabase zkDb = zkServer.getZKDatabase();
                if (!zkDb.isInitialized()) {
                    throw new ServiceUnavailableException(
                        "Quorum is not running");
                }

                handle(
                    new ServerContext(
                        new CgiParams(request),
                        logger,
                        request,
                        response,
                        zkServer,
                        zkDb));
            } catch (KeeperException e) {
                response.setStatusCode(HttpStatus.SC_SERVICE_UNAVAILABLE);
                response.setEntity(describeException(e));
            }
        }

        public abstract void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException;
    }

    private class QueueStatusHandler extends ZKDbAccessingRequestHandlerBase {

        @Override
        public void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException
        {
            final ZKDatabase zkDb = context.zkDb();
            final CgiParams cgiParams = context.cgiParams();

            final String service = cgiParams.getString("service");
            final int shardId =
                    (int) Math.abs(cgiParams.getLong("prefix") % MAGIC_SHARDS);
            final boolean strict = cgiParams.getBoolean("strict", false);
            final boolean all = cgiParams.getBoolean("all", false);
            final JsonType jsonType = JsonTypeExtractor.NULL.extract(cgiParams);

            final List<Status> statuses = new ArrayList<>(10);
            final String statusParentPath =
                '/' + service + '/' + shardId + "/status";
            final DataNode statusParent = zkDb.getNode(statusParentPath);
            if (statusParent != null) {
                final List<String> children =
                    statusParent.getChildren().getList();
                for (String child : children) {
                    final String childNodePath = statusParentPath + '/' + child;
                    final DataNode childNode = zkDb.getNode(childNodePath);
                    if (childNode == null) {
                        continue;
                    }
                    long pos = -1;
                    try {
                        pos = Long.parseLong(new String(childNode.data));
                    } catch (Exception e) {
                        if (strict) {
                            throw new ServiceUnavailableException(
                                "status node: " + statusParentPath + '/' + child
                                + " has unparseable value: "
                                + (childNode.data == null ? "null" :
                                    new String(childNode.data)));
                        }
                    }
                    statuses.add(new Status(child, pos));
                }
            } else {
                throw new BadRequestException("no such service/shard: "
                     + service + '/' + shardId);
            }
            final StringBuilderWriter sbw = sbWriterLocal.get();
            if (jsonType == null) {
                if (statuses.size() > 0) {
                    Collections.sort(statuses);
                    int size = statuses.size();
                    StringBuilder sb = sbw.sb();
                    sb.setLength(0);
                    if (all) {
                        for (int i = 0; i < size; ++i) {
                            Status status = statuses.get(i);
                            sb.append(status.host());
                            sb.append(' ');
                            sb.append(status.pos());
                            sb.append('\n');
                        }
                    } else {
                        Status top = statuses.get(0);
                        sb.append(top.host());
                        sb.append('\n');
                        for (int i = 1; i < size; ++i) {
                            Status status = statuses.get(i);
                            if (status.compareTo(top) != 0) {
                                break;
                            }
                            sb.append(status.host());
                            sb.append('\n');
                        }
                    }
                }
            } else {
                sbw.clear();
                final JsonWriter jsonWriter = jsonType.create(sbw);
                Collections.sort(statuses);
                int size = statuses.size();
                jsonWriter.startArray();
                for (int i = 0; i < size; ++i) {
                    Status status = statuses.get(i);
                    jsonWriter.startObject();
                    jsonWriter.key(status.host());
                    jsonWriter.value(status.pos());
                    jsonWriter.endObject();
                }
                jsonWriter.endArray();
                jsonWriter.close();
            }
            context.response().setEntity(new StringEntity(sbw.toString()));
        }
    }

    private class GetDataHandler extends ZKDbAccessingRequestHandlerBase {
        @Override
        public void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException
        {
            final ZKDatabase zkDb = context.zkDb();
            final CgiParams cgiParams = context.cgiParams();

            List<String> path = cgiParams.getAll("path");
            if (path.size() == 0) {
                throw new BadRequestException("Missing required parameter <path>");
            }
//            Collections.sort(path);
            MultiResponse mrsp = new MultiResponse();
            for (String p : path) {
                try {
                    Stat stat = new Stat();
                    byte[] data = zkDb.getData(p, stat, null);
                    mrsp.add( new OpResult.GetDataResult(data, stat) );
                } catch (KeeperException.NoNodeException e) {
                    mrsp.add( new OpResult.ErrorResult(KeeperException.Code.NONODE.intValue()) );
                }
            }
            DecodableByteArrayOutputStream streams = streamsLocal.get();
            if (streams == null) {
                streams = new DecodableByteArrayOutputStream();
                streamsLocal.set(streams);
            }
            streams.reset();
            BinaryOutputArchive boa = BinaryOutputArchive.getArchive(streams);
            mrsp.serialize(boa, "mrsp");
            context.response().setEntity(
                streams.processWith(ByteArrayEntityFactory.INSTANCE));
        }

    }

    private class GetData2Handler extends ZKDbAccessingRequestHandlerBase {
        private final boolean trailingNoNode;

        GetData2Handler(final boolean trailingNoNode) {
            this.trailingNoNode = trailingNoNode;
        }

        @Override
        public void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException
        {
            final ZKDatabase zkDb = context.zkDb();
            final CgiParams cgiParams = context.cgiParams();

            String service = cgiParams.getString("service");
            String shard = cgiParams.getString("shard");
            long startId = cgiParams.getLong("id");
            int count = cgiParams.getInt("count", 1);

            final EntityTemplate entity =
                new EntityTemplate(
                    new MultiResponseProducer(
                        context,
                        service,
                        shard,
                        startId,
                        count,
                        trailingNoNode));
            entity.setChunked(true);
            ContentType contentType = ContentType.APPLICATION_OCTET_STREAM;
            entity.setContentType(contentType.toString());
            context.response().setEntity(entity);
        }
    }

    private class GetPathByHashHandler extends ZKDbAccessingRequestHandlerBase {

        GetPathByHashHandler() {
        }

        @Override
        public void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException
        {
            final ZKDatabase zkDb = context.zkDb();
            final CgiParams cgiParams = context.cgiParams();

            String service = cgiParams.getString("service");
            String shard = cgiParams.getString("shard");
            String hash = cgiParams.getString("hash");
            boolean checkOnDisk = cgiParams.getBoolean("check-disk", true);
            boolean no404 = cgiParams.getBoolean("no404", false);

            StringBuilder sb = new StringBuilder(60);
            sb.append('/');
            sb.append(service);
            sb.append('/');
            sb.append(shard);
            sb.append("/forward");
            final String path = new String(sb);
            final String existingPath =
                context.zkDb().getNodePath(
                    path,
                    hash,
                    checkOnDisk);
            if (existingPath != null) {
                context.response().setEntity(
                    new StringEntity(existingPath));
            } else {
                if (no404) {
                    context.response().setStatusCode(
                        HttpStatus.SC_NO_CONTENT);
                } else {
                    context.response().setStatusCode(
                        HttpStatus.SC_NOT_FOUND);
                }
            }
        }
    }


    //Emulates MultiResponse.serialize()
    private static class MultiResponseProducer implements ContentProducer {
        private static final String TAG = "mrsp";

        private final ServerContext context;
        private final String service;
        private final String shard;
        private final long startId;
        private final int count;
        private final boolean trailingNoNode;

        MultiResponseProducer(
            final ServerContext context,
            final String service,
            final String shard,
            final long startId,
            final int count,
            final boolean trailingNoNode)
        {
            this.context = context;
            this.service = service;
            this.shard = shard;
            this.startId = startId;
            this.count = count;
            this.trailingNoNode = trailingNoNode;
        }

        private void produce(
            final BinaryOutputArchive boa,
            final OpResult result)
            throws IOException
        {
            final int err = result.getType() == ZooDefs.OpCode.error ?
                ((OpResult.ErrorResult) result).getErr() : 0;
            new MultiHeader(result.getType(), false, err).serialize(boa, TAG);
            switch (result.getType()) {
                case ZooDefs.OpCode.getData:
                    new GetDataResponse(
                        ((OpResult.GetDataResult) result).getData(),
                        ((OpResult.GetDataResult) result).getStat())
                            .serialize(boa, TAG);
                    break;
                case ZooDefs.OpCode.error:
                    new ErrorResponse(((OpResult.ErrorResult) result)
                        .getErr())
                            .serialize(boa, TAG);
                    break;
                default:
                    throw new IOException(
                        "Invalid type " + result.getType()
                        + " in MultiResponse");
            }
        }

        @Override
        public void writeTo(final OutputStream out) throws IOException {
            StringBuilder sb = new StringBuilder(60);
//            /opqueue/12145/forward/queue00000000000000691039
            sb.append('/');
            sb.append(service);
            sb.append('/');
            sb.append(shard);
            sb.append("/forward/queue");
            int prefixLen = sb.length();

            BinaryOutputArchive boa = BinaryOutputArchive.getArchive(out);
            for (long id = startId; id < startId + count; id++) {
                sb.setLength(prefixLen);
                sb.append(id);
                final int idLength = sb.length() - prefixLen;
                sb.insert(prefixLen, ZEROS, 0, 20 - idLength);
                final String path = new String(sb);
//                logger().info("Path: " + path);
                try {
                    Stat stat = new Stat();
                    byte[] data = context.zkDb().getData(path, stat, null);
                    produce(boa, new OpResult.GetDataResult(data, stat));
                } catch (KeeperException.NoNodeException e) {
                    produce(
                        boa,
                        new OpResult.ErrorResult(
                            KeeperException.Code.NONODE.intValue()));
                    if (trailingNoNode) {
                        //add NoNode only once to save traffic
                        break;
                    }
                } catch (KeeperException e) {
                    context.logger().log(
                        Level.SEVERE,
                        "zkDb.getData error",
                        e);
                    throw new IOException(e);
                }
            }
            new MultiHeader(-1, true, -1).serialize(boa, TAG);
        }

    }

    private class GetNextQueueChildHandler
        extends ZKDbAccessingRequestHandlerBase
    {
        @Override
        public void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException
        {
            final ZKDatabase zkDb = context.zkDb();
            final CgiParams cgiParams = context.cgiParams();

            String path = cgiParams.getString("path");
            if (path == null) {
                throw new BadRequestException("Missing required parameter <path>");
            }

            String parent = path.substring(0, path.lastIndexOf('/'));
            DataNode queue = zkDb.getNode(parent);
            if (!queue.hasQueue) {
                throw new BadRequestException("path " + parent + " is not a queue path");
            }

            String nextNode = zkDb.getNextNode(path);
            if (nextNode != null) {
                context.response().setEntity(new StringEntity(nextNode));
            } else {
                throw new NotFoundException();
            }
        }
    }

    private class StatHandler extends ZKDbAccessingRequestHandlerBase {
        @Override
        public void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException
        {
            final CgiParams cgiParams = context.cgiParams();

            if (!(context.zkServer() instanceof LeaderZooKeeperServer)) {
                context.response().setEntity(new StringEntity("Not a leader"));
                return;
            }
            LeaderZooKeeperServer leader =
                (LeaderZooKeeperServer) context.zkServer();
            context.response().setEntity(
                new StringEntity(
                    leader.commitProcessor.toString()));
        }

    }

    private class CacheStatsHandler extends ZKDbAccessingRequestHandlerBase {
        @Override
        public void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException
        {
            final ZKDatabase zkDb = context.zkDb();
            context.response().setEntity(new StringEntity(zkDb.cacheStats()));
        }

    }

    private class PushPendingHandler extends ZKDbAccessingRequestHandlerBase {
        @Override
        public void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException
        {
            if (!(context.zkServer() instanceof LeaderZooKeeperServer)) {
                context.response().setEntity(new StringEntity("Not a leader"));
                return;
            }
            LeaderZooKeeperServer leader =
                (LeaderZooKeeperServer) context.zkServer();
            leader.commitProcessor.pushPending();
            context.response().setEntity(new StringEntity("ok"));
        }

    }


    private class QueueLenHandler extends ZKDbAccessingRequestHandlerBase {
        @Override
        public void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException
        {
            final Charset acceptedCharset =
                CharsetUtils.acceptedCharset(
                    context.request());
            final EntityTemplate entity =
                new EntityTemplate(
                    new QueueLenProducer(context, acceptedCharset));
            entity.setChunked(true);
            final ContentType contentType = ContentType.TEXT_PLAIN;
            entity.setContentType(
                contentType.withCharset(acceptedCharset).toString());
            context.response().setEntity(entity);
        }
    }

    private class QueueLenProducer implements ContentProducer {
        private final ServerContext context;
        private final Charset acceptedCharset;

        QueueLenProducer(
            final ServerContext context,
            final Charset acceptedCharset)
        {
            this.context = context;
            this.acceptedCharset = acceptedCharset;
        }

        @Override
        public void writeTo(final OutputStream out) throws IOException {
            final Map<String, DataTree.Queue> queues = zkDb.getQueues();
            StringBuilder sb = new StringBuilder();
            StringBuilderVoidProcessor<byte[], IOException> encoder =
                new StringBuilderVoidProcessor<>(new Encoder(acceptedCharset));
            OutputStreamProcessorAdapter adapter =
                new OutputStreamProcessorAdapter(out);

            Stat stat = new Stat();
            long currentTime = TimeSource.INSTANCE.currentTimeMillis();
            for (Map.Entry<String, DataTree.Queue> entry : queues.entrySet()) {
                if (entry.getValue().queueNode.getChildren() == null) {
                    continue;
                }
                String lastQueueNode =
                    entry.getValue().queueNode.getNewestChild();
                long num =
                    Long.parseLong(
                        lastQueueNode.substring(lastQueueNode.length() - 19));
                long lastQueueNodeTime = -1;//nap.node.getMtime();
                DataNode lastAddedNode =
                    zkDb.getNode(entry.getKey() + '/' + lastQueueNode);
                if (lastAddedNode != null) {
                    lastQueueNodeTime = lastAddedNode.getCtime();
                }
                String[] cols = entry.getKey().split("/");
                String service = cols[1];
                String shard = cols[2];
                String statusPath = "/" + service + "/" + shard + "/status";
                long maxApplied = 0;
                long maxMTime = 0;
                long leastLiveness = 0;
                try
                {
                    DataNode status = zkDb.getNode(statusPath);
                    if (status != null) {
                        List<String> childs = null;
                        synchronized (status) {
                            ChildSet c = status.getChildren();
                            if( c != null )
                            {
                                childs = c.getList();
                            }
                        }
                        if (childs != null) {
                            for (String child : childs) {
                                byte[] data =
                                    zkDb.getData(
                                        statusPath + "/" + child,
                                        stat,
                                        null);
                                long applied = 0;
                                if (data != null) {
                                    String numStr = new String(data, "UTF8");
                                    applied = Long.parseLong(numStr);
                                }
                                long mTime = stat.getMtime();
                                sb.append(service);
                                sb.append('\t');
                                sb.append(shard);
                                sb.append('\t');
                                sb.append(child);
                                sb.append('\t');
                                sb.append(num - applied);
                                sb.append('\t');
                                sb.append(num);
                                sb.append('\t');
                                sb.append(applied);
                                sb.append('\t');
                                sb.append(mTime);
                                sb.append('\t');
                                sb.append(currentTime - mTime);
                                sb.append('\t');
                                long liveness =
                                    Math.max(0, lastQueueNodeTime - mTime);
                                sb.append(liveness);
                                sb.append('\n');
                                if (applied > maxApplied) {
                                    maxApplied = applied;
                                }
                                if (maxMTime < mTime) {
                                    maxMTime = mTime;
                                }
                                if (leastLiveness < liveness) {
                                    leastLiveness = liveness;
                                }
                            }
                        }
                    }
//                    System.err.println( statusPath );
                    sb.append(service);
                    sb.append('\t');
                    sb.append(shard);
                    sb.append('\t');
                    sb.append("MIN");
                    sb.append('\t');
                    sb.append(num - maxApplied);
                    sb.append('\t');
                    sb.append(num);
                    sb.append('\t');
                    sb.append(maxApplied);
                    sb.append('\t');
                    sb.append(maxMTime);
                    sb.append('\t');
                    sb.append(currentTime - maxMTime);
                    sb.append('\t');
                    sb.append(leastLiveness);
                    sb.append('\n');

                    encoder.process(sb);
                    encoder.processWith(adapter);

                    sb.setLength(0);
                } catch (Exception e) {
                    e.printStackTrace();
                    throw new IOException(e);
                }
            }
        }

    }

    private class DataWatchHandler extends ZKDbAccessingRequestHandlerBase {

        DataWatchHandler() {
        }

        @Override
        public void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException
        {
            final ZKDatabase zkDb = context.zkDb();
            final CgiParams cgiParams = context.cgiParams();

            String service = cgiParams.getString("service");
            final JsonType jsonType = JsonTypeExtractor.NORMAL.extract(cgiParams);
            String watchPath = '/' + service;
            DataWatchProducer watcher =
                new DataWatchProducer(watchPath, zkDb, jsonType);
            zkDb.registerWatcher(watchPath, watcher);
            final EntityTemplate entity = new EntityTemplate(watcher);
            entity.setChunked(true);
            ContentType contentType = ContentType.APPLICATION_OCTET_STREAM;
            entity.setContentType(contentType.toString());
            context.response().setEntity(entity);
        }
    }

    private static final DataWatchManager.Event CLOSE =
            new DataWatchManager.Event("", null, 0);
    private class DataWatchProducer
        implements DataWatchManager.Watcher, ContentProducer
    {
        private final LinkedBlockingQueue<DataWatchManager.Event> events =
            new LinkedBlockingQueue<>();
        private final String watchPath;
        private final ZKDatabase zkDb;
        private final JsonType jsonType;
        private volatile Utf8JsonWriter writer = null;
        private volatile boolean closed = false;

        DataWatchProducer(
            final String watchPath,
            final ZKDatabase zkDb,
            final JsonType jsonType)
        {
            this.watchPath = watchPath;
            this.zkDb = zkDb;
            this.jsonType = jsonType;
        }

        @Override
        public void notifyDataChange(final DataWatchManager.Event event) {
            if (event.path().contains("/status/")) {
                events.offer(event);
            }
        }

        @Override
        public void close() {
            events.offer(CLOSE);
        }

        @Override
        public boolean closed() {
            return closed;
        }

        private boolean dispatch(final DataWatchManager.Event event)
            throws IOException
        {
            if (event == null) {
                writer.startObject();
                writer.key("cmd");
                writer.value("ping");
                writer.endObject();
            } else if (event == CLOSE) {
                writer.startObject();
                writer.key("cmd");
                writer.value("close");
                writer.endObject();
                closed = true;
                return false;
            } else {
                writer.startObject();
                writer.key("cmd");
                writer.value("data");
                writer.key("path");
                writer.value(event.path());
                writer.key("data");
                writer.value(event.data());
                writer.endObject();
            }
            return true;
        }

        @Override
        public void writeTo(final OutputStream out) throws IOException {
            try {
                Utf8JsonWriter writer = jsonType.create(out);
                this.writer = writer;
                writer.startArray();
                while (true) {
                    DataWatchManager.Event event = null;
                    try {
                        int dispatched = 0;
                        for (
                            event = events.poll(1, TimeUnit.SECONDS);
                            event != null;
                            event = events.poll())
                        {
                            if (!dispatch(event)) {
                                writer.endArray();
                                writer.close();
                                return;
                            }
                            dispatched++;
                        }
                        if (dispatched == 0) {
                            dispatch(null);
                        }
                        writer.flush();
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
            } finally {
                zkDb.unregisterWatcher(watchPath, this);
            }
        }
    }

    private class StatusStreamHandler extends ZKDbAccessingRequestHandlerBase {

        StatusStreamHandler() {
        }

        @Override
        public void handle(final ServerContext context)
            throws HttpException, IOException, KeeperException
        {
            final ZKDatabase zkDb = context.zkDb();
            final CgiParams cgiParams = context.cgiParams();

            String service = cgiParams.getString("service");
            final JsonType jsonType = JsonTypeExtractor.NORMAL.extract(cgiParams);
            String watchPath = '/' + service;
            StatusStreamProducer watcher =
                new StatusStreamProducer(watchPath, zkDb, jsonType);
            zkDb.registerWatcher(watchPath, watcher);
            final EntityTemplate entity = new EntityTemplate(watcher);
            entity.setChunked(true);
            ContentType contentType = ContentType.APPLICATION_OCTET_STREAM;
            entity.setContentType(contentType.toString());
            context.response().setEntity(entity);
        }
    }

    private class StatusStreamProducer
        implements DataWatchManager.Watcher, ContentProducer
    {
        private static final String CLOSE = "close";
        private final ConcurrentLinkedQueue<String> events =
            new ConcurrentLinkedQueue<>();
        private final ConcurrentHashMap<
            String, DataWatchManager.Event> dirtyData =
                new ConcurrentHashMap<>();
        private final String watchPath;
        private final ZKDatabase zkDb;
        private final JsonType jsonType;
        private volatile Utf8JsonWriter writer = null;
        private volatile boolean closed = false;

        StatusStreamProducer(
            final String watchPath,
            final ZKDatabase zkDb,
            final JsonType jsonType)
        {
            this.watchPath = watchPath;
            this.zkDb = zkDb;
            this.jsonType = jsonType;
        }

        @Override
        public void notifyDataChange(final DataWatchManager.Event event) {
            if (writer == null) {
                return;
            }
            if (event.path().contains("/status/")) {
                while (true) {
                    DataWatchManager.Event old =
                        dirtyData.putIfAbsent(event.path(), event);
                    if (old == null) {
                        events.offer(event.path());
                        break;
                    } else {
                        if (old.zxid() < event.zxid()) {
                            if (dirtyData.replace(event.path(), old, event)) {
                                break;
                            }
                        } else {
                            break;
                        }
                    }
                }
            }
        }

        @Override
        public void close() {
            events.offer(CLOSE);
        }

        @Override
        public boolean closed() {
            return closed;
        }

        private void sendStatus(final String path, final byte[] data)
            throws IOException
        {
            int slash = path.lastIndexOf('/');
            final String host = path.substring(slash + 1);
            slash = path.indexOf('/', watchPath.length() + 1);
            final String shard =
                path.substring(watchPath.length() + 1, slash);
            writer.startObject();
            writer.key("cmd");
            writer.value("data");
            writer.key("host");
            writer.value(host);
            writer.key("shard");
            writer.value(shard);
            writer.key("data");
            if (data == null) {
                writer.value("-1");
            } else {
                writer.value(data);
            }
            writer.endObject();
        }

        private boolean dispatch(final String path)
            throws IOException
        {
            if (path == null) {
                writer.startObject();
                writer.key("cmd");
                writer.value("ping");
                writer.endObject();
            } else if (path == CLOSE) {
                writer.startObject();
                writer.key("cmd");
                writer.value("close");
                writer.endObject();
                closed = true;
                return false;
            } else {
                DataWatchManager.Event event = dirtyData.remove(path);
                if (event != null) {
                    sendStatus(path, event.data());
                }
            }
            return true;
        }

        private void dumpStatuses() throws IOException {
            final Map<String, DataTree.Queue> queues = zkDb.getQueues();

            Stat stat = new Stat();
            StringBuilder sb = new StringBuilder(watchPath);
            for (Map.Entry<String, DataTree.Queue> entry : queues.entrySet()) {
                final String path = entry.getKey();
                sb.setLength(watchPath.length());
                int slash = path.indexOf('/', watchPath.length() + 1);
                if (slash == -1) {
                    continue;
                }
                sb.append('/');
                sb.append(path.substring(watchPath.length() + 1, slash));
                sb.append("/status");
                String statusPath = new String(sb);
                DataNode status = zkDb.getNode(statusPath);
                if (status != null) {
                    List<String> childs = null;
                    synchronized (status) {
                        ChildSet c = status.getChildren();
                        if (c != null) {
                            childs = c.getList();
                        }
                    }
                    if (childs != null) {
                        for (String child : childs) {
                            final String nodePath = statusPath + '/' + child;
                            try {
                                byte[] data =
                                    zkDb.getData(
                                        nodePath,
                                        stat,
                                        null);
                                sendStatus(nodePath, data);
                            } catch (KeeperException e) {
                            }
                        }
                    }
                }
            }
        }

        @Override
        public void writeTo(final OutputStream out) throws IOException {
            try {
                Utf8JsonWriter writer = jsonType.create(out);
                this.writer = writer;
                writer.startArray();
                dumpStatuses();
                long prevPing = TimeSource.INSTANCE.currentTimeMillis();
                long prevFlush = prevPing;
                int dispatched = 0;
                while (true) {
                    String path = null;
                    try {
                        for (
                            path = events.poll();
                            path != null;
                            path = events.poll())
                        {
                            if (!dispatch(path)) {
                                writer.endArray();
                                writer.close();
                                return;
                            }
                            dispatched++;
                        }
                        long time = TimeSource.INSTANCE.currentTimeMillis();
                        boolean flush = false;
                        long timeFromFlush = time - prevFlush;
                        if (dispatched == 0) {
                            if (time - prevPing > STREAM_PING_INTERVAL) {
                                dispatch(null);
                                prevPing =
                                    TimeSource.INSTANCE.currentTimeMillis();
                                flush = true;
                            }
                        } else {
                            if (timeFromFlush > STATUS_SLEEP_DELAY) {
                                flush = true;
                            }
                        }
                        if (flush) {
                            writer.flush();
                            dispatched = 0;
                            prevFlush = TimeSource.INSTANCE.currentTimeMillis();
                        } else {
                            Thread.sleep(STATUS_SLEEP_DELAY - timeFromFlush);
                        }
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
            } finally {
                zkDb.unregisterWatcher(watchPath, this);
            }
        }
    }

    private class QuorumPeerStater implements Stater {
        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            final ZooKeeperServer zkServer = quorumPeer.getActiveServer();
            boolean inQuorum = true;
            long myId = -1;
            if (zkServer == null) {
                inQuorum = false;
            } else {
                myId = zkServer.getServerId();
                if (zkServer.getServerCnxnFactory() == null
                    || zkServer.getServerCnxnFactory().getZooKeeperServer() == null)
                {
                    inQuorum = false;
                }
                final ZKDatabase zkDb = zkServer.getZKDatabase();
                if (!zkDb.isInitialized()) {
                    inQuorum = false;
                }
            }
            boolean leader = inQuorum;
            long leaderId = 0;
            if (inQuorum) {
                if (zkServer instanceof LeaderZooKeeperServer) {
                    leader = true;
                    leaderId = myId;
                }
            }

            statsConsumer.stat("quorum-peer-in-quorum_ammt", inQuorum);
            statsConsumer.stat("quorum-peer-is-leader_ammt", leader);
            statsConsumer.stat("quorum-peer-leader-id_ammt", leaderId);
            statsConsumer.stat("quorum-peer-id_ammt", myId);
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);
            GolovanChart chart = new GolovanChart(
                "quorum-info",
                " quorum info",
                false,
                false,
                0d);
            chart.addSignal(
                new GolovanSignal(
                    statsPrefix + "quorum-peer-in-quorum_ammt",
                    config.tag(),
                    "in quorum",
                    null,
                    0,
                    false));
            chart.addSignal(
                new GolovanSignal(
                    statsPrefix + "quorum-peer-leader-id_ammt",
                    config.tag(),
                    "leader id",
                    null,
                    0,
                    false));
            group.addChart(chart);
            panel.addCharts(
                "quorum",
                null,
                group);
        }

        @Override
        public void addToAlertsConfig(
            final IniConfig alertsConfig,
            final ImmutableGolovanPanelConfig panelConfig,
            final String statsPrefix)
            throws BadRequestException
        {
            ImmutableGolovanAlertsConfig alerts = panelConfig.alerts();
            alerts.createAlert(
                alertsConfig,
                alerts.module() + "-has-quorum",
                statsPrefix + "quorum-peer-leader-id_ammt",
                new AlertThresholds(null, 0.5d, null));
        }
    }
}
