package org.apache.zookeeper.server.persistence;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.Iterator;
import java.util.Map;

import org.jctools.maps.NonBlockingHashMap;

final public class DoubleBarrelLRUCache<K, V> {
    private final Map<K,V> cache1;
    private final Map<K,V> cache2;
    private final AtomicInteger countdown;
    private volatile boolean swapped;
    private final int maxSize;

    public DoubleBarrelLRUCache(final int maxSize) {
        this.maxSize = maxSize;
        countdown = new AtomicInteger(maxSize);
        cache1 = new NonBlockingHashMap<K,V>();
        cache2 = new NonBlockingHashMap<K,V>();
    }

    public V get(final K key) {
        final Map<K,V> primary;
        final Map<K,V> secondary;
        if (swapped) {
            primary = cache2;
            secondary = cache1;
        } else {
            primary = cache1;
            secondary = cache2;
        }

        // Try primary first
        V result = primary.get(key);
        if (result == null) {
            // Not found -- try secondary
            result = secondary.get(key);
            if (result != null) {
                // Promote to primary
                put(key, result);
            }
        }
        return result;
    }

    public void put(final K key, final V value) {
        final Map<K,V> primary;
        final Map<K,V> secondary;
        if (swapped) {
            primary = cache2;
            secondary = cache1;
        } else {
            primary = cache1;
            secondary = cache2;
        }
        primary.put(key, value);

        if (countdown.decrementAndGet() == 0) {
            secondary.clear();
            swapped = !swapped;
            countdown.set(maxSize);
        }
    }

    public void remove(final K key) {
        cache1.remove(key);
        cache2.remove(key);
        countdown.incrementAndGet();
    }

    public void clear() {
        cache1.clear();
        cache2.clear();
        countdown.set(maxSize);
    }
}
